package org.nkjmlab.sorm4j.internal.context.common;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.nkjmlab.sorm4j.context.SormContext;
import org.nkjmlab.sorm4j.internal.sql.metadata.ColumnMetaData;
import org.nkjmlab.sorm4j.internal.sql.metadata.TableMetaData;

public final class TableMetaDataImpl implements TableMetaData {

  private final String tableName;
  private final List<String> columns;
  private final List<String> primaryKeys;
  private final List<String> autoGeneratedColumns;
  private final boolean hasPrimaryKey;
  private final boolean hasAutoGeneratedColumns;
  private final List<String> notAutoGeneratedColumns;

  private final List<String> columnsForUpdate;
  private final String[] autoGeneratedColumnsArray;
  private final List<String> notPrimaryKeys;
  private final List<String> columnsWithAliases;
  private final List<ColumnMetaData> columnsWithMetaData;

  public TableMetaDataImpl(
      String tableName,
      String columnAliasPrefix,
      List<ColumnMetaData> columnsWithMetaData,
      List<String> primaryKeys,
      List<String> autoGeneratedColumns) {
    this.tableName = tableName;

    this.columnsWithMetaData = List.copyOf(columnsWithMetaData);
    this.columns =
        Collections.unmodifiableList(
            columnsWithMetaData.stream().map(c -> c.getColumnName()).collect(Collectors.toList()));
    this.primaryKeys = Collections.unmodifiableList(primaryKeys);
    this.autoGeneratedColumns = Collections.unmodifiableList(autoGeneratedColumns);

    this.notAutoGeneratedColumns =
        Collections.unmodifiableList(
            columns.stream()
                .filter(col -> !autoGeneratedColumns.contains(col))
                .collect(Collectors.toList()));
    this.hasPrimaryKey = !primaryKeys.isEmpty();
    this.hasAutoGeneratedColumns = !autoGeneratedColumns.isEmpty();

    this.autoGeneratedColumnsArray = autoGeneratedColumns.toArray(String[]::new);

    this.notPrimaryKeys =
        Collections.unmodifiableList(
            columns.stream()
                .filter(col -> !primaryKeys.contains(col))
                .collect(Collectors.toList()));

    this.columnsForUpdate =
        Collections.unmodifiableList(
            Stream.concat(notPrimaryKeys.stream(), primaryKeys.stream())
                .collect(Collectors.toList()));

    this.columnsWithAliases =
        columns.stream()
            .map(
                col ->
                    tableName
                        + "."
                        + col
                        + " as "
                        + SormContext.getDefaultCanonicalStringCache()
                            .toCanonicalNameWithTableName(columnAliasPrefix, col))
            .collect(Collectors.toList());
  }

  @Override
  public List<String> getAutoGeneratedColumns() {
    return autoGeneratedColumns;
  }

  @Override
  public String[] getAutoGeneratedColumnsArray() {
    return autoGeneratedColumnsArray;
  }

  @Override
  public List<String> getColumnAliases() {
    return columnsWithAliases;
  }

  @Override
  public List<String> getColumns() {
    return columns;
  }

  @Override
  public List<String> getColumnsForUpdate() {
    return columnsForUpdate;
  }

  @Override
  public List<ColumnMetaData> getColumnsMetaData() {
    return columnsWithMetaData;
  }

  @Override
  public List<String> getNotAutoGeneratedColumns() {
    return notAutoGeneratedColumns;
  }

  @Override
  public List<String> getNotPrimaryKeys() {
    return notPrimaryKeys;
  }

  @Override
  public List<String> getPrimaryKeys() {
    return primaryKeys;
  }

  @Override
  public String getTableName() {
    return tableName;
  }

  @Override
  public boolean hasAutoGeneratedColumns() {
    return hasAutoGeneratedColumns;
  }

  @Override
  public boolean hasPrimaryKey() {
    return hasPrimaryKey;
  }

  @Override
  public String toString() {
    return "TableMetaDataImpl [tableName="
        + tableName
        + ", columns="
        + columns
        + ", primaryKeys="
        + primaryKeys
        + ", autoGeneratedColumns="
        + autoGeneratedColumns
        + ", hasPrimaryKey="
        + hasPrimaryKey
        + ", hasAutoGeneratedColumns="
        + hasAutoGeneratedColumns
        + ", notAutoGeneratedColumns="
        + notAutoGeneratedColumns
        + ", columnsForUpdate="
        + columnsForUpdate
        + ", autoGeneratedColumnsArray="
        + Arrays.toString(autoGeneratedColumnsArray)
        + ", notPrimaryKeys="
        + notPrimaryKeys
        + ", columnsWithAliases="
        + columnsWithAliases
        + ", columnsWithMetaData="
        + columnsWithMetaData
        + "]";
  }
}
