/*
 * Copyright 2017-2025 noear.org and authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.noear.solon.expression.snel;

import org.noear.solon.expression.Parser;
import org.noear.solon.expression.Expression;
import org.noear.solon.expression.util.LRUCache;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Solon 表达式语言模板解析器
 *
 * <p>
 * 支持以下示例：
 * 1."name is #{user.name}, and key is ${key}"
 * 2.支持嵌套表达式，如 "Hello, #{${user.name?:Tom}}!"
 * </p>
 * @author noear
 * @since 3.1
 */
public class SnelTemplateParser implements Parser<String> {
    private static final SnelTemplateParser INSTANCE = new SnelTemplateParser(2000);
    private final Map<String, Expression<String>> exprCached;

    public SnelTemplateParser(int cahceCapacity) {
        exprCached = Collections.synchronizedMap(new LRUCache<>(cahceCapacity));
    }

    public static final char MARK_START_EXPRESSION = '#';
    public static final char MARK_START_PROPERTIES = '$';
    private static final char MARK_BRACE_OPEN = '{';
    private static final char MARK_BRACE_CLOSE = '}';

    public static SnelTemplateParser getInstance() {
        return INSTANCE;
    }

    @Override
    public Expression<String> parse(String expr, boolean cached) {
        return cached ? exprCached.computeIfAbsent(expr, this::parseDo) : parseDo(expr);
    }

    private Expression<String> parseDo(String expr) {
        List<TemplateFragment> fragments = new ArrayList<>(8); // 基于经验值的容量初始化
        boolean inExpression = false;
        char marker = 0;
        int textStart = 0;
        int scanPosition = 0;
        final int length = expr.length();

        while (scanPosition < length) {
            if (!inExpression) {
                // 文本模式：批量扫描直到发现表达式起始标记
                int exprStart = findExpressionStart(expr, scanPosition);
                if (exprStart != -1) {
                    // 添加前置文本
                    if (textStart < exprStart) {
                        fragments.add(new TemplateFragment(TemplateMarker.TEXT, expr.substring(textStart, exprStart)));
                    }
                    marker = expr.charAt(exprStart);
                    inExpression = true;
                    textStart = scanPosition = exprStart + 2; // 跳过标记符（如 "#{"）
                } else {
                    // 剩余部分全部作为文本
                    fragments.add(new TemplateFragment(TemplateMarker.TEXT, expr.substring(textStart)));
                    break;
                }
            } else {
                // 表达式模式：需要平衡大括号以处理嵌套表达式
                int closePos = findExpressionEnd(expr, scanPosition);
                if (closePos != -1) {
                    String content = expr.substring(textStart, closePos);
                    if (MARK_START_PROPERTIES == marker) {
                        fragments.add(new TemplateFragment(TemplateMarker.PROPERTIES, content));
                    } else {
                        fragments.add(new TemplateFragment(TemplateMarker.EXPRESSION, content));
                    }

                    inExpression = false;
                    textStart = scanPosition = closePos + 1; // 跳过闭合标记 "}"
                } else {
                    // 未闭合表达式作为文本回退
                    fragments.add(new TemplateFragment(TemplateMarker.TEXT, expr.substring(textStart - 2, textStart) + expr.substring(textStart)));
                    break;
                }
            }
        }

        return new TemplateNode(fragments);
    }

    // 快速定位表达式起始标记（"#{" 或 "${"）
    private int findExpressionStart(String s, int start) {
        for (int i = start; i < s.length() - 1; i++) {
            char c = s.charAt(i);
            if ((c == MARK_START_EXPRESSION || c == MARK_START_PROPERTIES) && s.charAt(i + 1) == MARK_BRACE_OPEN) {
                return i;
            }
        }
        return -1;
    }

    // 查找表达式结束位置，平衡大括号以支持嵌套
    private int findExpressionEnd(String s, int start) {
        int braceCount = 1; // 已经有一个开括号
        for (int i = start; i < s.length(); i++) {
            char c = s.charAt(i);
            if (c == MARK_BRACE_OPEN) {
                braceCount++;
            } else if (c == MARK_BRACE_CLOSE) {
                braceCount--;
                if (braceCount == 0) {
                    return i;
                }
            }
        }
        return -1; // 未找到匹配的结束括号
    }
}