/*
 * Copyright 2017-2024 noear.org and authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.noear.solon.server.jetty;

import org.eclipse.jetty.ee11.jsp.JettyJspServlet;
import org.eclipse.jetty.ee11.servlet.ServletContextHandler;
import org.eclipse.jetty.ee11.servlet.ServletHolder;
import org.eclipse.jetty.util.resource.PathResourceFactory;
import org.noear.solon.core.util.ResourceUtil;
import org.noear.solon.server.jetty.jsp.JspLifeCycle;
import org.noear.solon.server.jetty.jsp.JspTldLocator;

import jakarta.servlet.ServletContext;
import jakarta.servlet.descriptor.TaglibDescriptor;

import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Map;

public class JettyServerAddJsp extends JettyServer {

    /**
     * 获取Server Handler
     */
    @Override
    protected ServletContextHandler buildHandler() throws IOException {
        ServletContextHandler handler = getServletHandler();

        //jsp 资源的根目录
        PathResourceFactory rf = new PathResourceFactory();
        handler.setBaseResource(rf.newResource(ResourceUtil.getResource("/")));

        addJspSupport(handler);
        addTdlSupport(handler.getServletContext());

        return handler;
    }

    private void addJspSupport(ServletContextHandler handler) throws IOException {

        // Set Classloader of Context to be sane (needed for JSTL)
        // JSP requires a non-System classloader, this simply wraps the
        // embedded System classloader in a way that makes it suitable
        // for JSP to use
        ClassLoader jspClassLoader = new URLClassLoader(new URL[0], this.getClass().getClassLoader());
        handler.setClassLoader(jspClassLoader);

        // Manually call JettyJasperInitializer on context startup
        handler.addBean(new JspLifeCycle(handler));

        // Create / Register JSP Servlet (must be named "jsp" per spec)
        ServletHolder holderJsp = new ServletHolder("jsp", JettyJspServlet.class);
        holderJsp.setInitOrder(0);

        handler.addServlet(holderJsp, "*.jsp");
    }

    private void addTdlSupport(ServletContext servletContext) throws IOException {
        Map<String, TaglibDescriptor> tagLibInfos = JspTldLocator.createTldInfos("WEB-INF", "templates");

        if (tagLibInfos.size() > 0) {
            ServletContextHandler.JspConfig jspConfig = (ServletContextHandler.JspConfig) servletContext.getJspConfigDescriptor();
            if (jspConfig == null) {
                jspConfig = new ServletContextHandler.JspConfig();
                ((ServletContextHandler.ServletContextApi) servletContext).setJspConfigDescriptor(jspConfig);
            }

            for (TaglibDescriptor descriptor : tagLibInfos.values()) {
                jspConfig.addTaglibDescriptor(descriptor);
            }
        }
    }
}