package org.oa4mp.server.admin.oauth2.tools;

import com.nimbusds.jose.util.Base64URL;
import edu.uiuc.ncsa.security.core.exceptions.GeneralException;
import edu.uiuc.ncsa.security.core.util.FileUtil;
import edu.uiuc.ncsa.security.servlet.ServiceClient;
import edu.uiuc.ncsa.security.util.cli.CLIDriver;
import edu.uiuc.ncsa.security.util.cli.CommonCommands2;
import edu.uiuc.ncsa.security.util.cli.InputLine;
import edu.uiuc.ncsa.security.util.crypto.KeyUtil;
import edu.uiuc.ncsa.security.util.jwk.JSONWebKey;
import edu.uiuc.ncsa.security.util.jwk.JSONWebKeys;
import edu.uiuc.ncsa.security.util.jwk.JWKUtil2;
import net.sf.json.JSON;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.codec.binary.Base64;
import org.oa4mp.delegation.server.JWTUtil;
import org.oa4mp.delegation.server.server.claims.OA2Claims;
import org.oa4mp.server.loader.oauth2.OA2SE;
import org.oa4mp.server.loader.qdl.util.SigningCommands;

import java.io.*;
import java.math.BigInteger;
import java.net.URI;
import java.security.KeyPair;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.TreeSet;

import static org.oa4mp.delegation.server.JWTUtil.decat;
import static org.oa4mp.delegation.server.JWTUtil.getJsonWebKeys;
import static org.oa4mp.server.loader.qdl.util.SigningCommands.createRSAJWK;

/**
 * <p>Created by Jeff Gaynor<br>
 * on 5/6/19 at  2:39 PM
 */
public class JWKUtilCommands extends CommonCommands2 {
    public JWKUtil2 getJwkUtil() {
        if (jwkUtil == null) {
            jwkUtil = new JWKUtil2();
        }
        return jwkUtil;
    }

    public void setJwkUtil(JWKUtil2 jwkUtil) {
        this.jwkUtil = jwkUtil;
    }

    JWKUtil2 jwkUtil;

    public JWKUtilCommands(CLIDriver driver) throws Throwable {
        super(driver);
    }

    @Override
    public void about(boolean showBanner, boolean showHeader) {

    }

    @Override
    public void initialize() throws Throwable {

    }

    @Override
    public void load(InputLine inputLine) throws Throwable {

    }

    @Override
    public String getName() {
        return "jwk";
    }

    @Override
    public String getPrompt() {
        return getName() + ">";
    }

    protected void createKeysHelps() {
        say("create_keys [" + CL_INPUT_FILE_FLAG + " set_of_keys " + CL_IS_PUBLIC_FLAG + "] | [" + CL_IS_PRIVATE_FLAG + "] " + FORCE_TO_STD_OUT_FLAG + "] " + CL_OUTPUT_FILE_FLAG + " file");
        sayi(CL_INPUT_FILE_FLAG + " - and input file of keys. Implies " + CL_IS_PUBLIC_FLAG + " and will extract the public keys.");
        sayi(CL_IS_PUBLIC_FLAG + " - extract public keys from the file specified by " + CL_INPUT_FILE_FLAG);
        sayi(CL_IS_PRIVATE_FLAG + " - (implied) generate full set of keys. Cannot be specified with  " + CL_IS_PUBLIC_FLAG);
        sayi(FORCE_TO_STD_OUT_FLAG + " - write the keys to standard out.");
        sayi(SET_DEFAULT_ID + " - specify the id of the default key (only at creation of a full set).");
        sayi(CL_OUTPUT_FILE_FLAG + " - write the keys to the given file.");
        sayi(CREATE_SINGLE_KEY_FLAG + " - create only a single RSA 256 key.");
        sayi(RSA_KEY_SIZE_FLAG + " - specify the key size for the RSA key. Default is 2048 and the value must be multiple of 256");
        sayi(ELLIPTIC_CURVE_FLAG + " - specify the specific elliptic curve to use. Invoke help with ec to see a list");
        sayi(ALGORITHM_KEY_FLAG + " - For both RSA and ellitpic functions. Specify the key generation algorithm. Invoke help");
        sayi("       with ec or rsa to see a list.");
        sayi(CREATE_ELLIPTIC_KEY_FLAG + " - if present will create an keys with the default elliptic function");
        sayi("Create a set of RSA JSON Web keys and store them in the given file");
        sayi("There are several modes of operation. If you do not specify an output file, then the keys are written ");
        sayi("to the command line.");
        sayi("   E.g.");
        sayi("   create_keys " + CL_OUTPUT_FILE_FLAG + " keys.jwk");
        sayi("       This will create a set of key pairs with random ids and store the result in the file kwys.jwk");
        sayi("   create_keys");
        sayi("        with no arguments, a full set of keys will be created and printed to the command line.");
        sayi("\nE.g.   ");
        sayi("   create_keys " + CL_IS_PUBLIC_FLAG + " " + CL_INPUT_FILE_FLAG + " keys.jwk " + CL_OUTPUT_FILE_FLAG + "  pub_keys.jwk");
        sayi("        This will take the full set of keys in keys.jwk extract the public keys and place the result in pub_keys.jwk");
        sayi("        Note: including the -public flag implies the -in argument is given and an error will result if it is not found");
        sayi("If you invoke help with an argument of ec, you will get information on generating elliptic curve keys. ");
        sayi("If you supply rsa as the argument, you will et information on generating RSA keys");
        sayi("E.g.");
        sayi("create_keys --help ec");
        say("See also create_public_keys");
    }

    /**
     * Little fudging of Throwable vs Exceptions. Introspection requires that certain methods
     * only throw {@link Exception} and the constructor throws {@link Throwable}.
     *
     * @param oa2SE
     * @return
     * @throws Exception
     */
    protected SigningCommands createSG(OA2SE oa2SE) throws Exception {
        SigningCommands sg;
        try {
            sg = new SigningCommands(getDriver(), oa2SE);
            return sg;
        } catch (Throwable t) {
            if (t instanceof Exception) {
                throw (Exception) t;
            }
            throw new GeneralException("error creating signing commands", t);
        }
    }

    /**
     * Generate and add keys to an existing key set. If the key set is empty or missing, it will be created.
     * Note that this generates full sets of keys. If a file is specified, then that will be updated rather than
     * the currently active set of keys.
     *
     * @param inputLine
     * @throws Exception
     */
    public void add_keys(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            addKeysHelps();
            return;
        }
        boolean hasInputFile = inputLine.hasArg(CL_INPUT_FILE_FLAG);
        boolean hasOutputFile = inputLine.hasArg(CL_OUTPUT_FILE_FLAG);
        boolean isPublic = inputLine.hasArg(CL_IS_PUBLIC_FLAG);
        // if this is set, they got confused and should be prompted.
        if (isPublic) {
            say("warning: This will create a new set of public and private keys. ");
            if (!"y".equalsIgnoreCase(getInput("Do you want to continue?[y/n]"))) {
                say("aborted...");
                return;
            }
        }
        JSONWebKeys sourceKeys = this.keys;
        if (hasInputFile) {
            String contents = null;
            try {
                contents = FileUtil.readFileAsString(inputLine.getNextArgFor(CL_INPUT_FILE_FLAG));
            } catch (Throwable e) {
                if(getDriver().isTraceOn()){
                    e.printStackTrace();
                }
                say("error reading input file: " + e.getMessage());
                return;
            }
            sourceKeys = getJwkUtil().fromJSON(contents);
        }
        SigningCommands sg = createSG(null);
        JSONWebKeys keys = sg.createJsonWebKeys();
        JSONObject jwks = getJwkUtil().toJSON(keys);
        // While really unlikely that there would be a key collision, having one could be catastrophic
        // for users. Therefore, only add keys if there are no clashes.
        for (String x : keys.keySet()) {
            if (!sourceKeys.containsKey(x)) {
                sourceKeys.put(keys.get(x));
            }
        }
        if (hasOutputFile) {
            writeFile(inputLine.getNextArgFor(CL_OUTPUT_FILE_FLAG), getJwkUtil().toJSON(sourceKeys).toString(2));
        }
        say("done!");

    }

    private void addKeysHelps() {
        say("add_keys [" + CL_INPUT_FILE_FLAG + " in_file " + CL_OUTPUT_FILE_FLAG + " out_file]");
        sayi("Generates a new set of private keys and adds them to an existing key store.");
        sayi("If " + CL_INPUT_FILE_FLAG + " is specified, then that is used as the existing set, otherwise ");
        sayi("the current set of keys is used. ");
        sayi("If " + CL_OUTPUT_FILE_FLAG + " is specified, the result is written to that file.");
        say("See also, create_keys, set_keys");
    }

    String FORCE_TO_STD_OUT_FLAG = "-o";
    String CREATE_SINGLE_KEY_FLAG = "-single";
    String CREATE_ELLIPTIC_KEY_FLAG = "-ec";
    String SET_DEFAULT_ID = "-default_id";
    String ALGORITHM_KEY_FLAG = "-alg";
    String ELLIPTIC_CURVE_FLAG = "-curve";
    String RSA_KEY_SIZE_FLAG = "-size";

    public void create_keys(InputLine inputLine) throws Exception {
        // Intercept the help request here since the one in the signing utility is a bit different.
        if (showHelp(inputLine)) {
            if (inputLine.hasNextArgFor("--help")) {
                TreeSet<String> algs = new TreeSet<>();
                switch (inputLine.getNextArgFor("--help")) {
                    case "ec":
                        TreeSet<String> curves = new TreeSet<>();
                        curves.addAll(Arrays.asList(JWKUtil2.ALL_EC_CURVES));
                        algs.addAll(Arrays.asList(JWKUtil2.ALL_EC_ALGORITHMS));
                        say("Supported curves");
                        say("----------------");
                        for (String x : curves) {
                            say(x);
                        }
                        say("The default is " + JWKUtil2.EC_CURVE_P_256);
                        say("\nSupported algorithms");
                        say("--------------------");
                        for (String x : algs) {
                            say(x);
                        }
                        say("The default is " + JWKUtil2.ES_256);
                        break;
                    case "rsa":
                        say("Supported algorithms");
                        say("--------------------");
                        algs.addAll(Arrays.asList(JWKUtil2.ALL_RSA_ALGORITHMS));
                        for (String x : algs) {
                            say(x);
                        }
                        say("The default is " + JWKUtil2.RS_256);
                        break;
                    default:
                        say("no help for this topic.");
                }
            } else {

                createKeysHelps();
            }
            return;
        }
        // Fingers and toes cases
        // #1 no arguments, create the keys and dump to std out
        if (!inputLine.hasArgs() || (inputLine.getArgCount() == 1 && inputLine.hasArg(CREATE_SINGLE_KEY_FLAG))) {
            SigningCommands sg = createSG(null);
            sg.create(inputLine);
            return;
        }
        boolean hasDefaultID = inputLine.hasArg(SET_DEFAULT_ID);
        String defaultID = null;
        if (hasDefaultID) {
            defaultID = inputLine.getNextArgFor(SET_DEFAULT_ID);
            inputLine.removeSwitchAndValue(SET_DEFAULT_ID);
        }
        boolean isCreateElliptic = inputLine.hasArg(CREATE_ELLIPTIC_KEY_FLAG);
        inputLine.removeSwitch(CREATE_ELLIPTIC_KEY_FLAG);
        boolean createSingleKey = inputLine.hasArg(CREATE_SINGLE_KEY_FLAG);
        inputLine.removeSwitch(CREATE_SINGLE_KEY_FLAG);
        boolean writeToStdOut = inputLine.hasArg(FORCE_TO_STD_OUT_FLAG);
        inputLine.removeSwitch(FORCE_TO_STD_OUT_FLAG);
        String algorithm = JWKUtil2.RS_256; // RSA default
        if (inputLine.hasArg(ALGORITHM_KEY_FLAG)) {
            algorithm = inputLine.getNextArgFor(ALGORITHM_KEY_FLAG);
            inputLine.removeSwitchAndValue(ALGORITHM_KEY_FLAG);
        } else {
            if (isCreateElliptic) {
                algorithm = JWKUtil2.ES_256; // Default for elliptic curves
            }
        }
        String ellipticCurve = null;
        if (inputLine.hasArg(ELLIPTIC_CURVE_FLAG)) {
            ellipticCurve = inputLine.getNextArgFor(ELLIPTIC_CURVE_FLAG);
            inputLine.removeSwitchAndValue(ELLIPTIC_CURVE_FLAG);
            isCreateElliptic = true; // implicit
        } else {
            // If they are requesting a single key and do not specify this,
            // use P-256. Otherwise, leave null so default set of keys is generated.
            if (isCreateElliptic && createSingleKey) {
                ellipticCurve = JWKUtil2.EC_CURVE_P_256; //
            }
        }
        int keySize = 2048;
        if (inputLine.hasArg(RSA_KEY_SIZE_FLAG)) {
            try {
                keySize = inputLine.getNextIntArg(RSA_KEY_SIZE_FLAG);
            } catch (Throwable t) {
                say("sorry, but " + inputLine.getNextArgFor(RSA_KEY_SIZE_FLAG) + " could not be parsed.");
                return;
            }
            inputLine.removeSwitchAndValue(RSA_KEY_SIZE_FLAG);
        }
        // #2 Error case that public keys are wanted, but no input file is specified.
        if (inputLine.hasArg(CL_IS_PUBLIC_FLAG) && !inputLine.hasArg(CL_INPUT_FILE_FLAG)) {
            say("Error! Request for public keys but no set odf keys supplied.");
            return;
        }
        boolean isPublic = inputLine.hasArg(CL_IS_PUBLIC_FLAG);
        inputLine.removeSwitch(CL_IS_PUBLIC_FLAG);
        boolean isPrivate = inputLine.hasArg(CL_IS_PRIVATE_FLAG);
        inputLine.removeSwitch(CL_IS_PRIVATE_FLAG);
        if (isPrivate && isPublic) {
            String err = " cannot specify both private and public keys at the same time";
            say(err);
            return;
        }
        boolean hasOutputFile = inputLine.hasArg(CL_OUTPUT_FILE_FLAG);
        String outFilename = null;
        if (hasOutputFile) {
            outFilename = inputLine.getNextArgFor(CL_OUTPUT_FILE_FLAG);
            inputLine.removeSwitchAndValue(CL_OUTPUT_FILE_FLAG);
        }
        boolean hasInputFile = inputLine.hasArg(CL_INPUT_FILE_FLAG);
        String inFilename = null;
        if (hasInputFile) {
            inFilename = inputLine.getNextArgFor(CL_INPUT_FILE_FLAG);
            inputLine.removeSwitchAndValue(CL_INPUT_FILE_FLAG);
        }
        if (!isPublic) {

            // next case is to just generate the full key set
            SigningCommands sg = createSG(null);
            JSONWebKeys keys;
            if (createSingleKey) {
                keys = new JSONWebKeys(null);
                if (isCreateElliptic) {
                    keys.put(SigningCommands.createECJWK(ellipticCurve, algorithm)); // make one
                } else {
                    keys.put(createRSAJWK(keySize, algorithm)); // make one
                }
            } else {
                if (isCreateElliptic) {
                    keys = sg.createECJsonWebKeys(ellipticCurve, defaultID); // make full set
                } else {
                    keys = sg.createRSAJsonWebKeys(keySize, defaultID); // make full set
                }

            }
            this.keys = keys;
            JSONObject jwks = getJwkUtil().toJSON(keys);
            if (hasOutputFile) {
                writeFile(outFilename, jwks.toString(2));
            }
            if (writeToStdOut) {
                say(jwks.toString(2));
            }
            return;
        }
        // final case, generate the public keys.
        String contents = null;
        try {
            contents = FileUtil.readFileAsString(inFilename);
        } catch (Throwable e) {
            if(getDriver().isTraceOn()){
                e.printStackTrace();
            }
            say("error reading " + inFilename);
            return;
        }

        JSONWebKeys keys = getJwkUtil().fromJSON(contents);
        JSONWebKeys targetKeys = getJwkUtil().makePublic(keys);
        JSONObject zzz = getJwkUtil().toJSON(targetKeys);
        if (hasOutputFile) {
            writeFile(outFilename, zzz.toString(2));
        } else {
            say(zzz.toString(2));
        }
    }

    protected void showSymmetricKeyHelp(SigningCommands signingCommands) {
        say("create_symmetric_keys [" + signingCommands.SYMMETRIC_KEY_ARG + " len + | " +
                signingCommands.SYMMETRIC_KEY_COUNT_ARG + " count | " + signingCommands.SYMMETRIC_KEY_FILE_ARG +
                " fileName] ");
        sayi("This will create a key for use as a symmetric key, i.e., this will produce");
        sayi("a base 64 encoded sequence of random bytes to be used as a symmetric key for");
        sayi("the given length. If no length is included, the default of " +
                signingCommands.defaultSymmetricKeyLength + " bytes is used.");
        sayi("If the " + signingCommands.SYMMETRIC_KEY_COUNT_ARG + " is given, this will produce that many keys");
        sayi("If the " + signingCommands.SYMMETRIC_KEY_FILE_ARG + " is given, this will write the keys to the given file, one per line.");

    }

    public void create_password(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            sayi("create_password [length] - create a password with the given length in bytes.");
            sayi("No argument implies an 8 byte password. Result is printed to standard out.");
            say("E.g.");
            say("\njwk>create_password 6");
            say("9DbPhjwB");
            sayi("\n6 bytes creates a password that is 6*(4/3) = 8 characters long.");
            return;
        }
        SigningCommands signingCommands = createSG(null);
        int defaultLength = 8;
        try {
            defaultLength = Integer.parseInt(inputLine.getLastArg()); // if it works, it works
        } catch (Throwable t) {
            // do nothing
        }
        // make sure these get propagated
        InputLine inputLine1 = new InputLine("x " + signingCommands.SYMMETRIC_KEY_ARG + " " + defaultLength + " " + signingCommands.SYMMETRIC_KEY_COUNT_ARG + " 1");
        signingCommands.create_symmetric_keys(inputLine1);
    }


    public void create_symmetric_keys(InputLine inputLine) throws Exception {
        SigningCommands signingCommands = createSG(null);
        // make sure these get propagated

        if (showHelp(inputLine)) {
            showSymmetricKeyHelp(signingCommands);
            return;
        }

        signingCommands.create_symmetric_keys(inputLine);
    }

    JSONWebKeys keys = null;

    String wellKnown = null;

    public void print_well_known(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printWellKnownHelp();
            return;
        }
        if (wellKnown == null || wellKnown.isEmpty()) {
            say("(not set)");
            return;
        }
        say("well known URL=\"" + wellKnown + "\"");
    }

    protected void printWellKnownHelp() {
        say("print_well_known: Prints the well-known URL that has been set.");
        sayi("Note that you set it in the set_keys call if you supply its URL");
        sayi("The well-known URL resides on a server and has the public keys listed");
        sayi("While you can validate a signature against it, you cannot create one since");
        sayi("the private key is never available through the well-knwon file.");
        say("Related: set_keys, validate_token");
    }


    protected void setKeysHelp() {
        say("set_keys: [" + CL_INPUT_FILE_FLAG + " filename | " + CL_WELL_KNOWN_FLAG + " uri]");
        sayi("Set the keys used for signing and validation in this session.");
        sayi("Either supplied a fully qualified path to the file or a uri. If you pass nothing");
        sayi("you will be prompted for a file. You can invoke this at any to change the keys.");
        say("See also: create_keys, set_default_id");
    }

    /**
     * Set the keys to be used for signing and validation.
     *
     * @param inputLine
     * @throws Exception
     */
    public void set_keys(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            setKeysHelp();
            return;
        }
        File f = null;
        if (inputLine.size() == 1) {
            boolean getFile = getBooleanInput("Did you want to enter a file name?");
            if (getFile) {
                String fileName = getInput("Enter file name");
                f = new File(fileName);
            } else {
                return;
            }
        }
        if (inputLine.hasArg(CL_INPUT_FILE_FLAG)) {

            f = new File(inputLine.getNextArgFor(CL_INPUT_FILE_FLAG));
            if (!f.exists()) {
                say("Sorry, the file you specified, \"" + (inputLine.getArg(1)) + "\" does not exist.");
                return;
            }
        }
        if (f != null) {
            // got a file from some place. Rock on.
            keys = readKeys(f);
            if (defaultKeyID != null) {
                if (keys.containsKey(defaultKeyID)) {
                    keys.setDefaultKeyID(defaultKeyID);
                }
            }
        } else {
            wellKnown = inputLine.getNextArgFor(CL_WELL_KNOWN_FLAG);
            try {
                keys = JWTUtil.getJsonWebKeys(new ServiceClient(URI.create("https://cilogon.org")), wellKnown);
            } catch (Throwable t) {
                sayi("Sorry, could not parse the url: \"" + t.getMessage() + "\"");
                //throw t;
            }
        }
    }


    protected JSONWebKeys readKeys(File file) throws Exception {
        return getJwkUtil().fromJSON(file);
    }

    public static String LOAD_KEY = "-load";
    public static String PEM_KEY = "-pem";

    public void read_key(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            say("read_key [" + LOAD_KEY + "][" + PEM_KEY + "] file_path - read a single JWK format key.");
            say(LOAD_KEY + " = load the key and set as the default");
            say(PEM_KEY + " = read the key in PEM (PKCS 1) format.");
            say("Not loading the key will simply print it out.");
            return;
        }
        boolean isLoad = inputLine.hasArg(LOAD_KEY);
        boolean isPem = inputLine.hasArg(PEM_KEY);
        inputLine.removeSwitch(LOAD_KEY);
        inputLine.removeSwitch(PEM_KEY);
        JSONWebKeys jsonWebKeys;
        if (isPem) {
            KeyPair keyPair = KeyUtil.keyPairFromPKCS1(new FileReader(inputLine.getLastArg()));
            JSONWebKey jwk = new JSONWebKey();
            jwk.id = "dummy";
            jwk.privateKey = keyPair.getPrivate();
            jwk.publicKey = keyPair.getPublic();
            jwk.type = "RSA";
            jwk.use = "sig";
            jwk.algorithm = "RS256"; //safe bet, but no actual way to tell.

            //jwk.algorithm;
            jsonWebKeys = new JSONWebKeys("dummy");
            jsonWebKeys.put(jwk);
        } else {
            jsonWebKeys = getJwkUtil().fromJSON(new File(inputLine.getLastArg()));
        }
        if (isLoad) {
            keys = jsonWebKeys;
        } else {
            say(getJwkUtil().toJSON(jsonWebKeys).toString(1));
        }

    }

    protected void listKeysHelp() {
        say("list_keys [" + showAllKeys + " " + CL_INPUT_FILE_FLAG + " file]:This will list all the public keys " +
                "in the key file in pem format.");
        sayi("Each key will be preceeded by its unique ID in the key file.");
        sayi("You may invoke this with no argument, in which case the default key file");
        sayi("as set in the set_keys command will be used, or you can supply a fully qualified");
        sayi("path to a JSON web key file that will be used.");
        sayi("If you supply the " + showAllKeys + " flag then the private key in PKCS 8 format will be shown");
        sayi("too. Note the default is to not show the private key.");
        say("  Related: set_keys, create_keys, print_public_keys (prints in JSON format)");
    }

    protected String showAllKeys = "-showAll";

    protected void printPublicKeysHelp() {
        say("print_public_keys [file]: This will print the public keys only for a key set.");
        sayi("Note that if no file is supplied the current key set is used.");
        sayi("The result is JSON formatted. If you need PEM format use list_keys instead.");
    }

    /**
     * Prints the public keys in JSON format.
     *
     * @param inputLine
     */
    public void print_public_keys(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printPublicKeysHelp();
            return;
        }
        JSONWebKeys localKeys = null;
        if (inputLine.hasArgs()) {
            File publicKeyFile = new File(inputLine.getLastArg());
            localKeys = readKeys(publicKeyFile);

        } else {
            localKeys = keys;
        }
        JSONWebKeys targetKeys = getJwkUtil().makePublic(localKeys);
        JSONObject zzz = getJwkUtil().toJSON(targetKeys);
        say(zzz.toString(2));
    }

    public String BASE64_FLAG = "-b64";

    protected void createPublicKeysHelp() {
        say("create_public_keys [" + BASE64_FLAG + "] [" + CL_INPUT_FILE_FLAG + " in_file] [" + CL_OUTPUT_FILE_FLAG + " out_file]");
        sayi("Take a set of private keys and extract the public keys.");
        sayi("If there is no input file, the current set of keys is used.");
        sayi("If the " + CL_OUTPUT_FILE_FLAG + " switch is given, the result will be written to the file.");
        sayi("If there is no output file specified, then the keys are printed at the console.");
        sayi("If the " + BASE64_FLAG + " flag is used, then the entire contents is base 64 encoded before");
        sayi("being displayed or written to the file.");
        say("See also: create_keys, base64");


    }

    public void create_public_keys(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            createPublicKeysHelp();
            return;
        }
        boolean hasInputFile = inputLine.hasArg(CL_INPUT_FILE_FLAG);
        boolean hasOutputFile = inputLine.hasArg(CL_OUTPUT_FILE_FLAG);
        boolean doB64 = inputLine.hasArg(BASE64_FLAG);


        JSONWebKeys localKeys = null;
        if (hasInputFile) {
            File publicKeyFile = new File(inputLine.getNextArgFor(CL_INPUT_FILE_FLAG));
            localKeys = readKeys(publicKeyFile);

        } else {
            if (keys == null) {
                say("Sorry, there is no set of active keys and no input file was specified. Exiting....");
                return;
            }
            localKeys = keys;
        }

        JSONWebKeys targetKeys = getJwkUtil().makePublic(localKeys);
        JSONObject zzz = getJwkUtil().toJSON(targetKeys);
        String finalOutput = zzz.toString(2);
        if (doB64) {
            finalOutput = Base64.encodeBase64String(finalOutput.getBytes());
        }
        if (hasOutputFile) {
            try {
                FileUtil.writeStringToFile(inputLine.getNextArgFor(CL_OUTPUT_FILE_FLAG), finalOutput);
            } catch (Throwable iox) {
                say("uh-oh... Could not write to the output file:" + iox.getMessage());
            }
        } else {
            say(finalOutput);
        }
    }

    /**
     * Write the contents of a file as a string.
     *
     * @param filename
     * @param contents
     * @throws Exception
     */
    protected void writeFile(String filename, String contents) throws Exception {
        File f = new File(filename);
        FileWriter fileWriter = new FileWriter(f);
        fileWriter.write(contents);
        fileWriter.flush();
        fileWriter.close();
    }


    public void list_keys(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            listKeysHelp();
            return;
        }
        boolean showPrivateKeys = inputLine.hasArg(showAllKeys);
        boolean hasInputFile = inputLine.hasArg(CL_INPUT_FILE_FLAG);
        JSONWebKeys localKeys = null;
        if (showPrivateKeys) {
            if (hasInputFile) {
                // ok.
            } else {
                say("warning, there are no private keys to display.");
                showPrivateKeys = false;
            }
        }
        if (hasInputFile) {
            File publicKeyFile = new File(inputLine.getNextArgFor(CL_INPUT_FILE_FLAG));
            localKeys = readKeys(publicKeyFile);

        } else {
            localKeys = keys;
        }
/*
        if (showPrivateKeys && !hasInputFile) {
            // try to use the defined keys
            if (keys == null || keys.isEmpty()) {
                say("Sorry, there are no keys specified. Either use set_keys or specify a key file.");
                return;
            }
            localKeys = keys;
        } else {
            File publicKeyFile = new File(inputLine.getNextArgFor(CL_INPUT_FILE_FLAG));
            localKeys = readKeys(publicKeyFile);
        }
*/
        boolean hasDefault = localKeys.hasDefaultKey();
        String defaultKey = null;
        if (hasDefault) {
            defaultKey = localKeys.getDefaultKeyID();
        }
        boolean isFirst = true;
        for (String key : localKeys.keySet()) {
            if (isFirst) {
                isFirst = false;
            } else {
                say(""); // blank line between keys.
            }
            if (hasDefault) {
                if (key.equals(defaultKey)) {
                    say("key id=" + key + " (default)");
                } else {
                    say("key id=" + key);
                }
            } else {
                say("key id=" + key);
            }
            say(KeyUtil.toX509PEM(localKeys.get(key).publicKey));
            if (showPrivateKeys) {
                say(KeyUtil.toPKCS8PEM(localKeys.get(key).privateKey));
            }
        }

    }


    protected void printCreateClaimsHelp() {
        say("create_claims: Prompt the user for key/value pairs and build a claims object. ");
        sayi("This will write the object to a file for future use.");
        sayi("Note: You may input JSON objects as values as well. There are various");
        sayi("places (such as creating a token) that requires a set of claims. This command");
        sayi("lets you create one.");
        say("See also: create_token, parse_claims");
    }

    /**
     * Create a set of claims and write them to a file in JSON format.
     *
     * @param inputLine
     * @throws Exception
     */
    public void create_claims(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printCreateClaimsHelp();
            return;
        }
        say("Enter a key then a value when prompted. You can enter multiple values separated by commas");
        say("Just hit return (no value) to exit");
        boolean isDone = false;
        JSONObject jsonObject = new JSONObject();
        while (!isDone) {
            String key = getInput("Enter key or return to exit.");
            if (isEmpty(key)) {
                isDone = true;
                continue;
            }
            String value = getInput("Enter value. multiple values should be comma separated");
            try {
                // try JSON first
                JSON json = JSONObject.fromObject(value);
                jsonObject.put(key, json);
            } catch (Throwable t) {
                // plan B. Did they give us a comma separated list we are to parse?
                if (0 < value.indexOf(",")) {
                    StringTokenizer st = new StringTokenizer(value, ",");
                    JSONArray array = new JSONArray();
                    while (st.hasMoreTokens()) {
                        array.add(st.nextToken());
                    }
                    jsonObject.put(key, array);
                } else {
                    // ok, nothing works, it's just a string...
                    jsonObject.put(key, value);

                }
            }
        }
        sayi("Here's what you inputted");
        say(jsonObject.toString(2));
        boolean isWrite = getBooleanInput("Would you like to write this to a file?[y/n]");
        //Boolean isWrite = Boolean.parseBoolean(writeToFile);
        if (isWrite) {
            String fileName = getInput("Enter filename");
            File f = new File(fileName);
            if (f.exists()) {
                String overwrite = getInput("This file exists. Do you want to overwrite it?", "false");
                if (!Boolean.parseBoolean(overwrite)) {
                    return;
                }
            }
            String output = jsonObject.toString(2);
            FileOutputStream fos = new FileOutputStream(f);
            fos.write(output.getBytes());
            fos.flush();
            fos.close();
            sayi(f + " written!");
        }
    }


    protected boolean getBooleanInput(String prompt) throws IOException {
        String x = getInput(prompt, "y");
        if (x.equalsIgnoreCase("y") || x.equalsIgnoreCase("yes") || x.equalsIgnoreCase("true")) return true;
        return false;
    }

    protected String getInput(String prompt) throws IOException {
        String inLine = readline(prompt + ":");
        if (isEmpty(inLine)) {
            return null; // no input. User hit a return
        }
        return inLine;
    }


    String defaultKeyID = null;

    protected void printSetDefaultIDHelp() {
        say("set_default_id [keyid]: This will set the default key id to be used for all signing and verification.");
        sayi("If this is not set, you will be prompted each time for an id.");
        sayi("Remember that a set of web keys does not have a default. If you import.");
        sayi("a set, you should set one as default.");
        say("See also: print_default_id");
    }

    public void set_default_id(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printSetDefaultIDHelp();
            return;
        }
        if (1 < inputLine.size()) {
            defaultKeyID = inputLine.getArg(1);
            return;
        }
        String x = getInput("Enter the key id");
        // do nothing if there is no value supplied.
        if (isEmpty(x)) {
            return;
        }
        defaultKeyID = x;

    }

    protected void printPrintDefaultIDHelp() {
        say("print_default_id: This will print the current default key id that is to be used for all signing and verification.");
        say("See also: set_default_id");
    }

    public void print_default_id(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printSetDefaultIDHelp();
            return;
        }
        if (defaultKeyID == null || defaultKeyID.isEmpty()) {
            say("(not set)");
            return;
        }
        say("default key id=\"" + defaultKeyID + "\"");
    }

    protected void printParseClaimsHelp() {
        say("parse_claims [" + CL_INPUT_FILE_FLAG + " filename]");
        sayi("Read a file and print out if it parses as JSON.");
        sayi("If the filename is omitted, you will be prompted for it.");
        sayi("Note that this will try to give some limited feedback in syntax errors.");
        sayi("The intent is that if you have written a file with claims, this lets you");
        sayi("validate the JSON before using it.");
        say("See also: create_claims");
    }

    /**
     * Read the claims in a file and verify that they are a valid JSON object.
     *
     * @param inputLine
     * @throws Exception
     */
    public void parse_claims(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printParseClaimsHelp();
            return;
        }
        boolean hasInputFile = inputLine.hasArg(CL_INPUT_FILE_FLAG);
        String filename = null;
        if (hasInputFile) {
            filename = inputLine.getNextArgFor(CL_INPUT_FILE_FLAG);
        } else {
            filename = getInput("Enter full path to the claims file.");
            if (isEmpty(filename)) {
                say("No claims file. Exiting...");
                return;
            }
        }
        String rawJSON = null;
        try {
            rawJSON = FileUtil.readFileAsString(filename);
        } catch (Throwable e) {
            if(getDriver().isTraceOn()){
                e.printStackTrace();
            }
            say("could not read file: " + filename);
            return;
        }
        if (rawJSON == null) {
            say("Could not read the file \"" + filename + "\"");
            return;
        }
        JSON jsonObject = null;
        try {
            jsonObject = JSONObject.fromObject(rawJSON);
        } catch (Throwable t) {
            say("Parsing fail with a message of \"" + t.getMessage() + "\"");
            return;
        }
        if (jsonObject != null) {
            say("success!");
            say(jsonObject.toString(3));
        } else {
            say("No JSON object resulted from parsing.");
        }
    }


    /**
     * This will take an input line and search for the arg, returning the next value.
     * E.g. if the input line is  "-file y -id z -v"
     * then supplying "-file" as the key here will return "y".
     * If there is no such key, then a null is returned.
     *
     * @param inputLine
     * @param key
     * @return
     */
    protected String getArgValue(InputLine inputLine, String key) {
        return inputLine.getNextArgFor(key);
    }

    // CL = command line flags. Once upon a time, but I decided to standardize the command line flags here.
    protected String CL_KEY_FILE_FLAG = "-keys";
    protected String CL_KEY_ID_FLAG = "-key_id";
    protected String CL_WELL_KNOWN_FLAG = "-key_id";
    protected String CL_IS_PUBLIC_FLAG = "-public";
    protected String CL_IS_PRIVATE_FLAG = "-private";

    protected void createTokenHelp() {
        say("create_token " + CL_INPUT_FILE_FLAG + " claims " +
                "[" + CL_KEY_FILE_FLAG + " keyfile " + CL_KEY_ID_FLAG + " id " + CL_OUTPUT_FILE_FLAG + " outputFile]");
        sayi("Interactive mode:                                                                     ");
        sayi("   This will take the current keys (uses default) and a file containing a JSON");
        sayi("   format set of claims. It will then sign the claims with the right headers etc.");
        sayi("   and optionally print out the resulting JWT to the console. Any of the arguments omitted ");
        sayi("   will cause you to be prompted. NOTE that this only signs the token! If you need to generate");
        sayi("   accounting information like the timestamps, please use generate_token instead");
        sayi("   If you have already set the key and keyid these will be used.");
        sayi("   If the output file is given, the token will be written there instead.");
        sayi("");
        sayi("   E.g.                                                                                            ");
        sayi("   create_token " + CL_KEY_FILE_FLAG + " keys.jwk " + CL_KEY_ID_FLAG + " ABC123 " + CL_INPUT_FILE_FLAG +
                " my_claims.txt " + CL_OUTPUT_FILE_FLAG + " my_token.jwt");
        sayi("  Will read the keys in the file keys.jwk, select the one with id ABC123 then                   ");
        sayi("  read in the my_claims.txt file (assumed to be a set of claims in JSON format)                 ");
        sayi("  and create the header and signature. It will then place the result into the file my_token.jwt ");
        sayi("                                                                                                ");
        sayi("create_token " + CL_WELL_KNOWN_FLAG + " https://fnord.baz/.well-known " + CL_KEY_ID_FLAG +
                " CAFEBEEF " + CL_INPUT_FILE_FLAG + " my_claims.txt           ");
        sayi("This will read the well-known file, parse it for the keys, load the keys, find the key       ");
        sayi("with id CAFEBEEF read in the claims file then print the resulting token to the command line. ");
        say("Related: generate_token, set_keys, set_default_id, print_token, verify_token");
    }

    String lastToken = null;

    public void create_token(InputLine inputLine) throws Exception {

        if (showHelp(inputLine)) {
            createTokenHelp();
            return;
        }
        // pull off the command line arguments
        File outputFile = null;
        if (inputLine.hasArg(CL_OUTPUT_FILE_FLAG)) {
            outputFile = new File(inputLine.getNextArgFor(CL_OUTPUT_FILE_FLAG));
        }

        JSONWebKeys localKeys = null;
        if (inputLine.hasArg(CL_KEY_FILE_FLAG)) {
            String fileName = getArgValue(inputLine, CL_KEY_FILE_FLAG);
            File f = new File(fileName);
            if (!f.exists()) {
                say("Sorry, that file does not seem to exist");
                return;
            }
            if (!f.isFile()) {
                say("Sorry, the thing you specified is not a file.");
                return;
            }
            localKeys = readKeys(f);
        }
        if (inputLine.hasArg(CL_WELL_KNOWN_FLAG)) {
            localKeys = getJsonWebKeys(inputLine.getNextArgFor(CL_WELL_KNOWN_FLAG));

        }
        if (localKeys == null) {
   /*         if (isBatchMode()) {
                // Error in this case -- there cannot be a set of keys defined, so exit
                sayv(" no keys specified");
                return;
            } else {*/
            if (keys == null || keys.isEmpty()) {
                if (getBooleanInput("No keys set. Would you like to specify keys for signing?")) {
                    String x = getInput("Enter fully qualified path and file name");
                    if (isEmpty(x)) {
                        say("no file entered, exiting...");
                        return;
                    }
                    localKeys = readKeys(new File(x));
                }

            } else {
                localKeys = keys;
            }
            //   }
        }
        String localDefaultID = null;
        if (inputLine.hasArg(CL_KEY_ID_FLAG)) {
            localDefaultID = getArgValue(inputLine, CL_KEY_ID_FLAG);
        } else {
            if (defaultKeyID != null) {
                localDefaultID = defaultKeyID;
            } else {
                if (getBooleanInput("No key id found. Do you want to enter one?")) {
                    localDefaultID = getInput("Enter key id:");
                } else {
                    return;
                }
            }
        }
        JSONObject claims = null;
        if (inputLine.hasArg(CL_INPUT_FILE_FLAG)) {
            try {
                claims = JSONObject.fromObject(FileUtil.readFileAsString(getArgValue(inputLine, CL_INPUT_FILE_FLAG)));
            } catch (Throwable e) {
                if(getDriver().isTraceOn()){
                    e.printStackTrace();
                }
                say("error reading input file: " + e.getMessage());
                return;
            }
        } else {
            String x = getInput("Enter the name of the file containing the JSON object to use:");
            if (isEmpty(x)) {
                say("No argument, exiting...");
                return;
            }
            try {
                claims = JSONObject.fromObject(FileUtil.readFileAsString(x));
            } catch (Throwable e) {
                if(getDriver().isTraceOn()){
                    e.printStackTrace();
                }
                say("error reading claims file \"" + x + "\":" + e.getMessage());
            }

        }
        String signedToken = JWTUtil.createJWT(claims, localKeys.get(localDefaultID));
        lastToken = signedToken;
        if (outputFile == null) {
            say(signedToken);
        } else {
            FileWriter fileWriter = new FileWriter(outputFile);
            fileWriter.write(signedToken);
            fileWriter.flush();
            fileWriter.close();
        }
    }

    protected void generateTokenHelp() {
        say("generate_token " +
                CL_INPUT_FILE_FLAG + "  claims " +
                CL_KEY_FILE_FLAG + " keyFile " +
                CL_KEY_ID_FLAG + " keyId " +
                "[" + JTI_FLAG + " | " +
                PRINT_CLAIMS_FLAG + " | " +
                LIFETIME_FLAG + "  lifetime | " +
                CL_OUTPUT_FILE_FLAG + " outFile]");
        sayi("Generate a token from the claims. This includes adding in the current time and using the lifetime (if given)");
        sayi("to create the token. A JTI will also be created. ");
        sayi("The meaning of the various optional flags is as follows");
        sayi(CL_INPUT_FILE_FLAG + " (required) The text file of a JSON object that has the claims.");
        sayi(CL_KEY_FILE_FLAG + " (required) + The JWK format file containing the keys. This must contain a private key.");
        sayi(CL_KEY_ID_FLAG + " (required) The id in the key file of the key to use.");
        sayi(JTI_FLAG + " (optional) If specified, generate a unique identifier for this id token. You may also just");
        sayi("    put one in the claims file if you need it immutable.");
        sayi(PRINT_CLAIMS_FLAG + " (optional) If specified, this will print out the generated claims (not token!) to the command line.");
        sayi(LIFETIME_FLAG + " (optional) Specifies the lifetime in seconds for this token. The default is " + DEFAULT_LIFETIME + " seconds.");
        sayi("    Note: not specifying an output file will print the resulting token.");
        sayi(CL_OUTPUT_FILE_FLAG + " (optional) The file to which the resulting token is written. Omitting this dumps it to the command line.");
    }


    protected String LIFETIME_FLAG = "-lifetime";
    protected String JTI_FLAG = "-jti";
    protected String PRINT_CLAIMS_FLAG = "-print_claims";
    protected long DEFAULT_LIFETIME = 600; // in seconds.
    /*
    This next constant takes a wee bit of explaining. We create a big int and then want to use it as an identifier.
    We also want to avoid collisions, so a rather long sequence of random bytes is good. Rather than just pass along
    a big integer, we could encode it with a radix of 16 (turning it into a hex number) or in this case, use a radix of
    36 which uses all 26 letters of the alphabet and digits. This makes it much more compact.
    If you really need to, you reconstruct this with the (String,int) constructor for BigInteger.
    (Being a Math guy this just seemed natural, but I decided to stick a note here to explain it.)
     */
    protected int JTI_RADIX = 36;

    public void generate_token(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            generateTokenHelp();
            return;
        }
        if (gracefulExit(!inputLine.hasArg(CL_INPUT_FILE_FLAG), "Missing claims file.")) return;
        if (gracefulExit(!inputLine.hasArg(CL_KEY_FILE_FLAG), "Missing keys file.")) return;
        if (gracefulExit(!inputLine.hasArg(CL_KEY_ID_FLAG), "Missing key id for signature.")) return;

        String localDefaultID = getArgValue(inputLine, CL_KEY_ID_FLAG);
        JSONWebKeys localKeys = readKeys(new File(inputLine.getNextArgFor(CL_KEY_FILE_FLAG)));
        if (gracefulExit(!localKeys.containsKey(localDefaultID), "The key id is not in the key set. Check the id."))
            return;

        long lifetime = DEFAULT_LIFETIME;

        if (inputLine.hasArg(LIFETIME_FLAG)) {
            lifetime = Long.parseLong(inputLine.getNextArgFor(LIFETIME_FLAG));
        }
        long issuedAt = new Date().getTime(); // in milliseconds. Eventually this turns into seconds.
String x = inputLine.getNextArgFor(CL_INPUT_FILE_FLAG);
        JSONObject claims = null;
        try {
            claims = readJSON(x);
        } catch (Throwable e) {
            if(getDriver().isTraceOn()){
                e.printStackTrace();
            }
            say("error reading file \"" + x + "\": " + e.getMessage());
            return;
        }
        // now we set the claims.

        claims.put(OA2Claims.ISSUED_AT, issuedAt / 1000);
        claims.put(OA2Claims.NOT_VALID_BEFORE, issuedAt / 1000);
        //  claims.put(OA2Claims.AUTH_TIME, issuedAt/1000);
        claims.put(OA2Claims.EXPIRATION, (issuedAt / 1000) + lifetime);
        if (inputLine.hasArg(JTI_FLAG)) {
            // A JTI flag means create a random JTI. Otherwise, the user should just stick it in the file..
            String jti = "";
            SecureRandom secureRandom = new SecureRandom();
            byte[] secret = new byte[32];
            secureRandom.nextBytes(secret);
            BigInteger bigInteger = new BigInteger(secret);
            bigInteger = bigInteger.abs(); // so no signs in final output

            jti = "jti://" + bigInteger.toString(JTI_RADIX);// default is lower case. Note this has 0 (zero) and o (letter "oh")!
            claims.put(OA2Claims.JWT_ID, jti);
        }
        if (inputLine.hasArg(PRINT_CLAIMS_FLAG)) {
            say(claims.toString(2));
        }
        String signedToken = JWTUtil.createJWT(claims, localKeys.get(localDefaultID));
        if (inputLine.hasArg(CL_OUTPUT_FILE_FLAG)) {
            writeFile(inputLine.getNextArgFor(CL_OUTPUT_FILE_FLAG), signedToken);
        } else {
            say(signedToken);
        }
    }


    protected void printTokenHelp() {
        say("print_token: [" + CL_INPUT_FILE_FLAG + " file | token] Print the given token's header and payload, doing no verification.");
        sayi("If you omit the argument, it will print the last token generated by the create_token call.");
        sayi("If there is no last token, that will be shown too. ");

        say("Related: create_token");
    }

    public void print_token(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printTokenHelp();
            return;
        }
        if (inputLine.isEmpty()) {
            if (lastToken == null) {
                say("(no token has been created)");
                return;
            }
            say(lastToken);
            return;
        }
        String rawToken = null;
        if (inputLine.hasArg(CL_INPUT_FILE_FLAG)) {
            String x = inputLine.getNextArgFor(CL_INPUT_FILE_FLAG);
            try {
                rawToken = FileUtil.readFileAsString(x);
            } catch (Throwable e) {
                if(getDriver().isTraceOn()){
                    e.printStackTrace();
                }
                say("error reading file \"" + x + "\"");
                return;
            }
        } else {
            rawToken = inputLine.getLastArg();
        }
        JSONObject[] payloads = JWTUtil.readJWT(rawToken);
        say("header");
        say(payloads[JWTUtil.HEADER_INDEX].toString(2));
        say("payload");
        say(payloads[JWTUtil.PAYLOAD_INDEX].toString(2));

    }

    protected void printListKeyIDs() {
        say("list_key_ids [filename]");
        sayi("List the unique key ids in the file");
        sayi("If you do not supply an argument, the globally set keys will be used");
        sayi("If there is no default set of keys, you will be prompted for a file");
        say("See also: set_keys, set_default_id");
    }

    public void list_key_ids(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printListKeyIDs();
            return;
        }
        JSONWebKeys jsonWebKeys = null;
        if (1 < inputLine.size()) {
            jsonWebKeys = getJwkUtil().fromJSON(new File(inputLine.getArg(1)));
        } else {
            if (keys == null) {
                if (getBooleanInput("Do you want to enter a file name?")) {
                    String x = getInput("Enter path and name of the key file");
                    jsonWebKeys = getJwkUtil().fromJSON(new File(x));
                } else {
                    return;
                }
            } else {
                jsonWebKeys = keys;

            }
        }
        String defaultWebKey = null;
        if (jsonWebKeys.hasDefaultKey()) {
            defaultWebKey = jsonWebKeys.getDefaultKeyID();
        } else {
            defaultWebKey = defaultKeyID;
        }
        for (String keyID : jsonWebKeys.keySet()) {
            JSONWebKey webKey = jsonWebKeys.get(keyID);
            boolean isDefault = webKey.id.equals(defaultWebKey);
            say("id=" + keyID + ", alg=" + webKey.algorithm + ", type=" + webKey.type + ", use=" + webKey.use + (isDefault ? " (default)" : ""));
        }
    }

    protected void printValidateTokenHelp() {
        say("validate_token [" + CL_WELL_KNOWN_FLAG + " url | " + CL_KEY_FILE_FLAG + " file " + CL_INPUT_FILE_FLAG + " filename  | token]");
        sayi("This will take a token and check the signature. It will also print out the payload");
        sayi("and header information.");
        sayi("The validation is against the current set of keys or against a URL specified with the");
        sayi("-wellKnown flag. You can also point to a key file (file with JSON web keys in it) with");
        sayi("the -keyFile flag.");
        sayi("You may supply either the token itself or specify with the -file flag that this is in a file.");
        sayi("E.g.s                                                                                                          ");
        sayi("validate_token " + CL_WELL_KNOWN_FLAG + " https://foo.bar/.well-known " + CL_INPUT_FILE_FLAG + " my_token.jwt                                         ");
        sayi("   This will read the keys in the well-known file and read the token in the file                                ");
        sayi("                                                                                                             ");
        sayi("validate_token " + CL_WELL_KNOWN_FLAG + " https://foo.bar/.well-known -v " + CL_INPUT_FILE_FLAG + " my_token.jwt                                      ");
        sayi("   Identical behavior to the first example but note the -v flag: This causes any information about              ");
        sayi("   the token to be printed. Normally this is not used except for trying to debug issues.                        ");
        sayi("                                    ");
        sayi("validate_token " + CL_KEY_FILE_FLAG + "  keys.jwk eyJ...........                                                                    ");
        sayi("   This will read in the keys from the give file and the assumption is that the last argument is the token itself");
        sayi("   Note that in this example the token is truncated so it fits here.                                     ");
        say("See also: create_token");
    }

    public void validate_token(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            printValidateTokenHelp();
            return;
        }
        String token = null;
        if (1 == inputLine.size()) {
            say("Sorry, no argument");
            return;
        }
        if (inputLine.hasArg(CL_INPUT_FILE_FLAG)) {
            token = inputLine.getNextArgFor(CL_INPUT_FILE_FLAG);
        } else {
            token = inputLine.getLastArg();
        }
        JSONWebKeys keys = this.keys;
        if (inputLine.hasArg(CL_WELL_KNOWN_FLAG)) {
            String wellKnown = inputLine.getNextArgFor(CL_WELL_KNOWN_FLAG);
            try {
                // Actually we don't use the uri below, but one is needed to create the class
                keys = JWTUtil.getJsonWebKeys(new ServiceClient(URI.create("https://cilogon.org")), wellKnown);
            } catch (Throwable t) {
                sayi("Sorry, could not parse the url: \"" + wellKnown + "\". Message=\"" + t.getMessage() + "\"");
            }
        }
        if (inputLine.hasArg(CL_KEY_FILE_FLAG) && !inputLine.hasArg(CL_WELL_KNOWN_FLAG)) { // only take one if both are specified and well known is preferred
            File f = new File(inputLine.getNextArgFor(CL_KEY_FILE_FLAG));

            if (gracefulExit(!f.exists(), "Sorry, the file \" + f + \" does not exist")) return;

            try {
                keys = readKeys(f);
            } catch (Throwable t) {
                if (gracefulExit(true, "Sorry, could not load the file: \"" +
                        inputLine.getNextArgFor("-keyFile") + "\". Message=\"" + t.getMessage() + "\"")) return;
            }
        }
        if (gracefulExit(keys == null, "Sorry, no keys set, please set keys or specify a well-known URL.")) return;

        String[] x = decat(token);
        /*
                JSONObject h = JSONObject.fromObject(new String(Base64.decodeBase64(x[0])));
        JSONObject p = JSONObject.fromObject(new String(Base64.decodeBase64(x[1])));
        if (JWTUtil.verify(h, p, x[2], keys.get(h.getString("kid")))) {
         */
        Base64URL h = new Base64URL(x[0]);
        Base64URL p = new Base64URL(x[1]);
        Base64URL s = null;
        JSONObject fullHeader = JSONObject.fromObject(new String(Base64.decodeBase64(x[0])));
        if (x.length == 3) {
            /// There is a signature
            s = new Base64URL(new String(x[2]));
        }
        if (JWTUtil.verify(h, p, s, keys.get(fullHeader.getString("kid")))) {
        /*    if (isBatch()) {
                sayv("token valid!");
                return;
            }*/
            say("token valid!");
        } else {
/*
            if (isBatch()) {
                sayv("could not validate token");
                System.exit(1);
            }
*/
            say("could not validate token");
        }
    }


    public static void main(String[] args) {
        try {
            String sig = "L2ZN8jp_-SmPmAiEels5DsGKx-nh--EPo3lgGTqp6Kpp5IpwKrgpK0Wc34Cs2iALYtQqyaqvrWVhr1kZxS9_TI4WrE84BIYlpuFc-hSqKl4JVRHhn0ij_Jg7_Y6KuwPdfKeWNq6L9wUxKJPyIMU3WxGV-Nrcl9nAYt9SlrqMBOA7bARuUQfl1maZ05HRZFImL0Ol1PbAOfnbff74P323dbwzGJ1AxqQIvfVmniJXwr_4K88yZxrcYTs81yse8oT1SAsTffiVKJvwoD4DctMxkYas-_mJPaW-WNylBME8GR-R3f0RjTxJ-xO5WlMP8kbVJ2V5rcdzjirqIWqfF9i1Eg";
            byte[] byetArray = Base64.decodeBase64(sig);
            String path = "/home/ncsa/temp/rokwire/sig.b";
            File f = new File(path);
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            baos.write(byetArray, 0, byetArray.length);
            FileOutputStream fileOutputStream = new FileOutputStream(f);
            baos.writeTo(fileOutputStream);
            fileOutputStream.flush();
            fileOutputStream.close();
        } catch (Throwable t) {
            t.printStackTrace();
        }


    }

    String base64Encode = "-encode";
    String base64Dencode = "-decode";
    String base64Bytes = "-binary";

    protected void base64Help() {
        say("base64 " + base64Encode + " | " + base64Dencode + " " + base64Bytes + " " + CL_INPUT_FILE_FLAG + " in_file " + CL_OUTPUT_FILE_FLAG + " out_file | arg");
        sayi("This will encode or decode a base 64 arg. ");
        sayi("You may specify which input or output. If none is given, then the assumption is that the input is the arg");
        sayi("and the output is to the terminal.");
        sayi(base64Dencode + ": the input is base64 encoded, output is plain text");
        sayi(base64Encode + ": the input is plain text, output is base 64 encoded.");
        sayi(base64Bytes + " treat the output as bytes. Generally this implies you have specified an output file.");
        sayi("Note: i the output is binary, you should specify a file as the target since otherwise you get gibbersih.");
    }

    public void base64(InputLine inputLine) throws Exception {
        if (showHelp(inputLine)) {
            base64Help();
            return;
        }
        boolean hasInfile = inputLine.hasArg(CL_INPUT_FILE_FLAG);
        boolean hasOutfile = inputLine.hasArg(CL_OUTPUT_FILE_FLAG);
        boolean isEncode = inputLine.hasArg(base64Encode);
        boolean isDecode = inputLine.hasArg(base64Dencode);
        boolean isBinary = inputLine.hasArg(base64Bytes);
        gracefulExit(isDecode && isEncode, "Sorry, you cannot specify both encoding and decoding at the same time");

        String input = "";
        String x = inputLine.getNextArgFor(CL_INPUT_FILE_FLAG);
        if (hasInfile) {
            try {
                input = FileUtil.readFileAsString(x);
            } catch (Throwable e) {
                if(getDriver().isTraceOn()){
                    e.printStackTrace();
                }
                say("error reading input file \"" + x + "\":" + e.getMessage());
            }
        } else {
            input = inputLine.getLastArg();
        }
        String output = "";
        if (isEncode) {
            output = Base64.encodeBase64String(input.getBytes());
            if (hasOutfile) {
                writeFile(inputLine.getNextArgFor(CL_OUTPUT_FILE_FLAG), output);
            } else {
                say(output);
            }
        } else {
            // decoding to bytes
            byte[] bytes = Base64.decodeBase64(input);
            if (!hasOutfile) {
                say(new String(bytes));
                return;
            }

            if (isBinary) {
                File f = new File(inputLine.getNextArgFor(CL_OUTPUT_FILE_FLAG));
                FileOutputStream fos = new FileOutputStream(f);
                fos.write(bytes);
                fos.flush();
                fos.close();
            } else {
                writeFile(inputLine.getNextArgFor(CL_OUTPUT_FILE_FLAG), new String(bytes));
            }
        }
    }
}
