package org.oa4mp.server.loader.oauth2.servlet;

import edu.uiuc.ncsa.security.core.Identifier;
import edu.uiuc.ncsa.security.core.exceptions.UnknownClientException;
import edu.uiuc.ncsa.security.core.util.BasicIdentifier;
import edu.uiuc.ncsa.security.core.util.StringUtils;
import edu.uiuc.ncsa.security.servlet.AbstractServlet;
import edu.uiuc.ncsa.security.servlet.HeaderUtils;
import net.sf.json.JSONObject;
import org.apache.http.HttpStatus;
import org.oa4mp.delegation.common.storage.clients.BaseClient;
import org.oa4mp.delegation.server.OA2Constants;
import org.oa4mp.delegation.server.OA2Errors;
import org.oa4mp.delegation.server.OA2GeneralError;
import org.oa4mp.delegation.server.jwt.MyOtherJWTUtil2;
import org.oa4mp.delegation.server.server.RFC7523Constants;
import org.oa4mp.delegation.server.server.RFC8628Constants;
import org.oa4mp.server.loader.oauth2.OA2SE;
import org.oa4mp.server.loader.oauth2.storage.clients.OA2Client;

import javax.servlet.http.HttpServletRequest;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;

import static org.oa4mp.delegation.server.server.claims.OA2Claims.*;
import static org.oa4mp.server.api.ServiceConstantKeys.CONSUMER_KEY;


/**
 * Utilities for dealing with getting tokens that may be either sent as parameters
 * or in the authorization header . Note that you should check that if a user sends both, that they match
 * and throw an exception if they do not.
 * <p>Created by Jeff Gaynor<br>
 * on 9/25/17 at  5:33 PM
 */
public class OA2HeaderUtils extends HeaderUtils {

    public static String getATFromParameter(HttpServletRequest request) {
        String rawID = request.getParameter(OA2Constants.ACCESS_TOKEN);
        if (StringUtils.isTrivial(rawID)) {
            return null;
        }
        return rawID;
    }

    public static Identifier getIDFromParameters(HttpServletRequest request) {
        Identifier paramID = null;

        // assume that the secret and id are in the request
        String rawID = request.getParameter(AbstractServlet.CONST(CONSUMER_KEY));
        if (StringUtils.isTrivial(rawID)) {
            return null;
        }
        return BasicIdentifier.newID(rawID);
    }

    /**
     * Finds the client from the  §2.1 JSON {@link RFC7523Constants#CLIENT_ASSERTION}-- admin or regular -- and verifies that it is valid, has been approved etc.
     * @param request
     * @param oa2SE
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeySpecException
     */
    public static BaseClient findRFC7523Client(HttpServletRequest request, OA2SE oa2SE, JSONObject json) throws NoSuchAlgorithmException, InvalidKeySpecException {
        String state = json.containsKey(OA2Constants.STATE) ? json.getString(OA2Constants.STATE) : null;
        if (!json.containsKey(SUBJECT)) {
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, "missing " + SUBJECT + " claim, i.e., no client ID", HttpStatus.SC_BAD_REQUEST, state);
        }
        Identifier clientID = BasicIdentifier.newID(json.getString(SUBJECT));
        BaseClient client;
        if (oa2SE.getClientStore().containsKey(clientID)) {
            client = (OA2Client) oa2SE.getClientStore().get(clientID);
        } else {
            if (oa2SE.getAdminClientStore().containsKey(clientID)) {
                client = oa2SE.getAdminClientStore().get(clientID);
            } else {
                throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, RFC7523Constants.CLIENT_ASSERTION + " is not a valid client", HttpStatus.SC_BAD_REQUEST, null);
            }
        }
        if (!oa2SE.getClientApprovalStore().isApproved(clientID)) {
            throw new OA2GeneralError(OA2Errors.UNAUTHORIZED_CLIENT, "client not approved", HttpStatus.SC_BAD_REQUEST, state);
        }
        if (!client.hasJWKS()) {
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, "client does not support RFC 7523", HttpStatus.SC_BAD_REQUEST, state);
        }

        return client;
    }

    /**
     * Assumption is that the request has the correct {@link RFC7523Constants#CLIENT_ASSERTION_TYPE} of
     * {@link RFC7523Constants#ASSERTION_JWT_BEARER}, so we are decoding that.
     *
     * @param request
     */
    public static BaseClient getAndVerifyRFC7523Client(HttpServletRequest request, OA2SE oa2SE) throws NoSuchAlgorithmException, InvalidKeySpecException {
        return getAndVerifyRFC7523Client(request, oa2SE, false); // default is to use token endpoint
    }

    /**
     * Just carries out verifying RFC 7523 §2.1. It returns the authorizing client
     * @param request
     * @param oa2SE
     * @param isDeviceFlow
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeySpecException
     */
    public static BaseClient getAndVerifyRFC7523Client(HttpServletRequest request, OA2SE oa2SE, boolean isDeviceFlow) throws NoSuchAlgorithmException, InvalidKeySpecException {
        String assertionType= request.getParameter(RFC7523Constants.CLIENT_ASSERTION_TYPE);
        if(!assertionType.equals(RFC7523Constants.ASSERTION_JWT_BEARER)){
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST,
                    "unsupported " + RFC7523Constants.CLIENT_ASSERTION_TYPE,
                    HttpStatus.SC_BAD_REQUEST, null);

        }
        String raw = request.getParameter(RFC7523Constants.CLIENT_ASSERTION);
        if (StringUtils.isTrivial(raw)) {
            // throw an exception
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST,
                    "missing " + RFC7523Constants.CLIENT_ASSERTION,
                    HttpStatus.SC_BAD_REQUEST, null);
        }
        JSONObject[] hp;
        try {
            hp = MyOtherJWTUtil2.readJWT(raw);
        } catch (IllegalArgumentException iax) {
            // means this is sent as a JWT, but is not one
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, RFC7523Constants.CLIENT_ASSERTION + " is not a JWT", HttpStatus.SC_BAD_REQUEST, null);
        } catch (Throwable t) {
            // In this case, it is something like an unsupported algorithm
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, "could not decode JWT:" + t.getMessage(), HttpStatus.SC_BAD_REQUEST, null);
        }
        // In order to decode this, we need to get the client ID (required in the sub claim) and grab the key.
        JSONObject json = hp[1];
        BaseClient client = findRFC7523Client(request, oa2SE, json);
        String state = json.containsKey(OA2Constants.STATE) ? json.getString(OA2Constants.STATE) : null;

        try {
            MyOtherJWTUtil2.verifyAndReadJWT(raw, client.getJWKS());
        } catch (Throwable t) {
            // We read the token before without verifying it because we could not. The only error(s) left are if the signature fails.
            throw new OA2GeneralError(OA2Errors.INVALID_TOKEN, "failed to verify token", HttpStatus.SC_BAD_REQUEST, state);
        }

        if (json.containsKey(AUDIENCE)) {
            String serverName = oa2SE.getServiceAddress().toString();
            if (isDeviceFlow) {
                serverName = serverName + (serverName.endsWith("/") ? "" : "/") + RFC8628Constants.DEVICE_AUTHORIZATION_ENDPOINT; // construct the device_authorization endpoint
            } else {
                serverName = serverName + (serverName.endsWith("/") ? "" : "/") + "token"; // construct the token endpoint
            }
            if (!json.getString(AUDIENCE).equals(serverName)) {
                throw new IllegalArgumentException("wrong " + AUDIENCE); // as per spec.
            }
        } else {
            throw new IllegalArgumentException("missing " + AUDIENCE);
        }
        // Not clear what the issuer should be, aside from the OIDC spec., so we accept that as
        // reasonable and assume it is just the client
        if (json.containsKey(ISSUER)) {
            Identifier id = BasicIdentifier.newID(json.getString(ISSUER));
            if (!client.getIdentifier().equals(id)) {
                throw new UnknownClientException("unknown " + ISSUER + " with id \"" + id + "\"");
            }

        } else {
            throw new IllegalArgumentException("missing " + ISSUER);
        }
        if (json.containsKey(EXPIRATION)) {
            if (json.getLong(EXPIRATION) * 1000 < System.currentTimeMillis()) {
                throw new IllegalArgumentException("Expired token ");
            }
        } else {
            throw new IllegalArgumentException("missing " + EXPIRATION);
        }
        // issued at does not concern us at this time. Might limit it by
        // policy in the future.
        if (json.containsKey(NOT_VALID_BEFORE)) {
            if (System.currentTimeMillis() < json.getLong(NOT_VALID_BEFORE) * 1000) {
                throw new IllegalArgumentException("Token is not valid yet");
            }
        }
        return client;
    }

    public static OA2Client getRFC7523Client(HttpServletRequest request, OA2SE oa2SE) throws NoSuchAlgorithmException, InvalidKeySpecException {
        String raw = request.getParameter(RFC7523Constants.CLIENT_ASSERTION);
        if (StringUtils.isTrivial(raw)) {
            // throw an exception
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST,
                    "missing " + RFC7523Constants.CLIENT_ASSERTION,
                    HttpStatus.SC_BAD_REQUEST, null);

        }
        JSONObject[] hp;
        try {
            hp = MyOtherJWTUtil2.readJWT(raw);
        } catch (IllegalArgumentException iax) {
            // means this is sent as a JWT, but is not one
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, RFC7523Constants.CLIENT_ASSERTION + " is not a JWT", HttpStatus.SC_BAD_REQUEST, null);
        } catch (Throwable t) {
            // In this case, it is something like an unsupported algorithm
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, "could not decode JWT:" + t.getMessage(), HttpStatus.SC_BAD_REQUEST, null);
        }
        // In order to decode this, we need to get the client ID (required in the sub claim) and grab the key.
        JSONObject json = hp[1];
        String state = json.containsKey(OA2Constants.STATE) ? json.getString(OA2Constants.STATE) : null;
        if (!json.containsKey(SUBJECT)) {
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, "missing " + SUBJECT + " claim, i.e., no client ID", HttpStatus.SC_BAD_REQUEST, state);
        }
        Identifier clientID = BasicIdentifier.newID(json.getString(SUBJECT));
        OA2Client client = (OA2Client) oa2SE.getClientStore().get(clientID);
        if (!oa2SE.getClientApprovalStore().isApproved(clientID)) {
            throw new OA2GeneralError(OA2Errors.UNAUTHORIZED_CLIENT, "client not approved", HttpStatus.SC_BAD_REQUEST, state);
        }
        return client;
    }

    public static void verifyRFC7523Client(OA2Client client, HttpServletRequest request, OA2SE oa2SE) throws NoSuchAlgorithmException, InvalidKeySpecException {
        String raw = request.getParameter(RFC7523Constants.CLIENT_ASSERTION);
        if (StringUtils.isTrivial(raw)) {
            // throw an exception
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST,
                    "missing " + RFC7523Constants.CLIENT_ASSERTION,
                    HttpStatus.SC_BAD_REQUEST, null);

        }
        JSONObject[] hp;
        try {
            hp = MyOtherJWTUtil2.readJWT(raw);
        } catch (IllegalArgumentException iax) {
            // means this is sent as a JWT, but is not one
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, RFC7523Constants.CLIENT_ASSERTION + " is not a JWT", HttpStatus.SC_BAD_REQUEST, null);
        } catch (Throwable t) {
            // In this case, it is something like an unsupported algorithm
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, "could not decode JWT:" + t.getMessage(), HttpStatus.SC_BAD_REQUEST, null);
        }
        JSONObject json = hp[1];
        String state = json.containsKey(OA2Constants.STATE) ? json.getString(OA2Constants.STATE) : null;

        if (!client.hasJWKS()) {
            throw new OA2GeneralError(OA2Errors.INVALID_REQUEST, "client does not support RFC 7523", HttpStatus.SC_BAD_REQUEST, state);
        }
        // Finally. We can verify the JWT
        try {
            MyOtherJWTUtil2.verifyAndReadJWT(raw, client.getJWKS());
        } catch (Throwable t) {
            // We read the token before without verifying it because we could not. The only error(s) left are if the signature fails.
            throw new OA2GeneralError(OA2Errors.INVALID_TOKEN, "failed to verify token", HttpStatus.SC_BAD_REQUEST, state);
        }

        if (json.containsKey(AUDIENCE)) {
            String serverName = oa2SE.getServiceAddress().toString();
            serverName = serverName + (serverName.endsWith("/") ? "" : "/") + "token"; // construct the token endpoint
            if (!json.getString(AUDIENCE).equals(serverName)) {
                throw new IllegalArgumentException("wrong " + AUDIENCE);
            }
        } else {
            throw new IllegalArgumentException("missing " + AUDIENCE);
        }
        // Not clear what the issuer should be, aside from the OIDC spec., so we accept that as
        // reasonable and assume it is just the client
        if (json.containsKey(ISSUER)) {
            Identifier id = BasicIdentifier.newID(json.getString(ISSUER));
            if (!client.getIdentifier().equals(id)) {
                throw new UnknownClientException("unknown " + ISSUER + " with id \"" + id + "\"");
            }

        } else {
            throw new IllegalArgumentException("missing " + ISSUER);
        }
        if (json.containsKey(EXPIRATION)) {
            if (json.getLong(EXPIRATION) * 1000 < System.currentTimeMillis()) {
                throw new IllegalArgumentException("Expired token ");
            }
        } else {
            throw new IllegalArgumentException("missing " + EXPIRATION);
        }
        // issued at does not concern us at this time. Might limit it by
        // policy in the future.
        if (json.containsKey(NOT_VALID_BEFORE)) {
            if (System.currentTimeMillis() < json.getLong(NOT_VALID_BEFORE) * 1000) {
                throw new IllegalArgumentException("Token is not valid yet");
            }
        }
    }


}