/* SPDX-License-Identifier: Apache-2.0 */
/* Copyright Contributors to the ODPi Egeria project. */
package org.odpi.openmetadata.accessservices.assetmanager.client.converters;

import org.odpi.openmetadata.accessservices.assetmanager.metadataelements.ValidValueElement;
import org.odpi.openmetadata.frameworks.connectors.ffdc.PropertyServerException;
import org.odpi.openmetadata.frameworks.governanceaction.properties.OpenMetadataElement;
import org.odpi.openmetadata.frameworks.governanceaction.search.ElementProperties;
import org.odpi.openmetadata.frameworks.governanceaction.search.PropertyHelper;
import org.odpi.openmetadata.frameworks.openmetadata.properties.validvalues.ValidValueProperties;

import java.lang.reflect.InvocationTargetException;


/**
 * ValidValueConverter generates a ValidValueElement from a ValidValue entity
 */
public class ValidValueConverter<B> extends AssetManagerConverterBase<B>
{
    /**
     * Constructor
     *
     * @param propertyHelper helper object to parse entity
     * @param serviceName name of this component
     * @param serverName local server name
     */
    public ValidValueConverter(PropertyHelper propertyHelper,
                               String         serviceName,
                               String         serverName)
    {
        super(propertyHelper, serviceName, serverName);
    }


    /**
     * Using the supplied openMetadataElement, return a new instance of the bean. This is used for most beans that have
     * a one to one correspondence with the repository instances.
     *
     * @param beanClass name of the class to create
     * @param openMetadataElement openMetadataElement containing the properties
     * @param methodName calling method
     * @return bean populated with properties from the instances supplied
     * @throws PropertyServerException there is a problem instantiating the bean
     */
    @Override
    public B getNewBean(Class<B>            beanClass,
                        OpenMetadataElement openMetadataElement,
                        String              methodName) throws PropertyServerException
    {
        try
        {
            /*
             * This is initial confirmation that the generic converter has been initialized with an appropriate bean class.
             */
            B returnBean = beanClass.getDeclaredConstructor().newInstance();

            if (returnBean instanceof ValidValueElement bean)
            {
                ValidValueProperties validValueProperties = new ValidValueProperties();

                bean.setElementHeader(super.getMetadataElementHeader(beanClass, openMetadataElement, methodName));

                ElementProperties elementProperties;

                /*
                 * The initial set of values come from the openMetadataElement.
                 */
                if (openMetadataElement != null)
                {
                    elementProperties = new ElementProperties(openMetadataElement.getElementProperties());

                    validValueProperties.setQualifiedName(this.removeQualifiedName(elementProperties));
                    validValueProperties.setAdditionalProperties(this.removeAdditionalProperties(elementProperties));
                    validValueProperties.setDisplayName(this.removeName(elementProperties));
                    validValueProperties.setDescription(this.removeDescription(elementProperties));
                    validValueProperties.setUsage(this.removeUsage(elementProperties));
                    validValueProperties.setCategory(this.removeCategory(elementProperties));
                    validValueProperties.setScope(this.removeScope(elementProperties));
                    validValueProperties.setIsDeprecated(this.removeIsDeprecated(elementProperties));
                    validValueProperties.setIsDeprecated(this.removeIsCaseSensitive(elementProperties));
                    validValueProperties.setPreferredValue(this.removePreferredValue(elementProperties));
                    validValueProperties.setDataType(this.removeDataType(elementProperties));
                    validValueProperties.setEffectiveFrom(openMetadataElement.getEffectiveFromTime());
                    validValueProperties.setEffectiveTo(openMetadataElement.getEffectiveToTime());

                    /*
                     * Any remaining properties are returned in the extended properties.  They are
                     * assumed to be defined in a subtype.
                     */
                    validValueProperties.setTypeName(bean.getElementHeader().getType().getTypeName());
                    validValueProperties.setExtendedProperties(this.getRemainingExtendedProperties(elementProperties));
                }
                else
                {
                    handleMissingMetadataInstance(beanClass.getName(), OpenMetadataElement.class.getName(), methodName);
                }

                bean.setValidValueProperties(validValueProperties);
            }

            return returnBean;
        }
        catch (IllegalAccessException | InstantiationException | ClassCastException | NoSuchMethodException | InvocationTargetException error)
        {
            super.handleInvalidBeanClass(beanClass.getName(), error, methodName);
        }

        return null;
    }
}
