/* SPDX-License-Identifier: Apache 2.0 */
/* Copyright Contributors to the ODPi Egeria project. */
package org.odpi.openmetadata.commonservices.generichandlers;

import org.odpi.openmetadata.frameworks.connectors.ffdc.InvalidParameterException;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.properties.instances.InstanceProperties;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.repositoryconnector.OMRSRepositoryHelper;

import java.util.Map;

/**
 * ConnectionBuilder is able to build the properties for both a Connection entity and a VirtualConnection entity.
 */
public class ConnectionBuilder extends ReferenceableBuilder
{
    private String              displayName;
    private String              description;
    private Map<String, String> securedProperties;
    private Map<String, Object> configurationProperties;
    private String              connectorUserId;
    private String              clearPassword;
    private String              encryptedPassword;


    /**
     * Constructor supporting all properties.
     *
     * @param qualifiedName unique name
     * @param displayName new value for the display name.
     * @param description new description for the connection.
     * @param additionalProperties additional properties
     * @param securedProperties protected properties
     * @param configurationProperties  properties passed to configure underlying technologies
     * @param connectorUserId user identity that the connector should use
     * @param clearPassword password for the userId in clear text
     * @param encryptedPassword encrypted password that the connector needs to decrypt before use
     * @param typeId identifier of the type for this connection
     * @param typeName name of the type for this connection
     * @param extendedProperties  properties from the subtype.
     * @param repositoryHelper helper methods
     * @param serviceName name of this OMAS
     * @param serverName name of local server
     */
    ConnectionBuilder(String                   qualifiedName,
                      String                   displayName,
                      String                   description,
                      Map<String, String>      additionalProperties,
                      Map<String, String>      securedProperties,
                      Map<String, Object>      configurationProperties,
                      String                   connectorUserId,
                      String                   clearPassword,
                      String                   encryptedPassword,
                      String                   typeId,
                      String                   typeName,
                      Map<String, Object>      extendedProperties,
                      OMRSRepositoryHelper     repositoryHelper,
                      String                   serviceName,
                      String                   serverName)
    {
        super(qualifiedName,
              additionalProperties,
              typeId,
              typeName,
              extendedProperties,
              repositoryHelper,
              serviceName,
              serverName);

        this.displayName = displayName;
        this.description = description;
        this.securedProperties = securedProperties;
        this.configurationProperties = configurationProperties;
        this.connectorUserId = connectorUserId;
        this.clearPassword = clearPassword;
        this.encryptedPassword = encryptedPassword;
    }


    /**
     * Return the supplied bean properties in an InstanceProperties object.
     *
     * @param methodName name of the calling method
     * @return InstanceProperties object
     * @throws InvalidParameterException there is a problem with the properties
     */
    @Override
    public InstanceProperties getInstanceProperties(String  methodName) throws InvalidParameterException
    {
        InstanceProperties properties = super.getInstanceProperties(methodName);

        if (displayName != null)
        {
            properties = repositoryHelper.addStringPropertyToInstance(serviceName,
                                                                      properties,
                                                                      OpenMetadataAPIMapper.DISPLAY_NAME_PROPERTY_NAME,
                                                                      displayName,
                                                                      methodName);
        }

        if (description != null)
        {
            properties = repositoryHelper.addStringPropertyToInstance(serviceName,
                                                                      properties,
                                                                      OpenMetadataAPIMapper.DESCRIPTION_PROPERTY_NAME,
                                                                      description,
                                                                      methodName);
        }

        if (securedProperties != null)
        {
            properties = repositoryHelper.addStringMapPropertyToInstance(serviceName,
                                                                         properties,
                                                                         OpenMetadataAPIMapper.SECURED_PROPERTIES_PROPERTY_NAME,
                                                                         securedProperties,
                                                                         methodName);
        }

        if (configurationProperties != null)
        {
            properties = repositoryHelper.addMapPropertyToInstance(serviceName,
                                                                   properties,
                                                                   OpenMetadataAPIMapper.CONFIGURATION_PROPERTIES_PROPERTY_NAME,
                                                                   configurationProperties,
                                                                   methodName);
        }

        if (connectorUserId != null)
        {
            properties = repositoryHelper.addStringPropertyToInstance(serviceName,
                                                                      properties,
                                                                      OpenMetadataAPIMapper.USER_ID_PROPERTY_NAME,
                                                                      connectorUserId,
                                                                      methodName);
        }

        if (clearPassword != null)
        {
            properties = repositoryHelper.addStringPropertyToInstance(serviceName,
                                                                      properties,
                                                                      OpenMetadataAPIMapper.CLEAR_PASSWORD_PROPERTY_NAME,
                                                                      clearPassword,
                                                                      methodName);
        }

        if (encryptedPassword != null)
        {
            properties = repositoryHelper.addStringPropertyToInstance(serviceName,
                                                                      properties,
                                                                      OpenMetadataAPIMapper.ENCRYPTED_PASSWORD_PROPERTY_NAME,
                                                                      encryptedPassword,
                                                                      methodName);
        }

        return properties;
    }
}
