/* SPDX-License-Identifier: Apache 2.0 */
/* Copyright Contributors to the ODPi Egeria project. */
package org.odpi.openmetadata.commonservices.generichandlers;

import org.odpi.openmetadata.commonservices.ffdc.InvalidParameterHandler;
import org.odpi.openmetadata.commonservices.repositoryhandler.RepositoryHandler;
import org.odpi.openmetadata.frameworks.auditlog.AuditLog;
import org.odpi.openmetadata.frameworks.connectors.ffdc.InvalidParameterException;
import org.odpi.openmetadata.frameworks.connectors.ffdc.PropertyServerException;
import org.odpi.openmetadata.frameworks.connectors.ffdc.UserNotAuthorizedException;
import org.odpi.openmetadata.metadatasecurity.server.OpenMetadataServerSecurityVerifier;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.properties.instances.Classification;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.properties.instances.EntityDetail;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.properties.instances.InstanceProperties;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.properties.instances.Relationship;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.repositoryconnector.OMRSRepositoryHelper;
import org.odpi.openmetadata.repositoryservices.ffdc.exception.ClassificationErrorException;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * SchemaAttributeHandler manages Schema Attribute objects.  It runs server-side in
 * the OMAG Server Platform and retrieves SchemaElement entities through the OMRSRepositoryConnector.
 * This handler does not support effectivity dates but probably should.
 */
public class SchemaAttributeHandler<SCHEMA_ATTRIBUTE, SCHEMA_TYPE> extends SchemaElementHandler<SCHEMA_ATTRIBUTE>
{
    private OpenMetadataAPIGenericConverter<SCHEMA_ATTRIBUTE> schemaAttributeConverter;
    private Class<SCHEMA_TYPE>                                schemaTypeBeanClass;
    private SchemaTypeHandler<SCHEMA_TYPE>                    schemaTypeHandler;


    /**
     * Construct the handler with information needed to work with B objects.
     *
     * @param schemaAttributeConverter specific converter for the SCHEMA_ATTRIBUTE bean class
     * @param schemaAttributeBeanClass name of bean class that is represented by the generic class SCHEMA_ATTRIBUTE
     * @param schemaTypeConverter specific converter for the SCHEMA_TYPE bean class
     * @param schemaTypeBeanClass name of bean class that is represented by the generic class SCHEMA_TYPE
     * @param serviceName      name of this service
     * @param serverName       name of the local server
     * @param invalidParameterHandler handler for managing parameter errors
     * @param repositoryHandler     manages calls to the repository services
     * @param repositoryHelper provides utilities for manipulating the repository services objects
     * @param localServerUserId userId for this server
     * @param securityVerifier open metadata security services verifier
     * @param supportedZones list of zones that the access service is allowed to serve B instances from.
     * @param defaultZones list of zones that the access service should set in all new B instances.
     * @param publishZones list of zones that the access service sets up in published B instances.
     * @param auditLog destination for audit log events.
     */
    public SchemaAttributeHandler(OpenMetadataAPIGenericConverter<SCHEMA_ATTRIBUTE> schemaAttributeConverter,
                                  Class<SCHEMA_ATTRIBUTE>                                 schemaAttributeBeanClass,
                                  OpenMetadataAPIGenericConverter<SCHEMA_TYPE> schemaTypeConverter,
                                  Class<SCHEMA_TYPE>                                      schemaTypeBeanClass,
                                  String                                                  serviceName,
                                  String                                                  serverName,
                                  InvalidParameterHandler invalidParameterHandler,
                                  RepositoryHandler repositoryHandler,
                                  OMRSRepositoryHelper repositoryHelper,
                                  String                                                  localServerUserId,
                                  OpenMetadataServerSecurityVerifier securityVerifier,
                                  List<String>                                            supportedZones,
                                  List<String>                                            defaultZones,
                                  List<String>                                            publishZones,
                                  AuditLog auditLog)
    {
        super(schemaAttributeConverter,
              schemaAttributeBeanClass,
              serviceName,
              serverName,
              invalidParameterHandler,
              repositoryHandler,
              repositoryHelper,
              localServerUserId,
              securityVerifier,
              supportedZones,
              defaultZones,
              publishZones,
              auditLog);

        /*
         * Schema Attributes need their own specialized converter because the schema type may be represented by multiple entities.
         */
        this.schemaAttributeConverter = schemaAttributeConverter;
        this.schemaTypeBeanClass      = schemaTypeBeanClass;
        this.schemaTypeHandler        = new SchemaTypeHandler<>(schemaTypeConverter,
                                                                schemaTypeBeanClass,
                                                                serviceName,
                                                                serverName,
                                                                invalidParameterHandler,
                                                                repositoryHandler,
                                                                repositoryHelper,
                                                                localServerUserId,
                                                                securityVerifier,
                                                                supportedZones,
                                                                defaultZones,
                                                                publishZones,
                                                                auditLog);
    }


    /**
     * Create a new metadata element to represent a schema attribute using an existing metadata element as a template.
     * The template defines additional classifications and relationships that should be added to the new schema attribute.
     *
     * @param userId calling user
     * @param externalSourceGUID     unique identifier of software server capability representing the caller
     * @param externalSourceName     unique name of software server capability representing the caller
     * @param parentElementGUID  element to connect this schema attribute to
     * @param parentElementGUIDParameterName parameter supplying parentElementGUID
     * @param templateGUID unique identifier of the metadata element to copy
     * @param qualifiedName unique name for the schema attribute - used in other configuration
     * @param displayName short display name for the schema attribute
     * @param description description of the schema attribute
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName calling method
     *
     * @return unique identifier of the new metadata element
     *
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public String createSchemaAttributeFromTemplate(String userId,
                                                    String externalSourceGUID,
                                                    String externalSourceName,
                                                    String parentElementGUID,
                                                    String parentElementGUIDParameterName,
                                                    String templateGUID,
                                                    String qualifiedName,
                                                    String displayName,
                                                    String description,
                                                    Date   effectiveTime,
                                                    String methodName) throws InvalidParameterException,
                                                                              UserNotAuthorizedException,
                                                                              PropertyServerException
    {
        final String templateGUIDParameterName   = "templateGUID";
        final String qualifiedNameParameterName  = "qualifiedName";

        invalidParameterHandler.validateUserId(userId, methodName);
        invalidParameterHandler.validateGUID(parentElementGUID, parentElementGUIDParameterName, methodName);
        invalidParameterHandler.validateGUID(templateGUID, templateGUIDParameterName, methodName);
        invalidParameterHandler.validateName(qualifiedName, qualifiedNameParameterName, methodName);

        SchemaAttributeBuilder builder = new SchemaAttributeBuilder(qualifiedName,
                                                                    displayName,
                                                                    description,
                                                                    repositoryHelper,
                                                                    serviceName,
                                                                    serverName);

        String schemaAttributeGUID = this.createBeanFromTemplate(userId,
                                                                 externalSourceGUID,
                                                                 externalSourceName,
                                                                 templateGUID,
                                                                 templateGUIDParameterName,
                                                                 OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_GUID,
                                                                 OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                                 qualifiedName,
                                                                 OpenMetadataAPIMapper.QUALIFIED_NAME_PROPERTY_NAME,
                                                                 builder,
                                                                 methodName);

        if (schemaAttributeGUID != null)
        {
            final String schemaAttributeGUIDParameterName = "schemaAttributeGUID";

            EntityDetail parentEntity = this.getEntityFromRepository(userId,
                                                                     parentElementGUID,
                                                                     parentElementGUIDParameterName,
                                                                     OpenMetadataAPIMapper.SCHEMA_ELEMENT_TYPE_NAME,
                                                                     null,
                                                                     null,
                                                                     false,
                                                                     false,
                                                                     supportedZones,
                                                                     effectiveTime,
                                                                     methodName);

            String parentElementTypeName = OpenMetadataAPIMapper.SCHEMA_TYPE_TYPE_NAME;
            String parentElementRelationshipTypeGUID = OpenMetadataAPIMapper.TYPE_TO_ATTRIBUTE_RELATIONSHIP_TYPE_GUID;
            String parentElementRelationshipTypeName = OpenMetadataAPIMapper.TYPE_TO_ATTRIBUTE_RELATIONSHIP_TYPE_NAME;

            if ((parentEntity != null) && (parentEntity.getType() != null))
            {
                if (repositoryHelper.isTypeOf(serviceName, parentEntity.getType().getTypeDefName(), OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME))
                {
                    parentElementTypeName = OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME;
                    parentElementRelationshipTypeGUID = OpenMetadataAPIMapper.NESTED_ATTRIBUTE_RELATIONSHIP_TYPE_GUID;
                    parentElementRelationshipTypeName = OpenMetadataAPIMapper.NESTED_ATTRIBUTE_RELATIONSHIP_TYPE_NAME;
                }
            }

            this.linkElementToElement(userId,
                                      externalSourceGUID,
                                      externalSourceName,
                                      parentElementGUID,
                                      parentElementGUIDParameterName,
                                      parentElementTypeName,
                                      schemaAttributeGUID,
                                      schemaAttributeGUIDParameterName,
                                      builder.getTypeName(),
                                      false,
                                      false,
                                      supportedZones,
                                      parentElementRelationshipTypeGUID,
                                      parentElementRelationshipTypeName,
                                      null,
                                      methodName);
        }

        return schemaAttributeGUID;
    }


    /**
     * Create a new schema attribute with its type attached.
     *
     * @param userId calling user
     * @param externalSourceGUID unique identifier of software server capability representing the caller
     * @param externalSourceName unique name of software server capability representing the caller
     * @param parentElementGUID unique identifier of the metadata element to connect the new schema attribute to
     * @param parentElementGUIDParameterName parameter name supplying parentElementGUID
     * @param qualifiedName unique identifier for this schema type
     * @param qualifiedNameParameterName name of parameter supplying the qualified name
     * @param schemaAttributeBuilder builder containing the properties of the schema type
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName calling method
     *
     * @return unique identifier of the new schema attribute already linked to its parent
     *
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public String createNestedSchemaAttribute(String                 userId,
                                              String                 externalSourceGUID,
                                              String                 externalSourceName,
                                              String                 parentElementGUID,
                                              String                 parentElementGUIDParameterName,
                                              String                 qualifiedName,
                                              String                 qualifiedNameParameterName,
                                              SchemaAttributeBuilder schemaAttributeBuilder,
                                              Date                   effectiveTime,
                                              String                 methodName) throws InvalidParameterException,
                                                                                        UserNotAuthorizedException,
                                                                                        PropertyServerException
    {
        invalidParameterHandler.validateUserId(userId, methodName);
        invalidParameterHandler.validateGUID(parentElementGUID, parentElementGUIDParameterName, methodName);
        invalidParameterHandler.validateName(qualifiedName, qualifiedNameParameterName, methodName);

        EntityDetail parentEntity = this.getEntityFromRepository(userId,
                                                                 parentElementGUID,
                                                                 parentElementGUIDParameterName,
                                                                 OpenMetadataAPIMapper.SCHEMA_ELEMENT_TYPE_NAME,
                                                                 null,
                                                                 null,
                                                                 false,
                                                                 false,
                                                                 supportedZones,
                                                                 effectiveTime,
                                                                 methodName);

        String parentAttributeTypeName             = OpenMetadataAPIMapper.COMPLEX_SCHEMA_TYPE_TYPE_NAME;
        String parentAttributeRelationshipTypeGUID = OpenMetadataAPIMapper.TYPE_TO_ATTRIBUTE_RELATIONSHIP_TYPE_GUID;
        String parentAttributeRelationshipTypeName = OpenMetadataAPIMapper.TYPE_TO_ATTRIBUTE_RELATIONSHIP_TYPE_NAME;

        if ((parentEntity != null) && (parentEntity.getType() != null))
        {
            if (repositoryHelper.isTypeOf(serviceName, parentEntity.getType().getTypeDefName(), OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME))
            {
                parentAttributeTypeName             = OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME;
                parentAttributeRelationshipTypeGUID = OpenMetadataAPIMapper.NESTED_ATTRIBUTE_RELATIONSHIP_TYPE_GUID;
                parentAttributeRelationshipTypeName = OpenMetadataAPIMapper.NESTED_ATTRIBUTE_RELATIONSHIP_TYPE_NAME;
            }

            /*
             * If the parent is set up with an anchor then this is propagated to the schema attribute
             */
            String anchorGUID = this.getAnchorGUIDFromAnchorsClassification(parentEntity, methodName);
            if (anchorGUID != null)
            {
                schemaAttributeBuilder.setAnchors(userId, anchorGUID, methodName);
            }

            return this.createNestedSchemaAttribute(userId,
                                                    externalSourceGUID,
                                                    externalSourceName,
                                                    parentElementGUID,
                                                    parentElementGUIDParameterName,
                                                    parentAttributeTypeName,
                                                    parentAttributeRelationshipTypeGUID,
                                                    parentAttributeRelationshipTypeName,
                                                    qualifiedName,
                                                    qualifiedNameParameterName,
                                                    schemaAttributeBuilder,
                                                    methodName);

        }
        else
        {
            /*
             * Either the entity has not been returned or it has no type info in it
             */
            invalidParameterHandler.throwUnknownElement(userId,
                                                        parentElementGUID,
                                                        OpenMetadataAPIMapper.SCHEMA_ELEMENT_TYPE_NAME,
                                                        serviceName,
                                                        serverName,
                                                        methodName);
        }

        return null;
    }


    /**
     * Create a new schema attribute with its type attached that is nested inside a complex schema type or a schema attribute.
     *
     * @param userId calling user
     * @param externalSourceGUID unique identifier of software server capability representing the caller
     * @param externalSourceName unique name of software server capability representing the caller
     * @param parentElementGUID unique identifier of the metadata element to connect the new schema attribute to
     * @param parentElementGUIDParameterName parameter name supplying parentElementGUID
     * @param qualifiedName unique identifier for this schema type
     * @param qualifiedNameParameterName name of parameter supplying the qualified name
     * @param displayName the stored display name property for the attribute
     * @param description the stored description property associated with the attribute
     * @param externalSchemaTypeGUID unique identifier of a schema Type that provides the type. If null, a private schema type is used
     * @param dataType data type name - for stored values
     * @param defaultValue string containing default value - for stored values
     * @param fixedValue string containing a fixed value - for a literal
     * @param validValuesSetGUID unique identifier of a valid value set that lists the valid values for this schema
     * @param formula String formula - for derived values
     * @param isDeprecated is this table deprecated?
     * @param elementPosition the position of this attribute in its parent type.
     * @param minCardinality minimum number of repeating instances allowed for this attribute - typically 1
     * @param maxCardinality the maximum number of repeating instances allowed for this column - typically 1
     * @param allowsDuplicateValues  whether the same value can be used by more than one instance of this attribute
     * @param orderedValues whether the attribute instances are arranged in an order
     * @param sortOrder the order that the attribute instances are arranged in - if any
     * @param minimumLength the minimum length of the data
     * @param length the length of the data field
     * @param significantDigits number of significant digits to the right of decimal point
     * @param isNullable whether the field is nullable or not
     * @param nativeJavaClass equivalent Java class implementation
     * @param defaultValueOverride default value for this column
     * @param aliases a list of alternative names for the attribute
     * @param additionalProperties any arbitrary properties not part of the type system
     * @param typeName name of the type that is a subtype of DeployedDatabaseSchema - or null to create standard type
     * @param extendedProperties properties from any subtype
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName calling method
     *
     * @return unique identifier of the new schema attribute already linked to its parent
     *
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public String createNestedSchemaAttribute(String               userId,
                                              String               externalSourceGUID,
                                              String               externalSourceName,
                                              String               parentElementGUID,
                                              String               parentElementGUIDParameterName,
                                              String               qualifiedName,
                                              String               qualifiedNameParameterName,
                                              String               displayName,
                                              String               description,
                                              String               externalSchemaTypeGUID,
                                              String               dataType,
                                              String               defaultValue,
                                              String               fixedValue,
                                              String               validValuesSetGUID,
                                              String               formula,
                                              boolean              isDeprecated,
                                              int                  elementPosition,
                                              int                  minCardinality,
                                              int                  maxCardinality,
                                              boolean              allowsDuplicateValues,
                                              boolean              orderedValues,
                                              String               defaultValueOverride,
                                              int                  sortOrder,
                                              int                  minimumLength,
                                              int                  length,
                                              int                  significantDigits,
                                              boolean              isNullable,
                                              String               nativeJavaClass,
                                              List<String>         aliases,
                                              Map<String, String>  additionalProperties,
                                              String               typeName,
                                              Map<String, Object>  extendedProperties,
                                              Date                 effectiveTime,
                                              String               methodName) throws InvalidParameterException,
                                                                                      UserNotAuthorizedException,
                                                                                      PropertyServerException
    {
        invalidParameterHandler.validateUserId(userId, methodName);
        invalidParameterHandler.validateGUID(parentElementGUID, parentElementGUIDParameterName, methodName);
        invalidParameterHandler.validateName(qualifiedName, qualifiedNameParameterName, methodName);

        /*
         * Check that the type name requested for the schema attribute is valid.
         */
        String schemaAttributeTypeName = OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME;
        String schemaAttributeTypeId   = OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_GUID;

        if (typeName != null)
        {
            schemaAttributeTypeName = typeName;
            schemaAttributeTypeId = invalidParameterHandler.validateTypeName(typeName,
                                                                             OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                                             serviceName,
                                                                             methodName,
                                                                             repositoryHelper);
        }

        /*
         * Load up the builder objects.  The builders manage the properties of the metadata elements that make up the schema attribute,
         * and the schemaTypeHandler manages the type.
         */
        SchemaAttributeBuilder schemaAttributeBuilder = new SchemaAttributeBuilder(qualifiedName,
                                                                                   displayName,
                                                                                   description,
                                                                                   elementPosition,
                                                                                   minCardinality,
                                                                                   maxCardinality,
                                                                                   isDeprecated,
                                                                                   defaultValueOverride,
                                                                                   allowsDuplicateValues,
                                                                                   orderedValues,
                                                                                   sortOrder,
                                                                                   minimumLength,
                                                                                   length,
                                                                                   significantDigits,
                                                                                   isNullable,
                                                                                   nativeJavaClass,
                                                                                   aliases,
                                                                                   additionalProperties,
                                                                                   schemaAttributeTypeId,
                                                                                   schemaAttributeTypeName,
                                                                                   extendedProperties,
                                                                                   repositoryHelper,
                                                                                   serviceName,
                                                                                   serverName);

        /*
         * The formula is set if the attribute is derived
         */
        if (formula != null)
        {
            schemaAttributeBuilder.setCalculatedValue(userId, externalSourceGUID, externalSourceName, formula, methodName);
        }

        SchemaTypeBuilder schemaTypeBuilder = this.getSchemaTypeBuilder(qualifiedName,
                                                                        externalSchemaTypeGUID,
                                                                        dataType,
                                                                        defaultValue,
                                                                        fixedValue,
                                                                        validValuesSetGUID);

        schemaAttributeBuilder.setSchemaType(userId, schemaTypeBuilder, methodName);

        return this.createNestedSchemaAttribute(userId,
                                                externalSourceGUID,
                                                externalSourceName,
                                                parentElementGUID,
                                                parentElementGUIDParameterName,
                                                qualifiedName,
                                                qualifiedNameParameterName,
                                                schemaAttributeBuilder,
                                                effectiveTime,
                                                methodName);
    }


    /**
     * Set up the schema type builder for the column's type.
     *
     * @param qualifiedName qualified name for the column
     * @param externalSchemaTypeGUID unique identifier of a schema Type that provides the type. If null, a private schema type is used
     * @param dataType data type name - for stored values
     * @param defaultValue string containing default value - for stored values
     * @param fixedValue string containing a fixed value - for a literal
     * @param validValuesSetGUID unique identifier of a valid value set that lists the valid values for this schema
     * @return filled out schema type builder
     */
    SchemaTypeBuilder getSchemaTypeBuilder(String qualifiedName,
                                           String externalSchemaTypeGUID,
                                           String dataType,
                                           String defaultValue,
                                           String fixedValue,
                                           String validValuesSetGUID)
    {
        String schemaTypeGUID = OpenMetadataAPIMapper.STRUCT_SCHEMA_TYPE_TYPE_GUID;
        String schemaTypeName = OpenMetadataAPIMapper.STRUCT_SCHEMA_TYPE_TYPE_NAME;

        if (externalSchemaTypeGUID != null)
        {
            schemaTypeGUID = OpenMetadataAPIMapper.EXTERNAL_SCHEMA_TYPE_TYPE_GUID;
            schemaTypeName = OpenMetadataAPIMapper.EXTERNAL_SCHEMA_TYPE_TYPE_NAME;
        }
        else if (validValuesSetGUID != null)
        {
            schemaTypeGUID = OpenMetadataAPIMapper.ENUM_SCHEMA_TYPE_TYPE_GUID;
            schemaTypeName = OpenMetadataAPIMapper.ENUM_SCHEMA_TYPE_TYPE_NAME;
        }
        else if (fixedValue != null)
        {
            schemaTypeGUID = OpenMetadataAPIMapper.LITERAL_SCHEMA_TYPE_TYPE_GUID;
            schemaTypeName = OpenMetadataAPIMapper.LITERAL_SCHEMA_TYPE_TYPE_NAME;
        }
        else if (dataType != null)
        {
            schemaTypeGUID = OpenMetadataAPIMapper.PRIMITIVE_SCHEMA_TYPE_TYPE_GUID;
            schemaTypeName = OpenMetadataAPIMapper.PRIMITIVE_SCHEMA_TYPE_TYPE_NAME;
        }

        SchemaTypeBuilder schemaTypeBuilder = new SchemaTypeBuilder(qualifiedName + ":Type",
                                                                    schemaTypeGUID,
                                                                    schemaTypeName,
                                                                    repositoryHelper,
                                                                    serviceName,
                                                                    serverName);
        schemaTypeBuilder.setDataType(dataType);
        schemaTypeBuilder.setDefaultValue(defaultValue);
        schemaTypeBuilder.setFixedValue(fixedValue);
        schemaTypeBuilder.setExternalSchemaTypeGUID(externalSchemaTypeGUID);
        schemaTypeBuilder.setValidValuesSetGUID(validValuesSetGUID);

        return schemaTypeBuilder;
    }



    /**
     * Create a new schema attribute with its type attached.
     *
     * @param userId calling user
     * @param externalSourceGUID unique identifier of software server capability representing the caller
     * @param externalSourceName unique name of software server capability representing the caller
     * @param parentElementGUID unique identifier of the metadata element to connect the new schema attribute to
     * @param parentElementGUIDParameterName parameter name supplying parentElementGUID
     * @param parentElementTypeName type of the parent element - may be a schema attribute or a schema type
     * @param parentAttributeRelationshipTypeGUID unique identifier of the relationship from the new schema type to the parent
     * @param parentAttributeRelationshipTypeName unique name of the relationship from the new schema type to the parent
     * @param qualifiedName unique identifier for this schema type
     * @param qualifiedNameParameterName name of parameter supplying the qualified name
     * @param schemaAttributeBuilder builder containing the properties of the schema type
     * @param methodName calling method
     *
     * @return unique identifier of the new schema attribute already linked to its parent
     *
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public String createNestedSchemaAttribute(String                 userId,
                                              String                 externalSourceGUID,
                                              String                 externalSourceName,
                                              String                 parentElementGUID,
                                              String                 parentElementGUIDParameterName,
                                              String                 parentElementTypeName,
                                              String                 parentAttributeRelationshipTypeGUID,
                                              String                 parentAttributeRelationshipTypeName,
                                              String                 qualifiedName,
                                              String                 qualifiedNameParameterName,
                                              SchemaAttributeBuilder schemaAttributeBuilder,
                                              String                 methodName) throws InvalidParameterException,
                                                                                        UserNotAuthorizedException,
                                                                                        PropertyServerException
    {
        invalidParameterHandler.validateUserId(userId, methodName);
        invalidParameterHandler.validateGUID(parentElementGUID, parentElementGUIDParameterName, methodName);
        invalidParameterHandler.validateName(qualifiedName, qualifiedNameParameterName, methodName);

        /*
         * Now create the table itself along with its schema type.  It also links the resulting table to the database schema type.
         * The returned value is the guid of the table.
         */
        String schemaAttributeGUID = this.createBeanInRepository(userId,
                                                                 externalSourceGUID,
                                                                 externalSourceName,
                                                                 schemaAttributeBuilder.getTypeGUID(),
                                                                 schemaAttributeBuilder.getTypeName(),
                                                                 qualifiedName,
                                                                 OpenMetadataAPIMapper.QUALIFIED_NAME_PROPERTY_NAME,
                                                                 schemaAttributeBuilder,
                                                                 methodName);

        if (schemaAttributeGUID != null)
        {
            final String schemaAttributeGUIDParameterName = "schemaAttributeGUID";

            this.addEmbeddedTypes(userId,
                                  externalSourceGUID,
                                  externalSourceName,
                                  schemaAttributeGUID,
                                  schemaAttributeGUIDParameterName,
                                  schemaAttributeBuilder.getTypeName(),
                                  schemaAttributeBuilder.getSchemaTypeBuilder(),
                                  methodName);

            this.linkElementToElement(userId,
                                      externalSourceGUID,
                                      externalSourceName,
                                      parentElementGUID,
                                      parentElementGUIDParameterName,
                                      parentElementTypeName,
                                      schemaAttributeGUID,
                                      schemaAttributeGUIDParameterName,
                                      schemaAttributeBuilder.getTypeName(),
                                      false,
                                      false,
                                      supportedZones,
                                      parentAttributeRelationshipTypeGUID,
                                      parentAttributeRelationshipTypeName,
                                      null,
                                      methodName);


            return schemaAttributeGUID;
        }

        /*
         * Not reachable because any failures result in exceptions.
         */
        return null;
    }


    /**
     * Returns a list of schema attributes that are linked to a ComplexSchemaType via the AttributeForSchema relationship.
     * It validates the unique identifier of the complex schema type, and the visibility/security of any attached asset using the supported zones
     * for this component.  Then begins to extract the schema attributes.  Exceptions occur if a schema attribute does not have a type.
     *
     * @param userId         String   userId of user making request.
     * @param schemaTypeGUID String   unique id for containing schema type.
     * @param guidParameterName String name of the parameter supplying the guid.
     * @param requiredClassificationName  String the name of the classification that must be on the schema attribute or linked schema type entity.
     * @param omittedClassificationName   String the name of a classification that must not be on the schema attribute or linked schema type entity.
     * @param startFrom   int      starting position for first returned element.
     * @param pageSize    int      maximum number of elements to return on the call.
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName     calling method
     *
     * @return a schema attributes response
     * @throws InvalidParameterException - the GUID is not recognized or the paging values are invalid or
     * @throws PropertyServerException - there is a problem retrieving the asset properties from the property server or
     * @throws UserNotAuthorizedException - the requesting user is not authorized to issue this request.
     */
    public List<SCHEMA_ATTRIBUTE> getSchemaAttributesForComplexSchemaType(String userId,
                                                                          String schemaTypeGUID,
                                                                          String guidParameterName,
                                                                          String requiredClassificationName,
                                                                          String omittedClassificationName,
                                                                          int    startFrom,
                                                                          int    pageSize,
                                                                          Date   effectiveTime,
                                                                          String methodName) throws InvalidParameterException,
                                                                                                    PropertyServerException,
                                                                                                    UserNotAuthorizedException
    {
        return this.getSchemaAttributesForComplexSchemaType(userId,
                                                            schemaTypeGUID,
                                                            guidParameterName,
                                                            OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                            requiredClassificationName,
                                                            omittedClassificationName,
                                                            supportedZones,
                                                            startFrom,
                                                            pageSize,
                                                            effectiveTime,
                                                            methodName);
    }


    /**
     * Returns a list of schema attributes that are linked to a ComplexSchemaType via the AttributeForSchema relationship.
     * It validates the unique identifier of the complex schema type, and the visibility/security of any attached asset using the
     * supplied supported zones (needed for the calls from the OCF Metadata REST Services).
     * Then it begins to extract the schema attributes. Exceptions occur if a schema attribute does not have a type.
     *
     * @param userId         String   userId of user making request.
     * @param schemaTypeGUID String   unique id for containing schema type.
     * @param schemaTypeGUIDParameterName String name of the parameter supplying the guid.
     * @param requiredClassificationName  String the name of the classification that must be on the schema attribute or linked schema type entity.
     * @param omittedClassificationName   String the name of a classification that must not be on the schema attribute or linked schema type entity.
     * @param serviceSupportedZones list of zone names for calling service
   v  * @param startFrom   int      starting position for first returned element.
     * @param pageSize    int      maximum number of elements to return on the call.
     * @param methodName     calling method
     *
     * @return a schema attributes response
     * @throws InvalidParameterException - the GUID is not recognized or the paging values are invalid or
     * @throws PropertyServerException - there is a problem retrieving the asset properties from the property server or
     * @throws UserNotAuthorizedException - the requesting user is not authorized to issue this request.
     */
    public List<SCHEMA_ATTRIBUTE> getSchemaAttributesForComplexSchemaType(String       userId,
                                                                          String       schemaTypeGUID,
                                                                          String       schemaTypeGUIDParameterName,
                                                                          String       schemaAttributeTypeName,
                                                                          String       requiredClassificationName,
                                                                          String       omittedClassificationName,
                                                                          List<String> serviceSupportedZones,
                                                                          int          startFrom,
                                                                          int          pageSize,
                                                                          Date         effectiveTime,
                                                                          String       methodName) throws InvalidParameterException,
                                                                                                          PropertyServerException,
                                                                                                          UserNotAuthorizedException
    {
        final String schemaAttributeGUIDParameterName = "schemaAttributeEntity.getGUID()";

        String typeName = OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME;

        if (schemaAttributeTypeName != null)
        {
            typeName = schemaAttributeTypeName;
        }

        List<EntityDetail>  entities = this.getAttachedEntities(userId,
                                                                schemaTypeGUID,
                                                                schemaTypeGUIDParameterName,
                                                                OpenMetadataAPIMapper.SCHEMA_TYPE_TYPE_NAME,
                                                                OpenMetadataAPIMapper.TYPE_TO_ATTRIBUTE_RELATIONSHIP_TYPE_GUID,
                                                                OpenMetadataAPIMapper.TYPE_TO_ATTRIBUTE_RELATIONSHIP_TYPE_NAME,
                                                                typeName,
                                                                requiredClassificationName,
                                                                omittedClassificationName,
                                                                0,
                                                                false,
                                                                false,
                                                                serviceSupportedZones,
                                                                startFrom,
                                                                pageSize,
                                                                effectiveTime,
                                                                methodName);

        List<SCHEMA_ATTRIBUTE>  results = new ArrayList<>();

        if (entities != null)
        {
            for (EntityDetail schemaAttributeEntity : entities)
            {
                if (schemaAttributeEntity != null)
                {
                    /*
                     * This method verifies the visibility of the entity and the security permission.
                     */
                    results.add(this.getSchemaAttributeFromEntity(userId,
                                                                  schemaAttributeEntity.getGUID(),
                                                                  schemaAttributeGUIDParameterName,
                                                                  schemaAttributeEntity,
                                                                  effectiveTime,
                                                                  methodName));
                }
            }
        }

        if (results.isEmpty())
        {
            return null;
        }
        else
        {
            return results;
        }
    }


    /**
     * Returns a list of schema attributes that are linked to a parent schema element.  This may be a complex schema type or a
     * schema attribute.  It is necessary to find out the type of the parent schema element to be sure which type of
     * retrieval is needed.
      *
     * @param userId         String   userId of user making request.
     * @param parentElementGUID String   unique id for parent schema element.
     * @param parentElementGUIDParameterName String name of the parameter supplying the guid.
     * @param schemaAttributeTypeName sub type of schema attribute or null
     * @param startFrom   int      starting position for first returned element.
     * @param pageSize    int      maximum number of elements to return on the call.
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName     calling method
     *
     * @return a schema attributes response
     * @throws InvalidParameterException - the GUID is not recognized or the paging values are invalid or
     * @throws PropertyServerException - there is a problem retrieving the asset properties from the property server or
     * @throws UserNotAuthorizedException - the requesting user is not authorized to issue this request.
     */
    public List<SCHEMA_ATTRIBUTE> getAttachedSchemaAttributes(String userId,
                                                              String parentElementGUID,
                                                              String parentElementGUIDParameterName,
                                                              String schemaAttributeTypeName,
                                                              int    startFrom,
                                                              int    pageSize,
                                                              Date   effectiveTime,
                                                              String methodName) throws InvalidParameterException,
                                                                                        PropertyServerException,
                                                                                        UserNotAuthorizedException
    {
        EntityDetail parentEntity = this.getEntityFromRepository(userId,
                                                                 parentElementGUID,
                                                                 parentElementGUIDParameterName,
                                                                 OpenMetadataAPIMapper.SCHEMA_ELEMENT_TYPE_NAME,
                                                                 null,
                                                                 null,
                                                                 false,
                                                                 false,
                                                                 supportedZones,
                                                                 effectiveTime,
                                                                 methodName);

        if ((parentEntity != null) && (parentEntity.getType() != null))
        {
            if (repositoryHelper.isTypeOf(serviceName, parentEntity.getType().getTypeDefName(), OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME))
            {
                return this.getNestedSchemaAttributes(userId,
                                                      parentElementGUID,
                                                      parentElementGUIDParameterName,
                                                      schemaAttributeTypeName,
                                                      supportedZones,
                                                      startFrom,
                                                      pageSize,
                                                      effectiveTime,
                                                      methodName);
            }
            else
            {
                return this.getSchemaAttributesForComplexSchemaType(userId,
                                                                    parentElementGUID,
                                                                    parentElementGUIDParameterName,
                                                                    schemaAttributeTypeName,
                                                                    null,
                                                                    null,
                                                                    supportedZones,
                                                                    startFrom,
                                                                    pageSize,
                                                                    effectiveTime,
                                                                    methodName);
            }
        }

        return null;
    }


    /**
     * Returns a list of schema attributes that are linked to a schema attribute via the NestedSchemaAttribute relationship.
     * It validates the unique identifier of the parent schema attribute, and the visibility/security of any attached asset using the
     * supplied supported zones (needed for the calls from the OCF Metadata REST Services).
     * Then it begins to extract the schema attributes. Exceptions occur if a schema attribute does not have a type.
     *
     * @param userId         String   userId of user making request.
     * @param schemaAttributeGUID String   unique id for containing schema attribute.
     * @param schemaAttributeGUIDParameterName String name of the parameter supplying the guid.
     * @param startFrom   int      starting position for first returned element.
     * @param pageSize    int      maximum number of elements to return on the call.
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName     calling method
     *
     * @return a schema attributes response
     * @throws InvalidParameterException - the GUID is not recognized or the paging values are invalid or
     * @throws PropertyServerException - there is a problem retrieving the asset properties from the property server or
     * @throws UserNotAuthorizedException - the requesting user is not authorized to issue this request.
     */
    public List<SCHEMA_ATTRIBUTE> getNestedSchemaAttributes(String userId,
                                                            String schemaAttributeGUID,
                                                            String schemaAttributeGUIDParameterName,
                                                            int    startFrom,
                                                            int    pageSize,
                                                            Date   effectiveTime,
                                                            String methodName) throws InvalidParameterException,
                                                                                      PropertyServerException,
                                                                                      UserNotAuthorizedException
    {
        return getNestedSchemaAttributes(userId,
                                         schemaAttributeGUID,
                                         schemaAttributeGUIDParameterName,
                                         OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                         supportedZones,
                                         startFrom,
                                         pageSize,
                                         effectiveTime,
                                         methodName);
    }


    /**
     * Returns a list of schema attributes that are linked to a schema attribute via the NestedSchemaAttribute relationship.
     * It validates the unique identifier of the parent schema attribute, and the visibility/security of any attached asset using the
     * supplied supported zones (needed for the calls from the OCF Metadata REST Services).
     * Then it begins to extract the schema attributes. Exceptions occur if a schema attribute does not have a type.
     *
     * @param userId         String   userId of user making request.
     * @param schemaAttributeGUID String   unique id for containing schema attribute.
     * @param schemaAttributeGUIDParameterName String name of the parameter supplying the guid.
     * @param schemaAttributeTypeName subtype of schema attribute (or null)
     * @param serviceSupportedZones list of zone names for calling service
     * @param startFrom   int      starting position for first returned element.
     * @param pageSize    int      maximum number of elements to return on the call.
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName     calling method
     *
     * @return a schema attributes response
     * @throws InvalidParameterException - the GUID is not recognized or the paging values are invalid or
     * @throws PropertyServerException - there is a problem retrieving the asset properties from the property server or
     * @throws UserNotAuthorizedException - the requesting user is not authorized to issue this request.
     */
    public List<SCHEMA_ATTRIBUTE> getNestedSchemaAttributes(String       userId,
                                                            String       schemaAttributeGUID,
                                                            String       schemaAttributeGUIDParameterName,
                                                            String       schemaAttributeTypeName,
                                                            List<String> serviceSupportedZones,
                                                            int          startFrom,
                                                            int          pageSize,
                                                            Date         effectiveTime,
                                                            String       methodName) throws InvalidParameterException,
                                                                                            PropertyServerException,
                                                                                            UserNotAuthorizedException
    {
        final String nestedSchemaAttributeGUIDParameterName = "schemaAttributeEntity.getGUID()";

        String resultTypeName = OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME;

        if (schemaAttributeTypeName != null)
        {
            resultTypeName = schemaAttributeTypeName;
        }

        List<EntityDetail>  entities = this.getAttachedEntities(userId,
                                                                schemaAttributeGUID,
                                                                schemaAttributeGUIDParameterName,
                                                                OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                                OpenMetadataAPIMapper.NESTED_ATTRIBUTE_RELATIONSHIP_TYPE_GUID,
                                                                OpenMetadataAPIMapper.NESTED_ATTRIBUTE_RELATIONSHIP_TYPE_NAME,
                                                                resultTypeName,
                                                                null,
                                                                null,
                                                                2,
                                                                false,
                                                                false,
                                                                serviceSupportedZones,
                                                                startFrom,
                                                                pageSize,
                                                                effectiveTime,
                                                                methodName);

        List<SCHEMA_ATTRIBUTE>  results = new ArrayList<>();

        if (entities != null)
        {
            for (EntityDetail schemaAttributeEntity : entities)
            {
                if (schemaAttributeEntity != null)
                {
                    /*
                     * This method verifies the visibility of the entity and the security permission.
                     */
                    results.add(this.getSchemaAttributeFromEntity(userId,
                                                                  schemaAttributeEntity.getGUID(),
                                                                  nestedSchemaAttributeGUIDParameterName,
                                                                  schemaAttributeEntity,
                                                                  effectiveTime,
                                                                  methodName));
                }
            }
        }
        else
        {
            /*
             * Using the old pattern where a schema type is between the nested schema attributes.
             */
            EntityDetail entity = this.getAttachedEntity(userId,
                                                         schemaAttributeGUID,
                                                         schemaAttributeGUIDParameterName,
                                                         OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                         OpenMetadataAPIMapper.ATTRIBUTE_TO_TYPE_RELATIONSHIP_TYPE_GUID,
                                                         OpenMetadataAPIMapper.ATTRIBUTE_TO_TYPE_RELATIONSHIP_TYPE_NAME,
                                                         OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                         false,
                                                         false,
                                                         supportedZones,
                                                         effectiveTime,
                                                         methodName);

            if (entity != null)
            {
                String schemaTypeGUIDParameterName = "schemaTypeGUID";

                return getSchemaAttributesForComplexSchemaType(userId,
                                                               entity.getGUID(),
                                                               schemaTypeGUIDParameterName,
                                                               schemaAttributeTypeName,
                                                               null,
                                                               null,
                                                               supportedZones,
                                                               startFrom,
                                                               pageSize,
                                                               effectiveTime,
                                                               methodName);
            }
        }

        if (results.isEmpty())
        {
            return null;
        }
        else
        {
            return results;
        }
    }


    /**
     * Retrieve the special links (like foreign keys) between attributes.
     *
     * @param userId calling user
     * @param schemaAttributeEntity details from the repository
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName calling method
     * @return list of schema attribute relationships populated with information from the repository
     * @throws InvalidParameterException  the parameters are invalid
     * @throws UserNotAuthorizedException user not authorized to issue this request
     * @throws PropertyServerException    problem accessing the property server
     */
    private List<Relationship> getSchemaAttributeRelationships(String       userId,
                                                               EntityDetail schemaAttributeEntity,
                                                               Date         effectiveTime,
                                                               String       methodName) throws InvalidParameterException,
                                                                                                PropertyServerException,
                                                                                                UserNotAuthorizedException
    {
        final String schemaAttributeGUIDParameterName = "schemaAttributeEntity";

        List<Relationship> results = new ArrayList<>();

        List<Relationship> relationships = this.getAttachmentLinks(userId,
                                                                   schemaAttributeEntity.getGUID(),
                                                                   schemaAttributeGUIDParameterName,
                                                                   OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                                   null,
                                                                   null,
                                                                   OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                                   0,
                                                                   invalidParameterHandler.getMaxPagingSize(),
                                                                   effectiveTime,
                                                                   methodName);

        if (relationships != null)
        {
            for (Relationship relationship : relationships)
            {
                if ((relationship != null) && (relationship.getType() != null))
                {
                    String typeName = relationship.getType().getTypeDefName();

                    if (OpenMetadataAPIMapper.FOREIGN_KEY_RELATIONSHIP_TYPE_NAME.equals(typeName))
                    {
                        results.add(relationship);
                    }
                    else if (OpenMetadataAPIMapper.GRAPH_EDGE_LINK_RELATIONSHIP_TYPE_NAME.equals(typeName))
                    {
                        results.add(relationship);
                    }
                }
            }
        }

        if (results.isEmpty())
        {
            return null;
        }
        else
        {
            return results;
        }
    }


    /**
     * From the schema attribute entity, gather the related schema type, schema link and/or schema attribute relationships
     * and create a schema attribute bean.  The caller is expected to have validated that is is ok to return this schema attribute.
     *
     * @param userId calling userId
     * @param schemaAttributeGUID unique identifier of schema attribute
     * @param schemaAttributeGUIDParameterName parameter passing the schemaAttributeGUID
     * @param schemaAttributeEntity entity retrieved for the schema attribute
     * @param effectiveTime time that the element should be active
     * @param methodName calling method
     * @return a new schema attribute object, or null if the schema attribute is either not visible to the user, or its classifications
     *         are not what are requested.
     *
     * @throws InvalidParameterException the guid or bean properties are invalid
     * @throws UserNotAuthorizedException user not authorized to issue this request
     * @throws PropertyServerException problem accessing the property server
     */
    private SCHEMA_ATTRIBUTE getSchemaAttributeFromEntity(String       userId,
                                                          String       schemaAttributeGUID,
                                                          String       schemaAttributeGUIDParameterName,
                                                          EntityDetail schemaAttributeEntity,
                                                          Date         effectiveTime,
                                                          String       methodName) throws InvalidParameterException,
                                                                                             PropertyServerException,
                                                                                             UserNotAuthorizedException
    {
        if ((schemaAttributeEntity != null) && (schemaAttributeEntity.getType() != null))
        {
            /*
             * Extra entities are required depending on the type of the schema.
             */
            SCHEMA_TYPE  schemaType  = null;

            /*
             * The table may have its type stored as a classification, or as a linked schema type.  The column is linked to
             * the attribute in the first case, and the schema type in the second case.
             */
            try
            {
                Classification typeClassification = repositoryHelper.getClassificationFromEntity(serviceName,
                                                                                                 schemaAttributeEntity,
                                                                                                 OpenMetadataAPIMapper.TYPE_EMBEDDED_ATTRIBUTE_CLASSIFICATION_TYPE_NAME,
                                                                                                 methodName);

                if ((typeClassification != null) && (typeClassification.getProperties() != null))
                {
                    String schemaTypeName = repositoryHelper.getStringProperty(serviceName,
                                                                               OpenMetadataAPIMapper.TYPE_NAME_PROPERTY_NAME,
                                                                               typeClassification.getProperties(),
                                                                               methodName);

                    schemaType = schemaTypeHandler.getSchemaTypeFromInstance(userId,
                                                                             schemaAttributeEntity,
                                                                             schemaTypeName,
                                                                             typeClassification.getProperties(),
                                                                             schemaAttributeEntity.getClassifications(),
                                                                             effectiveTime,
                                                                             methodName);
                }
            }
            catch (ClassificationErrorException classificationNotKnown)
            {
                /*
                 * Type classification not supported.
                 */
            }


            if (schemaType == null)
            {
                /*
                 * Look for an explicit private schema type
                 */
                schemaType = schemaTypeHandler.getSchemaTypeForParent(userId,
                                                                      schemaAttributeGUID,
                                                                      schemaAttributeGUIDParameterName,
                                                                      OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                                      OpenMetadataAPIMapper.ATTRIBUTE_TO_TYPE_RELATIONSHIP_TYPE_GUID,
                                                                      OpenMetadataAPIMapper.ATTRIBUTE_TO_TYPE_RELATIONSHIP_TYPE_NAME,
                                                                      effectiveTime,
                                                                      methodName);
            }


            List<Relationship> attributeRelationships = this.getSchemaAttributeRelationships(userId, schemaAttributeEntity, effectiveTime, methodName);

            return schemaAttributeConverter.getNewSchemaAttributeBean(beanClass,
                                                                      schemaAttributeEntity,
                                                                      schemaTypeBeanClass,
                                                                      schemaType,
                                                                      attributeRelationships,
                                                                      methodName);
        }

        return null;
    }


    /**
     * Work through the schema attributes adding or updating the instances
     *
     * @param userId calling userId
     * @param schemaAttributeEntities list of retrieved entities
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName calling method
     * @return list of new schema attributes
     *
     * @throws InvalidParameterException the guid or bean properties are invalid
     * @throws UserNotAuthorizedException user not authorized to issue this request
     * @throws PropertyServerException problem accessing the property server
     */
    private List<SCHEMA_ATTRIBUTE> getSchemaAttributesFromEntities(String             userId,
                                                                   List<EntityDetail> schemaAttributeEntities,
                                                                   Date               effectiveTime,
                                                                   String             methodName) throws InvalidParameterException,
                                                                                                         PropertyServerException,
                                                                                                         UserNotAuthorizedException
    {
        final String parameterName = "schemaAttributeEntities";

        if (schemaAttributeEntities != null)
        {
            List<SCHEMA_ATTRIBUTE> schemaAttributes = new ArrayList<>();

            for (EntityDetail entity : schemaAttributeEntities)
            {
                if (entity != null)
                {
                    schemaAttributes.add(this.getSchemaAttributeFromEntity(userId,
                                                                           entity.getGUID(),
                                                                           parameterName,
                                                                           entity,
                                                                           effectiveTime,
                                                                           methodName));
                }
            }

            if (!schemaAttributes.isEmpty())
            {
                return schemaAttributes;
            }
        }

        return null;
    }


    /**
     * Retrieve a specific schema attribute by GUID.   It is only returned if the guid is known, the entity is of the correct type,
     * the classifications are as expected and any asset it is connected to is both visible via the zones setting and the
     * security verifier allows the update.
     *
     * @param userId calling userId
     * @param schemaAttributeGUID unique identifier of schema attribute
     * @param schemaAttributeGUIDParameterName parameter passing the schemaAttributeGUID
     * @param expectedTypeName type or subtype of schema attribute
     * @param requiredClassificationName a classification that must be either on the schema attribute or its type.
     * @param omittedClassificationName a classification that must NOT be on either the schema attribute or its type.
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName calling method
     * @return guid of new schema
     *
     * @throws InvalidParameterException the guid or bean properties are invalid
     * @throws UserNotAuthorizedException user not authorized to issue this request
     * @throws PropertyServerException problem accessing the property server
     */
    public SCHEMA_ATTRIBUTE getSchemaAttribute(String userId,
                                               String schemaAttributeGUID,
                                               String schemaAttributeGUIDParameterName,
                                               String expectedTypeName,
                                               String requiredClassificationName,
                                               String omittedClassificationName,
                                               Date   effectiveTime,
                                               String methodName) throws InvalidParameterException,
                                                                         PropertyServerException,
                                                                         UserNotAuthorizedException
    {
        final String typeParameterName = "expectedTypeName";

        invalidParameterHandler.validateGUID(schemaAttributeGUID, schemaAttributeGUIDParameterName, methodName);
        invalidParameterHandler.validateName(expectedTypeName, typeParameterName, methodName);

        /*
         * This method verifies the GUID and the type
         */
        EntityDetail schemaAttributeEntity = this.getEntityFromRepository(userId,
                                                                          schemaAttributeGUID,
                                                                          schemaAttributeGUIDParameterName,
                                                                          expectedTypeName,
                                                                          requiredClassificationName,
                                                                          omittedClassificationName,
                                                                          false,
                                                                          false,
                                                                          supportedZones,
                                                                          effectiveTime,
                                                                          methodName);

        /*
         * This method issues additional retrieves to the metadata repositories to build the schema attribute bean, its type and any attribute
         * relationships.
         */
        return this.getSchemaAttributeFromEntity(userId,
                                                 schemaAttributeGUID,
                                                 schemaAttributeGUIDParameterName,
                                                 schemaAttributeEntity,
                                                 effectiveTime,
                                                 methodName);
    }


    /**
     * Returns a list of specifically typed schema attributes with matching names - either display names or qualified names
     *
     * @param userId         String   userId of user making request.
     * @param typeGUID       String   unique identifier of type of schema attribute required.
     * @param typeName       String  unique name of type of schema attribute required.
     * @param name           String   name (qualified or display name) of schema attribute.
     * @param requiredClassificationName  String name of classification that must be present
     * @param omittedClassificationName  String name of classification that must not be present
     * @param elementStart   int      starting position for first returned element.
     * @param maxElements    int      maximum number of elements to return on the call.
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName     calling method
     *
     * @return a schema attributes response
     * @throws InvalidParameterException - the GUID is not recognized or the paging values are invalid or
     * @throws PropertyServerException - there is a problem retrieving the asset properties from the property server or
     * @throws UserNotAuthorizedException - the requesting user is not authorized to issue this request.
     */
    public List<SCHEMA_ATTRIBUTE> getSchemaAttributesByName(String       userId,
                                                            String       typeGUID,
                                                            String       typeName,
                                                            String       name,
                                                            String       requiredClassificationName,
                                                            String       omittedClassificationName,
                                                            int          elementStart,
                                                            int          maxElements,
                                                            Date         effectiveTime,
                                                            String       methodName) throws InvalidParameterException,
                                                                                            PropertyServerException,
                                                                                            UserNotAuthorizedException
    {
        return getSchemaAttributesByName(userId,
                                         typeGUID,
                                         typeName,
                                         name,
                                         requiredClassificationName,
                                         omittedClassificationName,
                                         supportedZones,
                                         elementStart,
                                         maxElements,
                                         effectiveTime,
                                         methodName);
    }


    /**
     * Returns a list of specifically typed schema attributes with matching names - either display names or qualified names
     *
     * @param userId         String   userId of user making request
     * @param typeGUID       String   unique identifier of type of schema attribute required
     * @param typeName       String  unique name of type of schema attribute required
     * @param name           String   name (qualified or display name) of schema attribute
     * @param requiredClassificationName  String name of classification that must be present
     * @param omittedClassificationName  String name of classification that must not be present
     * @param serviceSupportedZones zones that assets must be;ong in to be visible
     * @param startFrom   int      starting position for first returned element
     * @param pageSize    int      maximum number of elements to return on the call
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName     calling method
     *
     * @return a schema attributes response
     * @throws InvalidParameterException - the GUID is not recognized or the paging values are invalid or
     * @throws PropertyServerException - there is a problem retrieving the asset properties from the property server or
     * @throws UserNotAuthorizedException - the requesting user is not authorized to issue this request.
     */
    public List<SCHEMA_ATTRIBUTE> getSchemaAttributesByName(String       userId,
                                                            String       typeGUID,
                                                            String       typeName,
                                                            String       name,
                                                            String       requiredClassificationName,
                                                            String       omittedClassificationName,
                                                            List<String> serviceSupportedZones,
                                                            int          startFrom,
                                                            int          pageSize,
                                                            Date         effectiveTime,
                                                            String       methodName) throws InvalidParameterException,
                                                                                            PropertyServerException,
                                                                                            UserNotAuthorizedException
    {
        final String nameParameterName = "name";

        List<String> specificMatchPropertyNames = new ArrayList<>();

        specificMatchPropertyNames.add(OpenMetadataAPIMapper.QUALIFIED_NAME_PROPERTY_NAME);
        specificMatchPropertyNames.add(OpenMetadataAPIMapper.DISPLAY_NAME_PROPERTY_NAME);

        List<EntityDetail>  schemaAttributeEntities = this.getEntitiesByValue(userId,
                                                                              name,
                                                                              nameParameterName,
                                                                              typeGUID,
                                                                              typeName,
                                                                              specificMatchPropertyNames,
                                                                              true,
                                                                              requiredClassificationName,
                                                                              omittedClassificationName,
                                                                              false,
                                                                              false,
                                                                              serviceSupportedZones,
                                                                              null,
                                                                              startFrom,
                                                                              pageSize,
                                                                              effectiveTime,
                                                                              methodName);

        return this.getSchemaAttributesFromEntities(userId, schemaAttributeEntities, effectiveTime, methodName);
    }


    /**
     * Retrieve the list of database table metadata elements that contain the search string.
     * The search string is treated as a regular expression.
     *
     * @param userId the searchString of the calling user.
     * @param searchString searchString of endpoint.  This may include wild card characters.
     * @param searchStringParameterName name of parameter providing search string
     * @param resultTypeGUID unique identifier of the type that the results should match with
     * @param resultTypeName unique value of the type that the results should match with
     * @param requiredClassificationName  String the name of the classification that must be on the entity.
     * @param omittedClassificationName   String the name of a classification that must not be on the entity.
     * @param startFrom  index of the list ot start from (0 for start)
     * @param pageSize   maximum number of elements to return.
     * @param effectiveTime the time that the retrieved elements must be effective for (null for any time, new Date() for now)
     * @param methodName calling method
     *
     * @return list of matching schema attribute elements
     *
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public List<SCHEMA_ATTRIBUTE>   findSchemaAttributes(String userId,
                                                         String searchString,
                                                         String searchStringParameterName,
                                                         String resultTypeGUID,
                                                         String resultTypeName,
                                                         String requiredClassificationName,
                                                         String omittedClassificationName,
                                                         int    startFrom,
                                                         int    pageSize,
                                                         Date   effectiveTime,
                                                         String methodName) throws InvalidParameterException,
                                                                                    UserNotAuthorizedException,
                                                                                    PropertyServerException
    {
        List<EntityDetail> schemaAttributeEntities = this.findEntities(userId,
                                                                       searchString,
                                                                       searchStringParameterName,
                                                                       resultTypeGUID,
                                                                       resultTypeName,
                                                                       requiredClassificationName,
                                                                       omittedClassificationName,
                                                                       null,
                                                                       startFrom,
                                                                       pageSize,
                                                                       effectiveTime,
                                                                       methodName);

        return this.getSchemaAttributesFromEntities(userId, schemaAttributeEntities, effectiveTime, methodName);
    }


    /**
     * Update the properties in a schema attribute.
     *
     * @param userId calling user
     * @param externalSourceGUID unique identifier of software server capability representing the caller
     * @param externalSourceName unique name of software server capability representing the caller
     * @param schemaAttributeGUID unique identifier of the metadata element to connect the new schema attribute to
     * @param schemaAttributeGUIDParameterName parameter name supplying schemaAttributeGUID
     * @param qualifiedName unique identifier for this schema type
     * @param qualifiedNameParameterName name of parameter supplying the qualified name
     * @param displayName the stored display name property for the database table
     * @param description the stored description property associated with the database table
     * @param externalSchemaTypeGUID unique identifier of an external schema identifier
     * @param dataType data type name - for stored values
     * @param defaultValue string containing default value - for stored values
     * @param fixedValue string containing fixed value - for literals
     * @param validValuesSetGUID unique identifier for a valid values set to support
     * @param formula String formula - for derived values
     * @param isDeprecated is this table deprecated?
     * @param elementPosition the position of this column in its parent table.
     * @param minCardinality minimum number of repeating instances allowed for this column - typically 1
     * @param maxCardinality the maximum number of repeating instances allowed for this column - typically 1
     * @param allowsDuplicateValues  whether the same value can be used by more than one instance of this attribute
     * @param orderedValues whether the attribute instances are arranged in an order
     * @param sortOrder the order that the attribute instances are arranged in - if any
     * @param minimumLength the minimum length of the data
     * @param length the length of the data field
     * @param significantDigits number of significant digits to the right of decimal point
     * @param isNullable whether the field is nullable or not
     * @param nativeJavaClass equivalent Java class implementation
     * @param defaultValueOverride default value for this column
     * @param aliases a list of alternative names for the attribute
     * @param additionalProperties any arbitrary properties not part of the type system
     * @param extendedProperties properties from any subtype
     * @param isMergeUpdate should the new properties be merged with existing properties (true) or completely replace them (false)?
     * @param methodName calling method
     *
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public void   updateSchemaAttribute(String               userId,
                                        String               externalSourceGUID,
                                        String               externalSourceName,
                                        String               schemaAttributeGUID,
                                        String               schemaAttributeGUIDParameterName,
                                        String               qualifiedName,
                                        String               qualifiedNameParameterName,
                                        String               displayName,
                                        String               description,
                                        String               externalSchemaTypeGUID,
                                        String               dataType,
                                        String               defaultValue,
                                        String               fixedValue,
                                        String               validValuesSetGUID,
                                        String               formula,
                                        boolean              isDeprecated,
                                        int                  elementPosition,
                                        int                  minCardinality,
                                        int                  maxCardinality,
                                        boolean              allowsDuplicateValues,
                                        boolean              orderedValues,
                                        String               defaultValueOverride,
                                        int                  sortOrder,
                                        int                  minimumLength,
                                        int                  length,
                                        int                  significantDigits,
                                        boolean              isNullable,
                                        String               nativeJavaClass,
                                        List<String>         aliases,
                                        Map<String, String>  additionalProperties,
                                        String               typeName,
                                        Map<String, Object>  extendedProperties,
                                        boolean              isMergeUpdate,
                                        String               methodName) throws InvalidParameterException,
                                                                                UserNotAuthorizedException,
                                                                                PropertyServerException
    {
        invalidParameterHandler.validateUserId(userId, methodName);
        invalidParameterHandler.validateGUID(schemaAttributeGUID, schemaAttributeGUIDParameterName, methodName);

        if (! isMergeUpdate)
        {
            invalidParameterHandler.validateName(qualifiedName, qualifiedNameParameterName, methodName);
        }

        /*
         * Check that the type name requested is valid.
         */
        String attributeTypeName = OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME;
        String attributeTypeId   = OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_GUID;

        if (typeName != null)
        {
            attributeTypeName = typeName;
            attributeTypeId   = invalidParameterHandler.validateTypeName(typeName,
                                                                         OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                                                         serviceName,
                                                                         methodName,
                                                                         repositoryHelper);
        }

        EntityDetail schemaAttributeEntity = this.getEntityFromRepository(userId,
                                                                          schemaAttributeGUID,
                                                                          schemaAttributeGUIDParameterName,
                                                                          attributeTypeName,
                                                                          null,
                                                                          null,
                                                                          false,
                                                                          false,
                                                                          null,
                                                                          methodName);

        if (schemaAttributeEntity != null)
        {
            /*
             * Load up the builder objects.  The builders manage the properties of the metadata elements that make up the schema attribute,
             * and the schemaTypeHandler manages the type.
             */
            SchemaAttributeBuilder schemaAttributeBuilder = new SchemaAttributeBuilder(qualifiedName,
                                                                                       displayName,
                                                                                       description,
                                                                                       elementPosition,
                                                                                       minCardinality,
                                                                                       maxCardinality,
                                                                                       isDeprecated,
                                                                                       defaultValueOverride,
                                                                                       allowsDuplicateValues,
                                                                                       orderedValues,
                                                                                       sortOrder,
                                                                                       minimumLength,
                                                                                       length,
                                                                                       significantDigits,
                                                                                       isNullable,
                                                                                       nativeJavaClass,
                                                                                       aliases,
                                                                                       additionalProperties,
                                                                                       attributeTypeId,
                                                                                       attributeTypeName,
                                                                                       extendedProperties,
                                                                                       repositoryHelper,
                                                                                       serviceName,
                                                                                       serverName);


            InstanceProperties instanceProperties = schemaAttributeBuilder.getInstanceProperties(methodName);

            this.updateBeanInRepository(userId,
                                        externalSourceGUID,
                                        externalSourceName,
                                        schemaAttributeGUID,
                                        schemaAttributeGUIDParameterName,
                                        OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_GUID,
                                        OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                        false,
                                        false,
                                        supportedZones,
                                        instanceProperties,
                                        true,
                                        this.getEffectiveTime(instanceProperties),
                                        methodName);

            SchemaTypeBuilder schemaTypeBuilder = this.getSchemaTypeBuilder(qualifiedName,
                                                                            externalSchemaTypeGUID,
                                                                            dataType,
                                                                            defaultValue,
                                                                            fixedValue,
                                                                            validValuesSetGUID);

            // todo this logic assumes the schema type is stored as a classification
            setClassificationInRepository(userId,
                                          externalSourceGUID,
                                          externalSourceName,
                                          schemaAttributeEntity,
                                          schemaAttributeGUIDParameterName,
                                          attributeTypeName,
                                          OpenMetadataAPIMapper.TYPE_EMBEDDED_ATTRIBUTE_CLASSIFICATION_TYPE_GUID,
                                          OpenMetadataAPIMapper.TYPE_EMBEDDED_ATTRIBUTE_CLASSIFICATION_TYPE_NAME,
                                          schemaTypeBuilder.getTypeEmbeddedInstanceProperties(methodName),
                                          isMergeUpdate,
                                          false,
                                          false,
                                          null,
                                          methodName);

            /*
             * The formula is set if the column is derived
             */
            if (formula != null)
            {
                schemaAttributeBuilder.setCalculatedValue(userId, externalSourceGUID, externalSourceName, formula, methodName);

                setClassificationInRepository(userId,
                                              externalSourceGUID,
                                              externalSourceName,
                                              schemaAttributeEntity,
                                              schemaAttributeGUIDParameterName,
                                              attributeTypeName,
                                              OpenMetadataAPIMapper.TYPE_EMBEDDED_ATTRIBUTE_CLASSIFICATION_TYPE_GUID,
                                              OpenMetadataAPIMapper.TYPE_EMBEDDED_ATTRIBUTE_CLASSIFICATION_TYPE_NAME,
                                              schemaTypeBuilder.getTypeEmbeddedInstanceProperties(methodName),
                                              isMergeUpdate,
                                              false,
                                              false,
                                              null,
                                              methodName);
            }
        }
    }


    /**
     * Update a schema attribute
     *
     * @param userId                      calling user
     * @param externalSourceGUID          unique identifier of software server capability representing the caller
     * @param externalSourceName          unique name of software server capability representing the caller
     * @param schemaAttributeGUID         unique identifier of schema attribute
     * @param instanceProperties          the schema attribute's properties
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public void updateSchemaAttribute(String             userId,
                                      String             externalSourceGUID,
                                      String             externalSourceName,
                                      String             schemaAttributeGUID,
                                      InstanceProperties instanceProperties) throws InvalidParameterException,
                                                                                    PropertyServerException,
                                                                                    UserNotAuthorizedException
    {
        final String methodName = "updateSchemaAttribute";
        final String parameterName = "schemaAttributeGUID";

        this.updateBeanInRepository(userId,
                                    externalSourceGUID,
                                    externalSourceName,
                                    schemaAttributeGUID,
                                    parameterName,
                                    OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_GUID,
                                    OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                    false,
                                    false,
                                    supportedZones,
                                    instanceProperties,
                                    true,
                                    this.getEffectiveTime(instanceProperties),
                                    methodName);
    }


    /**
     * Update a schema attribute
     *
     * @param userId                      calling user
     * @param externalSourceGUID          unique identifier of software server capability representing the caller
     * @param externalSourceName          unique name of software server capability representing the caller
     * @param schemaAttributeGUID         unique identifier of schema attribute
     * @param schemaAttributeGUIDParameterName  parameter supplying schemaAttributeGUID
     * @param instanceProperties          the schema attribute's properties
     * @param isMergeUpdate               should the properties be merged with existing properties of replace them?
     * @param methodName                  calling method
     *
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public void updateSchemaAttribute(String             userId,
                                      String             externalSourceGUID,
                                      String             externalSourceName,
                                      String             schemaAttributeGUID,
                                      String             schemaAttributeGUIDParameterName,
                                      InstanceProperties instanceProperties,
                                      boolean            isMergeUpdate,
                                      String             methodName) throws InvalidParameterException,
                                                                            PropertyServerException,
                                                                            UserNotAuthorizedException
    {
        this.updateBeanInRepository(userId,
                                    externalSourceGUID,
                                    externalSourceName,
                                    schemaAttributeGUID,
                                    schemaAttributeGUIDParameterName,
                                    OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_GUID,
                                    OpenMetadataAPIMapper.SCHEMA_ATTRIBUTE_TYPE_NAME,
                                    false,
                                    false,
                                    supportedZones,
                                    instanceProperties,
                                    isMergeUpdate,
                                    this.getEffectiveTime(instanceProperties),
                                    methodName);
    }


    /**
     * Remove any links to schema types
     *
     * @param userId                      calling user
     * @param externalSourceGUID          unique identifier of software server capability representing the caller
     * @param externalSourceName          unique name of software server capability representing the caller
     * @param schemaAttributeGUID         unique identifier of schema attribute
     * @param schemaAttributeGUIDParameterName  parameter supplying schemaAttributeGUID
     * @param methodName                  calling method
     *
     * @throws InvalidParameterException  one of the parameters is invalid
     * @throws UserNotAuthorizedException the user is not authorized to issue this request
     * @throws PropertyServerException    there is a problem reported in the open metadata server(s)
     */
    public void removeSchemaTypes(String             userId,
                                  String             externalSourceGUID,
                                  String             externalSourceName,
                                  String             schemaAttributeGUID,
                                  String             schemaAttributeGUIDParameterName,
                                  String             methodName) throws InvalidParameterException,
                                                                        PropertyServerException,
                                                                        UserNotAuthorizedException
    {
        // todo retrieve relationships and remove those that link a schema attribute to its type(s)
    }
}
