/*
 * Decompiled with CFR 0.152.
 */
package org.janusgraph.diskstorage;

import com.google.common.base.Preconditions;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import org.apache.commons.lang.StringUtils;
import org.apache.tinkerpop.gremlin.process.traversal.util.TraversalInterruptedException;
import org.janusgraph.core.JanusGraphException;
import org.janusgraph.diskstorage.BackendException;
import org.janusgraph.diskstorage.BaseTransactionConfig;
import org.janusgraph.diskstorage.Entry;
import org.janusgraph.diskstorage.EntryList;
import org.janusgraph.diskstorage.LoggableTransaction;
import org.janusgraph.diskstorage.PermanentBackendException;
import org.janusgraph.diskstorage.StaticBuffer;
import org.janusgraph.diskstorage.indexing.IndexQuery;
import org.janusgraph.diskstorage.indexing.IndexTransaction;
import org.janusgraph.diskstorage.indexing.RawQuery;
import org.janusgraph.diskstorage.keycolumnvalue.KeyIterator;
import org.janusgraph.diskstorage.keycolumnvalue.KeyRangeQuery;
import org.janusgraph.diskstorage.keycolumnvalue.KeySliceQuery;
import org.janusgraph.diskstorage.keycolumnvalue.SliceQuery;
import org.janusgraph.diskstorage.keycolumnvalue.StoreFeatures;
import org.janusgraph.diskstorage.keycolumnvalue.StoreTransaction;
import org.janusgraph.diskstorage.keycolumnvalue.cache.CacheTransaction;
import org.janusgraph.diskstorage.keycolumnvalue.cache.KCVSCache;
import org.janusgraph.diskstorage.log.kcvs.ExternalCachePersistor;
import org.janusgraph.diskstorage.util.BackendOperation;
import org.janusgraph.diskstorage.util.BufferUtil;
import org.janusgraph.graphdb.database.serialize.DataOutput;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackendTransaction
implements LoggableTransaction {
    private static final Logger log = LoggerFactory.getLogger(BackendTransaction.class);
    public static final int MIN_TASKS_TO_PARALLELIZE = 2;
    public static final StaticBuffer EDGESTORE_MIN_KEY = BufferUtil.zeroBuffer(8);
    public static final StaticBuffer EDGESTORE_MAX_KEY = BufferUtil.oneBuffer(8);
    private final CacheTransaction storeTx;
    private final BaseTransactionConfig txConfig;
    private final StoreFeatures storeFeatures;
    private final KCVSCache edgeStore;
    private final KCVSCache indexStore;
    private final KCVSCache txLogStore;
    private final Duration maxReadTime;
    private final Executor threadPool;
    private final Map<String, IndexTransaction> indexTx;
    private boolean acquiredLock = false;
    private boolean cacheEnabled = true;

    public BackendTransaction(CacheTransaction storeTx, BaseTransactionConfig txConfig, StoreFeatures features, KCVSCache edgeStore, KCVSCache indexStore, KCVSCache txLogStore, Duration maxReadTime, Map<String, IndexTransaction> indexTx, Executor threadPool) {
        this.storeTx = storeTx;
        this.txConfig = txConfig;
        this.storeFeatures = features;
        this.edgeStore = edgeStore;
        this.indexStore = indexStore;
        this.txLogStore = txLogStore;
        this.maxReadTime = maxReadTime;
        this.indexTx = indexTx;
        this.threadPool = threadPool;
    }

    public boolean hasAcquiredLock() {
        return this.acquiredLock;
    }

    public StoreTransaction getStoreTransaction() {
        return this.storeTx;
    }

    public ExternalCachePersistor getTxLogPersistor() {
        return new ExternalCachePersistor(this.txLogStore, this.storeTx);
    }

    public BaseTransactionConfig getBaseTransactionConfig() {
        return this.txConfig;
    }

    public IndexTransaction getIndexTransaction(String index) {
        Preconditions.checkArgument(StringUtils.isNotBlank(index));
        IndexTransaction itx = this.indexTx.get(index);
        return Preconditions.checkNotNull(itx, "Unknown index: " + index);
    }

    public void disableCache() {
        this.cacheEnabled = false;
    }

    public void enableCache() {
        this.cacheEnabled = true;
    }

    public void commitStorage() throws BackendException {
        this.storeTx.commit();
    }

    public Map<String, Throwable> commitIndexes() {
        HashMap<String, Throwable> exceptions = new HashMap<String, Throwable>(this.indexTx.size());
        for (Map.Entry<String, IndexTransaction> indexTransactionEntry : this.indexTx.entrySet()) {
            try {
                indexTransactionEntry.getValue().commit();
            }
            catch (Throwable e) {
                exceptions.put(indexTransactionEntry.getKey(), e);
            }
        }
        return exceptions;
    }

    @Override
    public void commit() throws BackendException {
        this.storeTx.commit();
        for (IndexTransaction itx : this.indexTx.values()) {
            itx.commit();
        }
    }

    @Override
    public void rollback() throws BackendException {
        Throwable exception = null;
        for (IndexTransaction itx : this.indexTx.values()) {
            try {
                itx.rollback();
            }
            catch (Throwable e) {
                exception = e;
            }
        }
        this.storeTx.rollback();
        if (exception != null) {
            if (exception instanceof BackendException) {
                throw (BackendException)exception;
            }
            throw new PermanentBackendException("Unexpected exception", exception);
        }
    }

    @Override
    public void logMutations(DataOutput out) {
        this.storeTx.logMutations(out);
        for (Map.Entry<String, IndexTransaction> itx : this.indexTx.entrySet()) {
            out.writeObjectNotNull(itx.getKey());
            itx.getValue().logMutations(out);
        }
    }

    public void mutateEdges(StaticBuffer key, List<Entry> additions, List<Entry> deletions) throws BackendException {
        this.edgeStore.mutateEntries(key, additions, deletions, this.storeTx);
    }

    public void mutateIndex(StaticBuffer key, List<Entry> additions, List<Entry> deletions) throws BackendException {
        this.indexStore.mutateEntries(key, additions, deletions, this.storeTx);
    }

    public void acquireEdgeLock(StaticBuffer key, StaticBuffer column) throws BackendException {
        this.acquiredLock = true;
        this.edgeStore.acquireLock(key, column, null, this.storeTx);
    }

    public void acquireEdgeLock(StaticBuffer key, Entry entry) throws BackendException {
        this.acquiredLock = true;
        this.edgeStore.acquireLock(key, entry.getColumnAs(StaticBuffer.STATIC_FACTORY), entry.getValueAs(StaticBuffer.STATIC_FACTORY), this.storeTx);
    }

    public void acquireIndexLock(StaticBuffer key, StaticBuffer column) throws BackendException {
        this.acquiredLock = true;
        this.indexStore.acquireLock(key, column, null, this.storeTx);
    }

    public void acquireIndexLock(StaticBuffer key, Entry entry) throws BackendException {
        this.acquiredLock = true;
        this.indexStore.acquireLock(key, entry.getColumnAs(StaticBuffer.STATIC_FACTORY), entry.getValueAs(StaticBuffer.STATIC_FACTORY), this.storeTx);
    }

    public EntryList edgeStoreQuery(final KeySliceQuery query) {
        return this.executeRead(new Callable<EntryList>(){

            @Override
            public EntryList call() throws Exception {
                return BackendTransaction.this.cacheEnabled ? BackendTransaction.this.edgeStore.getSlice(query, BackendTransaction.this.storeTx) : BackendTransaction.this.edgeStore.getSliceNoCache(query, BackendTransaction.this.storeTx);
            }

            public String toString() {
                return "EdgeStoreQuery";
            }
        });
    }

    public Map<StaticBuffer, EntryList> edgeStoreMultiQuery(final List<StaticBuffer> keys, final SliceQuery query) {
        if (this.storeFeatures.hasMultiQuery()) {
            return this.executeRead(new Callable<Map<StaticBuffer, EntryList>>(){

                @Override
                public Map<StaticBuffer, EntryList> call() throws Exception {
                    return BackendTransaction.this.cacheEnabled ? BackendTransaction.this.edgeStore.getSlice(keys, query, BackendTransaction.this.storeTx) : BackendTransaction.this.edgeStore.getSliceNoCache(keys, query, BackendTransaction.this.storeTx);
                }

                public String toString() {
                    return "MultiEdgeStoreQuery";
                }
            });
        }
        HashMap<StaticBuffer, EntryList> results = new HashMap<StaticBuffer, EntryList>(keys.size());
        if (this.threadPool == null || keys.size() < 2) {
            for (StaticBuffer key : keys) {
                results.put(key, this.edgeStoreQuery(new KeySliceQuery(key, query)));
            }
        } else {
            int i;
            CountDownLatch doneSignal = new CountDownLatch(keys.size());
            AtomicInteger failureCount = new AtomicInteger(0);
            Object[] resultArray = new EntryList[keys.size()];
            for (i = 0; i < keys.size(); ++i) {
                this.threadPool.execute(new SliceQueryRunner(new KeySliceQuery(keys.get(i), query), doneSignal, failureCount, resultArray, i));
            }
            try {
                doneSignal.await();
            }
            catch (InterruptedException e) {
                throw new JanusGraphException("Interrupted while waiting for multi-query to complete", e);
            }
            if (failureCount.get() > 0) {
                throw new JanusGraphException("Could not successfully complete multi-query. " + failureCount.get() + " individual queries failed.");
            }
            for (i = 0; i < keys.size(); ++i) {
                assert (resultArray[i] != null);
                results.put(keys.get(i), (EntryList)resultArray[i]);
            }
        }
        return results;
    }

    public KeyIterator edgeStoreKeys(final SliceQuery sliceQuery) {
        if (!this.storeFeatures.hasScan()) {
            throw new UnsupportedOperationException("The configured storage backend does not support global graph operations - use Faunus instead");
        }
        return this.executeRead(new Callable<KeyIterator>(){

            @Override
            public KeyIterator call() throws Exception {
                return BackendTransaction.this.storeFeatures.isKeyOrdered() ? BackendTransaction.this.edgeStore.getKeys(new KeyRangeQuery(EDGESTORE_MIN_KEY, EDGESTORE_MAX_KEY, sliceQuery), (StoreTransaction)BackendTransaction.this.storeTx) : BackendTransaction.this.edgeStore.getKeys(sliceQuery, (StoreTransaction)BackendTransaction.this.storeTx);
            }

            public String toString() {
                return "EdgeStoreKeys";
            }
        });
    }

    public KeyIterator edgeStoreKeys(final KeyRangeQuery range) {
        Preconditions.checkArgument(this.storeFeatures.hasOrderedScan(), "The configured storage backend does not support ordered scans");
        return this.executeRead(new Callable<KeyIterator>(){

            @Override
            public KeyIterator call() throws Exception {
                return BackendTransaction.this.edgeStore.getKeys(range, (StoreTransaction)BackendTransaction.this.storeTx);
            }

            public String toString() {
                return "EdgeStoreKeys";
            }
        });
    }

    public EntryList indexQuery(final KeySliceQuery query) {
        return this.executeRead(new Callable<EntryList>(){

            @Override
            public EntryList call() throws Exception {
                return BackendTransaction.this.cacheEnabled ? BackendTransaction.this.indexStore.getSlice(query, BackendTransaction.this.storeTx) : BackendTransaction.this.indexStore.getSliceNoCache(query, BackendTransaction.this.storeTx);
            }

            public String toString() {
                return "VertexIndexQuery";
            }
        });
    }

    public Stream<String> indexQuery(String index, final IndexQuery query) {
        final IndexTransaction indexTx = this.getIndexTransaction(index);
        return this.executeRead(new Callable<Stream<String>>(){

            @Override
            public Stream<String> call() throws Exception {
                return indexTx.queryStream(query);
            }

            public String toString() {
                return "IndexQuery";
            }
        });
    }

    public Stream<RawQuery.Result<String>> rawQuery(String index, final RawQuery query) {
        final IndexTransaction indexTx = this.getIndexTransaction(index);
        return this.executeRead(new Callable<Stream<RawQuery.Result<String>>>(){

            @Override
            public Stream<RawQuery.Result<String>> call() throws Exception {
                return indexTx.queryStream(query);
            }

            public String toString() {
                return "RawQuery";
            }
        });
    }

    public Long totals(String index, RawQuery query) {
        IndexTransaction indexTx = this.getIndexTransaction(index);
        return this.executeRead(new TotalsCallable(query, indexTx));
    }

    private <V> V executeRead(Callable<V> exe) throws JanusGraphException {
        try {
            return BackendOperation.execute(exe, this.maxReadTime);
        }
        catch (JanusGraphException e) {
            if (Thread.interrupted()) {
                throw new TraversalInterruptedException();
            }
            throw e;
        }
    }

    private class TotalsCallable
    implements Callable<Long> {
        private final RawQuery query;
        private final IndexTransaction indexTx;

        public TotalsCallable(RawQuery query, IndexTransaction indexTx) {
            this.query = query;
            this.indexTx = indexTx;
        }

        @Override
        public Long call() throws Exception {
            return this.indexTx.totals(this.query);
        }

        public String toString() {
            return "Totals";
        }
    }

    private class SliceQueryRunner
    implements Runnable {
        final KeySliceQuery kq;
        final CountDownLatch doneSignal;
        final AtomicInteger failureCount;
        final Object[] resultArray;
        final int resultPosition;

        private SliceQueryRunner(KeySliceQuery kq, CountDownLatch doneSignal, AtomicInteger failureCount, Object[] resultArray, int resultPosition) {
            this.kq = kq;
            this.doneSignal = doneSignal;
            this.failureCount = failureCount;
            this.resultArray = resultArray;
            this.resultPosition = resultPosition;
        }

        @Override
        public void run() {
            try {
                EntryList result = BackendTransaction.this.edgeStoreQuery(this.kq);
                this.resultArray[this.resultPosition] = result;
            }
            catch (Exception e) {
                this.failureCount.incrementAndGet();
                log.warn("Individual query in multi-transaction failed: ", e);
            }
            finally {
                this.doneSignal.countDown();
            }
        }
    }
}

