/*
 * Decompiled with CFR 0.152.
 */
package org.janusgraph.diskstorage.es;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterators;
import com.google.common.collect.LinkedListMultimap;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Spliterators;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.tinkerpop.shaded.jackson.databind.ObjectMapper;
import org.apache.tinkerpop.shaded.jackson.databind.ObjectWriter;
import org.apache.tinkerpop.shaded.jackson.databind.SerializationFeature;
import org.janusgraph.core.Cardinality;
import org.janusgraph.core.JanusGraphException;
import org.janusgraph.core.attribute.Cmp;
import org.janusgraph.core.attribute.Geo;
import org.janusgraph.core.attribute.Geoshape;
import org.janusgraph.core.attribute.Text;
import org.janusgraph.core.schema.Mapping;
import org.janusgraph.core.schema.Parameter;
import org.janusgraph.diskstorage.BackendException;
import org.janusgraph.diskstorage.BaseTransaction;
import org.janusgraph.diskstorage.BaseTransactionConfig;
import org.janusgraph.diskstorage.BaseTransactionConfigurable;
import org.janusgraph.diskstorage.PermanentBackendException;
import org.janusgraph.diskstorage.TemporaryBackendException;
import org.janusgraph.diskstorage.configuration.ConfigNamespace;
import org.janusgraph.diskstorage.configuration.ConfigOption;
import org.janusgraph.diskstorage.configuration.Configuration;
import org.janusgraph.diskstorage.es.ElasticSearchClient;
import org.janusgraph.diskstorage.es.ElasticSearchMutation;
import org.janusgraph.diskstorage.es.ElasticSearchRequest;
import org.janusgraph.diskstorage.es.ElasticSearchResponse;
import org.janusgraph.diskstorage.es.ElasticSearchScroll;
import org.janusgraph.diskstorage.es.ElasticSearchSetup;
import org.janusgraph.diskstorage.es.IndexMappings;
import org.janusgraph.diskstorage.es.compat.AbstractESCompat;
import org.janusgraph.diskstorage.es.compat.ES5Compat;
import org.janusgraph.diskstorage.es.compat.ES6Compat;
import org.janusgraph.diskstorage.es.rest.util.HttpAuthTypes;
import org.janusgraph.diskstorage.indexing.IndexEntry;
import org.janusgraph.diskstorage.indexing.IndexFeatures;
import org.janusgraph.diskstorage.indexing.IndexMutation;
import org.janusgraph.diskstorage.indexing.IndexProvider;
import org.janusgraph.diskstorage.indexing.IndexQuery;
import org.janusgraph.diskstorage.indexing.KeyInformation;
import org.janusgraph.diskstorage.indexing.RawQuery;
import org.janusgraph.diskstorage.util.DefaultTransaction;
import org.janusgraph.graphdb.configuration.GraphDatabaseConfiguration;
import org.janusgraph.graphdb.configuration.PreInitializeConfigOptions;
import org.janusgraph.graphdb.database.serialize.AttributeUtil;
import org.janusgraph.graphdb.query.JanusGraphPredicate;
import org.janusgraph.graphdb.query.condition.And;
import org.janusgraph.graphdb.query.condition.Condition;
import org.janusgraph.graphdb.query.condition.Not;
import org.janusgraph.graphdb.query.condition.Or;
import org.janusgraph.graphdb.query.condition.PredicateCondition;
import org.janusgraph.graphdb.types.ParameterType;
import org.locationtech.spatial4j.shape.Rectangle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PreInitializeConfigOptions
public class ElasticSearchIndex
implements IndexProvider {
    private static final Logger log = LoggerFactory.getLogger(ElasticSearchIndex.class);
    private static final String STRING_MAPPING_SUFFIX = "__STRING";
    public static final ConfigNamespace ELASTICSEARCH_NS = new ConfigNamespace(GraphDatabaseConfiguration.INDEX_NS, "elasticsearch", "Elasticsearch index configuration");
    public static final ConfigOption<String> INTERFACE = new ConfigOption<String>(ELASTICSEARCH_NS, "interface", "Interface for connecting to Elasticsearch. TRANSPORT_CLIENT and NODE were previously supported, but now are required to migrate to REST_CLIENT. See the JanusGraph upgrade instructions for more details.", ConfigOption.Type.MASKABLE, String.class, ElasticSearchSetup.REST_CLIENT.toString(), ConfigOption.disallowEmpty(String.class));
    public static final ConfigOption<String> HEALTH_REQUEST_TIMEOUT = new ConfigOption<String>(ELASTICSEARCH_NS, "health-request-timeout", "When JanusGraph initializes its ES backend, JanusGraph waits up to this duration for the ES cluster health to reach at least yellow status.  This string should be formatted as a natural number followed by the lowercase letter \"s\", e.g. 3s or 60s.", ConfigOption.Type.MASKABLE, "30s");
    public static final ConfigOption<Integer> MAX_RETRY_TIMEOUT = new ConfigOption<Integer>(ELASTICSEARCH_NS, "max-retry-timeout", "Sets the maximum timeout (in milliseconds) to honour in case of multiple retries of the same request sent using the ElasticSearch Rest Client by JanusGraph.", ConfigOption.Type.MASKABLE, Integer.class);
    public static final ConfigOption<String> BULK_REFRESH = new ConfigOption<String>(ELASTICSEARCH_NS, "bulk-refresh", "Elasticsearch bulk API refresh setting used to control when changes made by this request are made visible to search", ConfigOption.Type.MASKABLE, "false");
    public static final ConfigNamespace ES_CREATE_NS = new ConfigNamespace(ELASTICSEARCH_NS, "create", "Settings related to index creation");
    public static final ConfigOption<Long> CREATE_SLEEP = new ConfigOption<Long>(ES_CREATE_NS, "sleep", "How long to sleep, in milliseconds, between the successful completion of a (blocking) index creation request and the first use of that index.  This only applies when creating an index in ES, which typically only happens the first time JanusGraph is started on top of ES. If the index JanusGraph is configured to use already exists, then this setting has no effect.", ConfigOption.Type.MASKABLE, 200L);
    public static final ConfigNamespace ES_CREATE_EXTRAS_NS = new ConfigNamespace(ES_CREATE_NS, "ext", "Overrides for arbitrary settings applied at index creation", true);
    public static final ConfigOption<Boolean> USE_EXTERNAL_MAPPINGS = new ConfigOption<Boolean>(ES_CREATE_NS, "use-external-mappings", "Whether JanusGraph should make use of an external mapping when registering an index.", ConfigOption.Type.MASKABLE, false);
    public static final ConfigOption<Boolean> ALLOW_MAPPING_UPDATE = new ConfigOption<Boolean>(ES_CREATE_NS, "allow-mapping-update", "Whether JanusGraph should allow a mapping update when registering an index. Only applicable when " + USE_EXTERNAL_MAPPINGS.getName() + " is true.", ConfigOption.Type.MASKABLE, false);
    public static final ConfigOption<Boolean> USE_ALL_FIELD = new ConfigOption<Boolean>(ELASTICSEARCH_NS, "use-all-field", "Whether JanusGraph should add an \"all\" field mapping. When enabled field mappings will include a \"copy_to\" parameter referencing the \"all\" field. This is supported since Elasticsearch 6.x  and is required when using wildcard fields starting in Elasticsearch 6.x.", ConfigOption.Type.GLOBAL_OFFLINE, true);
    public static final ConfigOption<Boolean> USE_DEPRECATED_MULTITYPE_INDEX = new ConfigOption<Boolean>(ELASTICSEARCH_NS, "use-deprecated-multitype-index", "Whether JanusGraph should group these indices into a single Elasticsearch index (requires Elasticsearch 5.x or earlier).", ConfigOption.Type.GLOBAL_OFFLINE, false);
    public static final ConfigOption<Integer> ES_SCROLL_KEEP_ALIVE = new ConfigOption<Integer>(ELASTICSEARCH_NS, "scroll-keep-alive", "How long (in seconds) elasticsearch should keep alive the scroll context.", ConfigOption.Type.GLOBAL_OFFLINE, 60);
    public static final ConfigNamespace ES_INGEST_PIPELINES = new ConfigNamespace(ELASTICSEARCH_NS, "ingest-pipeline", "Ingest pipeline applicable to a store of an index.");
    public static final ConfigNamespace SSL_NS = new ConfigNamespace(ELASTICSEARCH_NS, "ssl", "Elasticsearch SSL configuration");
    public static final ConfigOption<Boolean> SSL_ENABLED = new ConfigOption<Boolean>(SSL_NS, "enabled", "Controls use of the SSL connection to Elasticsearch.", ConfigOption.Type.LOCAL, false);
    public static final ConfigOption<Boolean> SSL_DISABLE_HOSTNAME_VERIFICATION = new ConfigOption<Boolean>(SSL_NS, "disable-hostname-verification", "Disables the SSL hostname verification if set to true. Hostname verification is enabled by default.", ConfigOption.Type.LOCAL, false);
    public static final ConfigOption<Boolean> SSL_ALLOW_SELF_SIGNED_CERTIFICATES = new ConfigOption<Boolean>(SSL_NS, "allow-self-signed-certificates", "Controls the accepting of the self-signed SSL certificates.", ConfigOption.Type.LOCAL, false);
    public static final ConfigNamespace SSL_TRUSTSTORE_NS = new ConfigNamespace(SSL_NS, "truststore", "Configuration options for SSL Truststore.");
    public static final ConfigOption<String> SSL_TRUSTSTORE_LOCATION = new ConfigOption<String>(SSL_TRUSTSTORE_NS, "location", "Marks the location of the SSL Truststore.", ConfigOption.Type.LOCAL, "");
    public static final ConfigOption<String> SSL_TRUSTSTORE_PASSWORD = new ConfigOption<String>(SSL_TRUSTSTORE_NS, "password", "The password to access SSL Truststore.", ConfigOption.Type.LOCAL, "", Objects::nonNull);
    public static final ConfigNamespace SSL_KEYSTORE_NS = new ConfigNamespace(SSL_NS, "keystore", "Configuration options for SSL Keystore.");
    public static final ConfigOption<String> SSL_KEYSTORE_LOCATION = new ConfigOption<String>(SSL_KEYSTORE_NS, "location", "Marks the location of the SSL Keystore.", ConfigOption.Type.LOCAL, "");
    public static final ConfigOption<String> SSL_KEYSTORE_PASSWORD = new ConfigOption<String>(SSL_KEYSTORE_NS, "storepassword", "The password to access SSL Keystore.", ConfigOption.Type.LOCAL, "", Objects::nonNull);
    public static final ConfigOption<String> SSL_KEY_PASSWORD = new ConfigOption<String>(SSL_KEYSTORE_NS, "keypassword", "The password to access the key in the SSL Keystore. If the option is not present, the value of \"storepassword\" is used.", ConfigOption.Type.LOCAL, "", Objects::nonNull);
    public static final ConfigNamespace ES_HTTP_NS = new ConfigNamespace(ELASTICSEARCH_NS, "http", "Configuration options for HTTP(S) transport.");
    public static final ConfigNamespace ES_HTTP_AUTH_NS = new ConfigNamespace(ES_HTTP_NS, "auth", "Configuration options for HTTP(S) authentication.");
    public static final ConfigOption<String> ES_HTTP_AUTH_TYPE = new ConfigOption<String>(ES_HTTP_AUTH_NS, "type", "Authentication type to be used for HTTP(S) access.", ConfigOption.Type.LOCAL, HttpAuthTypes.NONE.toString());
    public static final ConfigNamespace ES_HTTP_AUTH_BASIC_NS = new ConfigNamespace(ES_HTTP_AUTH_NS, "basic", "Configuration options for HTTP(S) Basic authentication.");
    public static final ConfigOption<String> ES_HTTP_AUTH_USERNAME = new ConfigOption<String>(ES_HTTP_AUTH_BASIC_NS, "username", "Username for HTTP(S) authentication.", ConfigOption.Type.LOCAL, "");
    public static final ConfigOption<String> ES_HTTP_AUTH_PASSWORD = new ConfigOption<String>(ES_HTTP_AUTH_BASIC_NS, "password", "Password for HTTP(S) authentication.", ConfigOption.Type.LOCAL, "");
    public static final ConfigOption<String> ES_HTTP_AUTH_REALM = new ConfigOption<String>(ES_HTTP_AUTH_BASIC_NS, "realm", "Realm value for HTTP(S) authentication. If empty, any realm is accepted.", ConfigOption.Type.LOCAL, "");
    public static final ConfigNamespace ES_HTTP_AUTH_CUSTOM_NS = new ConfigNamespace(ES_HTTP_AUTH_NS, "custom", "Configuration options for custom HTTP(S) authenticator.");
    public static final ConfigOption<String> ES_HTTP_AUTHENTICATOR_CLASS = new ConfigOption<String>(ES_HTTP_AUTH_CUSTOM_NS, "authenticator-class", "Authenticator fully qualified class name.", ConfigOption.Type.LOCAL, "");
    public static final ConfigOption<String[]> ES_HTTP_AUTHENTICATOR_ARGS = new ConfigOption<String[]>(ES_HTTP_AUTH_CUSTOM_NS, "authenticator-args", "Comma-separated custom authenticator constructor arguments.", ConfigOption.Type.LOCAL, new String[0]);
    public static final int HOST_PORT_DEFAULT = 9200;
    public static final int DEFAULT_GEO_MAX_LEVELS = 20;
    public static final double DEFAULT_GEO_DIST_ERROR_PCT = 0.025;
    private static final ObjectWriter mapWriter;
    private static final Parameter[] NULL_PARAMETERS;
    private final AbstractESCompat compat;
    private final ElasticSearchClient client;
    private final String indexName;
    private final int batchSize;
    private final boolean useExternalMappings;
    private final boolean allowMappingUpdate;
    private final Map<String, Object> indexSetting;
    private final long createSleep;
    private final boolean useAllField;
    private final boolean useMultitypeIndex;
    private final Map<String, Object> ingestPipelines;

    public ElasticSearchIndex(Configuration config) throws BackendException {
        this.indexName = config.get(GraphDatabaseConfiguration.INDEX_NAME, new String[0]);
        this.useAllField = config.get(USE_ALL_FIELD, new String[0]);
        this.useExternalMappings = config.get(USE_EXTERNAL_MAPPINGS, new String[0]);
        this.allowMappingUpdate = config.get(ALLOW_MAPPING_UPDATE, new String[0]);
        this.createSleep = config.get(CREATE_SLEEP, new String[0]);
        this.ingestPipelines = config.getSubset(ES_INGEST_PIPELINES, new String[0]);
        ElasticSearchSetup.Connection c = this.interfaceConfiguration(config);
        this.client = c.getClient();
        this.batchSize = config.get(GraphDatabaseConfiguration.INDEX_MAX_RESULT_SET_SIZE, new String[0]);
        log.debug("Configured ES query nb result by query to {}", (Object)this.batchSize);
        switch (this.client.getMajorVersion()) {
            case FIVE: {
                this.compat = new ES5Compat();
                break;
            }
            case SIX: {
                this.compat = new ES6Compat();
                break;
            }
            default: {
                throw new PermanentBackendException("Unsupported Elasticsearch version: " + (Object)((Object)this.client.getMajorVersion()));
            }
        }
        try {
            this.client.clusterHealthRequest(config.get(HEALTH_REQUEST_TIMEOUT, new String[0]));
        }
        catch (IOException e) {
            throw new PermanentBackendException(e.getMessage(), e);
        }
        if (!config.has(USE_DEPRECATED_MULTITYPE_INDEX, new String[0]) && this.client.isIndex(this.indexName)) {
            this.useMultitypeIndex = true;
        } else {
            this.useMultitypeIndex = config.get(USE_DEPRECATED_MULTITYPE_INDEX, new String[0]);
            Preconditions.checkArgument(!this.useMultitypeIndex || !this.client.isAlias(this.indexName), "The key '" + USE_DEPRECATED_MULTITYPE_INDEX + "' cannot be true when existing index is split.");
            Preconditions.checkArgument(this.useMultitypeIndex || !this.client.isIndex(this.indexName), "The key '" + USE_DEPRECATED_MULTITYPE_INDEX + "' cannot be false when existing index contains multiple types.");
        }
        this.indexSetting = new HashMap<String, Object>();
        ElasticSearchSetup.applySettingsFromJanusGraphConf(this.indexSetting, config);
        this.indexSetting.put("index.max_result_window", Integer.MAX_VALUE);
    }

    private void checkForOrCreateIndex(String index) throws IOException {
        Preconditions.checkNotNull(this.client);
        Preconditions.checkNotNull(index);
        if (!this.useExternalMappings && !this.client.indexExists(index)) {
            this.client.createIndex(index, this.indexSetting);
            try {
                log.debug("Sleeping {} ms after {} index creation returned from actionGet()", (Object)this.createSleep, (Object)index);
                Thread.sleep(this.createSleep);
            }
            catch (InterruptedException e) {
                throw new JanusGraphException("Interrupted while waiting for index to settle in", e);
            }
        }
        Preconditions.checkState(this.client.indexExists(index), "Could not create index: %s", new Object[]{index});
        if (!this.useMultitypeIndex) {
            this.client.addAlias(this.indexName, index);
        }
    }

    private ElasticSearchSetup.Connection interfaceConfiguration(Configuration config) {
        ElasticSearchSetup clientMode = ConfigOption.getEnumValue(config.get(INTERFACE, new String[0]), ElasticSearchSetup.class);
        try {
            return clientMode.connect(config);
        }
        catch (IOException e) {
            throw new JanusGraphException(e);
        }
    }

    private BackendException convert(Exception esException) {
        if (esException instanceof InterruptedException) {
            return new TemporaryBackendException("Interrupted while waiting for response", esException);
        }
        return new PermanentBackendException("Unknown exception while executing index operation", esException);
    }

    private static String getDualMappingName(String key) {
        return key + STRING_MAPPING_SUFFIX;
    }

    private String getIndexStoreName(String store) {
        return this.useMultitypeIndex ? this.indexName : this.indexName + "_" + store.toLowerCase();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void register(String store, String key, KeyInformation information, BaseTransaction tx) throws BackendException {
        Class<?> dataType = information.getDataType();
        Mapping map = Mapping.getMapping(information);
        Preconditions.checkArgument(map == Mapping.DEFAULT || AttributeUtil.isString(dataType) || map == Mapping.PREFIX_TREE && AttributeUtil.isGeo(dataType), "Specified illegal mapping [%s] for data type [%s]", new Object[]{map, dataType});
        String indexStoreName = this.getIndexStoreName(store);
        if (this.useExternalMappings) {
            try {
                IndexMappings.IndexMapping mappings = this.client.getMapping(indexStoreName, store);
                if (mappings == null || !mappings.isDynamic() && !mappings.getProperties().containsKey(key)) {
                    throw new PermanentBackendException("The external mapping for index '" + indexStoreName + "' and type '" + store + "' do not have property '" + key + "'");
                }
                if (!this.allowMappingUpdate || !mappings.isDynamic()) return;
                this.pushMapping(store, key, information);
                return;
            }
            catch (IOException e) {
                throw new PermanentBackendException(e);
            }
        }
        try {
            this.checkForOrCreateIndex(indexStoreName);
        }
        catch (IOException e) {
            throw new PermanentBackendException(e);
        }
        this.pushMapping(store, key, information);
    }

    private void pushMapping(String store, String key, KeyInformation information) throws AssertionError, PermanentBackendException, BackendException {
        ImmutableMap<String, Object> mapping;
        Class<?> dataType = information.getDataType();
        Mapping map = Mapping.getMapping(information);
        HashMap<String, Map<String, Object>> properties = new HashMap<String, Map<String, Object>>();
        if (AttributeUtil.isString(dataType)) {
            if (map == Mapping.DEFAULT) {
                map = Mapping.TEXT;
            }
            log.debug("Registering string type for {} with mapping {}", (Object)key, (Object)map);
            String stringAnalyzer = ParameterType.STRING_ANALYZER.findParameter(information.getParameters(), null);
            String textAnalyzer = ParameterType.TEXT_ANALYZER.findParameter(information.getParameters(), null);
            Map<String, Object> stringMapping = stringAnalyzer == null ? this.compat.createKeywordMapping() : this.compat.createTextMapping(stringAnalyzer);
            switch (map) {
                case STRING: {
                    properties.put(key, stringMapping);
                    break;
                }
                case TEXT: {
                    properties.put(key, this.compat.createTextMapping(textAnalyzer));
                    break;
                }
                case TEXTSTRING: {
                    properties.put(key, this.compat.createTextMapping(textAnalyzer));
                    properties.put(ElasticSearchIndex.getDualMappingName(key), stringMapping);
                    break;
                }
                default: {
                    throw new AssertionError((Object)("Unexpected mapping: " + (Object)((Object)map)));
                }
            }
        } else if (dataType == Float.class) {
            log.debug("Registering float type for {}", (Object)key);
            properties.put(key, ImmutableMap.of("type", "float"));
        } else if (dataType == Double.class) {
            log.debug("Registering double type for {}", (Object)key);
            properties.put(key, ImmutableMap.of("type", "double"));
        } else if (dataType == Byte.class) {
            log.debug("Registering byte type for {}", (Object)key);
            properties.put(key, ImmutableMap.of("type", "byte"));
        } else if (dataType == Short.class) {
            log.debug("Registering short type for {}", (Object)key);
            properties.put(key, ImmutableMap.of("type", "short"));
        } else if (dataType == Integer.class) {
            log.debug("Registering integer type for {}", (Object)key);
            properties.put(key, ImmutableMap.of("type", "integer"));
        } else if (dataType == Long.class) {
            log.debug("Registering long type for {}", (Object)key);
            properties.put(key, ImmutableMap.of("type", "long"));
        } else if (dataType == Boolean.class) {
            log.debug("Registering boolean type for {}", (Object)key);
            properties.put(key, ImmutableMap.of("type", "boolean"));
        } else if (dataType == Geoshape.class) {
            switch (map) {
                case PREFIX_TREE: {
                    int maxLevels = ParameterType.INDEX_GEO_MAX_LEVELS.findParameter(information.getParameters(), 20);
                    double distErrorPct = ParameterType.INDEX_GEO_DIST_ERROR_PCT.findParameter(information.getParameters(), 0.025);
                    log.debug("Registering geo_shape type for {} with tree_levels={} and distance_error_pct={}", key, maxLevels, distErrorPct);
                    properties.put(key, ImmutableMap.of("type", "geo_shape", "tree", "quadtree", "tree_levels", maxLevels, "distance_error_pct", distErrorPct));
                    break;
                }
                default: {
                    log.debug("Registering geo_point type for {}", (Object)key);
                    properties.put(key, ImmutableMap.of("type", "geo_point"));
                    break;
                }
            }
        } else if (dataType == Date.class || dataType == Instant.class) {
            log.debug("Registering date type for {}", (Object)key);
            properties.put(key, ImmutableMap.of("type", "date"));
        } else if (dataType == UUID.class) {
            log.debug("Registering uuid type for {}", (Object)key);
            properties.put(key, this.compat.createKeywordMapping());
        }
        if (this.useAllField && this.client.getMajorVersion().getValue() >= 6) {
            properties.put("all", this.compat.createTextMapping(null));
            if (properties.containsKey(key) && dataType != Geoshape.class) {
                HashMap<String, String> mapping2 = new HashMap<String, String>((Map)properties.get(key));
                mapping2.put("copy_to", "all");
                properties.put(key, mapping2);
            }
        }
        List<Parameter> customParameters = ParameterType.getCustomParameters(information.getParameters());
        if (properties.containsKey(key) && !customParameters.isEmpty()) {
            mapping = new HashMap((Map)properties.get(key));
            customParameters.forEach(p -> mapping.put(p.key(), p.value()));
            properties.put(key, mapping);
        }
        mapping = ImmutableMap.of("properties", properties);
        try {
            this.client.createMapping(this.getIndexStoreName(store), store, mapping);
        }
        catch (Exception e) {
            throw this.convert(e);
        }
    }

    private static Mapping getStringMapping(KeyInformation information) {
        assert (AttributeUtil.isString(information.getDataType()));
        Mapping map = Mapping.getMapping(information);
        if (map == Mapping.DEFAULT) {
            map = Mapping.TEXT;
        }
        return map;
    }

    private static boolean hasDualStringMapping(KeyInformation information) {
        return AttributeUtil.isString(information.getDataType()) && ElasticSearchIndex.getStringMapping(information) == Mapping.TEXTSTRING;
    }

    public Map<String, Object> getNewDocument(List<IndexEntry> additions, KeyInformation.StoreRetriever information) throws BackendException {
        LinkedListMultimap<String, IndexEntry> unique = LinkedListMultimap.create();
        for (IndexEntry e : additions) {
            unique.put(e.field, e);
        }
        HashMap<String, Object> doc = new HashMap<String, Object>();
        for (Map.Entry add : unique.asMap().entrySet()) {
            Object[] value;
            KeyInformation keyInformation = information.get((String)add.getKey());
            switch (keyInformation.getCardinality()) {
                case SINGLE: {
                    value = ElasticSearchIndex.convertToEsType(((IndexEntry)Iterators.getLast(add.getValue().iterator())).value, Mapping.getMapping(keyInformation));
                    break;
                }
                case SET: 
                case LIST: {
                    value = add.getValue().stream().map(v -> ElasticSearchIndex.convertToEsType(v.value, Mapping.getMapping(keyInformation))).filter(v -> {
                        Preconditions.checkArgument(!(v instanceof byte[]), "Collections not supported for " + (String)add.getKey());
                        return true;
                    }).toArray();
                    break;
                }
                default: {
                    value = null;
                }
            }
            doc.put((String)add.getKey(), value);
            if (!ElasticSearchIndex.hasDualStringMapping(information.get((String)add.getKey())) || keyInformation.getDataType() != String.class) continue;
            doc.put(ElasticSearchIndex.getDualMappingName((String)add.getKey()), value);
        }
        return doc;
    }

    private static Object convertToEsType(Object value, Mapping mapping) {
        if (value instanceof Number) {
            if (AttributeUtil.isWholeNumber((Number)value)) {
                return ((Number)value).longValue();
            }
            return ((Number)value).doubleValue();
        }
        if (AttributeUtil.isString(value)) {
            return value;
        }
        if (value instanceof Geoshape) {
            return ElasticSearchIndex.convertGeoshape((Geoshape)value, mapping);
        }
        if (value instanceof Date) {
            return value;
        }
        if (value instanceof Instant) {
            return Date.from((Instant)value);
        }
        if (value instanceof Boolean) {
            return value;
        }
        if (value instanceof UUID) {
            return value.toString();
        }
        throw new IllegalArgumentException("Unsupported type: " + value.getClass() + " (value: " + value + ")");
    }

    private static Object convertGeoshape(Geoshape geoshape, Mapping mapping) {
        if (geoshape.getType() == Geoshape.Type.POINT && Mapping.PREFIX_TREE != mapping) {
            Geoshape.Point p = geoshape.getPoint();
            return new double[]{p.getLongitude(), p.getLatitude()};
        }
        if (geoshape.getType() == Geoshape.Type.BOX) {
            Rectangle box = geoshape.getShape().getBoundingBox();
            HashMap<String, Object> map = new HashMap<String, Object>();
            map.put("type", "envelope");
            map.put("coordinates", new double[][]{{box.getMinX(), box.getMaxY()}, {box.getMaxX(), box.getMinY()}});
            return map;
        }
        if (geoshape.getType() == Geoshape.Type.CIRCLE) {
            try {
                Map<String, Object> map = geoshape.toMap();
                map.put("radius", map.get("radius") + (String)((Map)map.remove("properties")).get("radius_units"));
                return map;
            }
            catch (IOException e) {
                throw new IllegalArgumentException("Invalid geoshape: " + geoshape, e);
            }
        }
        try {
            return geoshape.toMap();
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Invalid geoshape: " + geoshape, e);
        }
    }

    @Override
    public void mutate(Map<String, Map<String, IndexMutation>> mutations, KeyInformation.IndexRetriever information, BaseTransaction tx) throws BackendException {
        ArrayList<ElasticSearchMutation> requests = new ArrayList<ElasticSearchMutation>();
        try {
            for (Map.Entry<String, Map<String, IndexMutation>> stores : mutations.entrySet()) {
                ArrayList<ElasticSearchMutation> requestByStore = new ArrayList<ElasticSearchMutation>();
                String storeName = stores.getKey();
                String indexStoreName = this.getIndexStoreName(storeName);
                for (Map.Entry<String, IndexMutation> entry : stores.getValue().entrySet()) {
                    Map<String, Object> doc;
                    String documentId = entry.getKey();
                    IndexMutation mutation = entry.getValue();
                    assert (mutation.isConsolidated());
                    Preconditions.checkArgument(!mutation.isNew() || !mutation.isDeleted());
                    Preconditions.checkArgument(!mutation.isNew() || !mutation.hasDeletions());
                    Preconditions.checkArgument(!mutation.isDeleted() || !mutation.hasAdditions());
                    if (mutation.hasDeletions()) {
                        if (mutation.isDeleted()) {
                            log.trace("Deleting entire document {}", (Object)documentId);
                            requestByStore.add(ElasticSearchMutation.createDeleteRequest(indexStoreName, storeName, documentId));
                        } else {
                            String script = this.getDeletionScript(information, storeName, mutation);
                            ImmutableMap doc2 = this.compat.prepareScript(script).build();
                            requestByStore.add(ElasticSearchMutation.createUpdateRequest(indexStoreName, storeName, documentId, doc2));
                            log.trace("Adding script {}", (Object)script);
                        }
                    }
                    if (!mutation.hasAdditions()) continue;
                    if (mutation.isNew()) {
                        log.trace("Adding entire document {}", (Object)documentId);
                        Map<String, Object> source = this.getNewDocument(mutation.getAdditions(), information.get(storeName));
                        requestByStore.add(ElasticSearchMutation.createIndexRequest(indexStoreName, storeName, documentId, source));
                        continue;
                    }
                    Map<String, Object> upsert = !mutation.hasDeletions() ? this.getNewDocument(mutation.getAdditions(), information.get(storeName)) : null;
                    String inline = this.getAdditionScript(information, storeName, mutation);
                    if (!inline.isEmpty()) {
                        ImmutableMap.Builder builder = this.compat.prepareScript(inline);
                        requestByStore.add(ElasticSearchMutation.createUpdateRequest(indexStoreName, storeName, documentId, builder, upsert));
                        log.trace("Adding script {}", (Object)inline);
                    }
                    if ((doc = this.getAdditionDoc(information, storeName, mutation)).isEmpty()) continue;
                    ImmutableMap.Builder<String, Map<String, Object>> builder = ImmutableMap.builder().put("doc", doc);
                    requestByStore.add(ElasticSearchMutation.createUpdateRequest(indexStoreName, storeName, documentId, builder, upsert));
                    log.trace("Adding update {}", (Object)doc);
                }
                if (!requestByStore.isEmpty() && this.ingestPipelines.containsKey(storeName)) {
                    this.client.bulkRequest(requestByStore, String.valueOf(this.ingestPipelines.get(storeName)));
                    continue;
                }
                if (requestByStore.isEmpty()) continue;
                requests.addAll(requestByStore);
            }
            if (!requests.isEmpty()) {
                this.client.bulkRequest(requests, null);
            }
        }
        catch (Exception e) {
            log.error("Failed to execute bulk Elasticsearch mutation", e);
            throw this.convert(e);
        }
    }

    private String getDeletionScript(KeyInformation.IndexRetriever information, String storeName, IndexMutation mutation) throws PermanentBackendException {
        StringBuilder script = new StringBuilder();
        String INDEX_NAME = "index";
        int i = 0;
        for (IndexEntry deletion : mutation.getDeletions()) {
            KeyInformation keyInformation = information.get(storeName).get(deletion.field);
            switch (keyInformation.getCardinality()) {
                case SINGLE: {
                    script.append("ctx._source.remove(\"").append(deletion.field).append("\");");
                    if (!ElasticSearchIndex.hasDualStringMapping(information.get(storeName, deletion.field))) break;
                    script.append("ctx._source.remove(\"").append(ElasticSearchIndex.getDualMappingName(deletion.field)).append("\");");
                    break;
                }
                case SET: 
                case LIST: {
                    String jsValue = ElasticSearchIndex.convertToJsType(deletion.value, this.compat.scriptLang(), Mapping.getMapping(keyInformation));
                    String index = "index" + i++;
                    script.append("def ").append(index).append(" = ctx._source[\"").append(deletion.field).append("\"].indexOf(").append(jsValue).append("); ctx._source[\"").append(deletion.field).append("\"].remove(").append(index).append(");");
                    if (!ElasticSearchIndex.hasDualStringMapping(information.get(storeName, deletion.field))) break;
                    index = "index" + i++;
                    script.append("def ").append(index).append(" = ctx._source[\"").append(ElasticSearchIndex.getDualMappingName(deletion.field)).append("\"].indexOf(").append(jsValue).append("); ctx._source[\"").append(ElasticSearchIndex.getDualMappingName(deletion.field)).append("\"].remove(").append(index).append(");");
                }
            }
        }
        return script.toString();
    }

    private String getAdditionScript(KeyInformation.IndexRetriever information, String storeName, IndexMutation mutation) throws PermanentBackendException {
        StringBuilder script = new StringBuilder();
        for (IndexEntry e : mutation.getAdditions()) {
            KeyInformation keyInformation = information.get(storeName).get(e.field);
            Cardinality cardinality = keyInformation.getCardinality();
            if (cardinality != Cardinality.SET && cardinality != Cardinality.LIST) continue;
            String value = ElasticSearchIndex.convertToJsType(e.value, this.compat.scriptLang(), Mapping.getMapping(keyInformation));
            this.appendAdditionScript(script, value, e.field, cardinality == Cardinality.SET);
            if (!ElasticSearchIndex.hasDualStringMapping(keyInformation)) continue;
            this.appendAdditionScript(script, value, ElasticSearchIndex.getDualMappingName(e.field), cardinality == Cardinality.SET);
        }
        return script.toString();
    }

    private void appendAdditionScript(StringBuilder script, String value, String fieldName, Boolean distinct) {
        script.append("if (ctx._source[\"").append(fieldName).append("\"] == null) ctx._source[\"").append(fieldName).append("\"] = [];");
        if (distinct.booleanValue()) {
            script.append("if (ctx._source[\"").append(fieldName).append("\"].indexOf(").append(value).append(") == -1) ");
        }
        script.append("ctx._source[\"").append(fieldName).append("\"].add(").append(value).append(");");
    }

    private Map<String, Object> getAdditionDoc(KeyInformation.IndexRetriever information, String store, IndexMutation mutation) throws PermanentBackendException {
        HashMap<String, Object> doc = new HashMap<String, Object>();
        for (IndexEntry e : mutation.getAdditions()) {
            KeyInformation keyInformation = information.get(store).get(e.field);
            if (keyInformation.getCardinality() != Cardinality.SINGLE) continue;
            doc.put(e.field, ElasticSearchIndex.convertToEsType(e.value, Mapping.getMapping(keyInformation)));
            if (!ElasticSearchIndex.hasDualStringMapping(keyInformation)) continue;
            doc.put(ElasticSearchIndex.getDualMappingName(e.field), ElasticSearchIndex.convertToEsType(e.value, Mapping.getMapping(keyInformation)));
        }
        return doc;
    }

    private static String convertToJsType(Object value, String scriptLang, Mapping mapping) throws PermanentBackendException {
        String esValue;
        try {
            esValue = mapWriter.writeValueAsString(ElasticSearchIndex.convertToEsType(value, mapping));
        }
        catch (IOException e) {
            throw new PermanentBackendException("Could not write json");
        }
        return scriptLang.equals("groovy") ? esValue.replace("$", "\\$") : esValue;
    }

    @Override
    public void restore(Map<String, Map<String, List<IndexEntry>>> documents, KeyInformation.IndexRetriever information, BaseTransaction tx) throws BackendException {
        ArrayList<ElasticSearchMutation> requests = new ArrayList<ElasticSearchMutation>();
        try {
            for (Map.Entry<String, Map<String, List<IndexEntry>>> stores : documents.entrySet()) {
                ArrayList<ElasticSearchMutation> requestByStore = new ArrayList<ElasticSearchMutation>();
                String store = stores.getKey();
                String indexStoreName = this.getIndexStoreName(store);
                for (Map.Entry<String, List<IndexEntry>> entry : stores.getValue().entrySet()) {
                    String docID = entry.getKey();
                    List<IndexEntry> content = entry.getValue();
                    if (content == null || content.size() == 0) {
                        if (log.isTraceEnabled()) {
                            log.trace("Deleting entire document {}", (Object)docID);
                        }
                        requestByStore.add(ElasticSearchMutation.createDeleteRequest(indexStoreName, store, docID));
                        continue;
                    }
                    if (log.isTraceEnabled()) {
                        log.trace("Adding entire document {}", (Object)docID);
                    }
                    Map<String, Object> source = this.getNewDocument(content, information.get(store));
                    requestByStore.add(ElasticSearchMutation.createIndexRequest(indexStoreName, store, docID, source));
                }
                if (!requestByStore.isEmpty() && this.ingestPipelines.containsKey(store)) {
                    this.client.bulkRequest(requestByStore, String.valueOf(this.ingestPipelines.get(store)));
                    continue;
                }
                if (requestByStore.isEmpty()) continue;
                requests.addAll(requestByStore);
            }
            if (!requests.isEmpty()) {
                this.client.bulkRequest(requests, null);
            }
        }
        catch (Exception e) {
            throw this.convert(e);
        }
    }

    public Map<String, Object> getFilter(Condition<?> condition, KeyInformation.StoreRetriever information) {
        if (condition instanceof PredicateCondition) {
            PredicateCondition atom = (PredicateCondition)condition;
            Object value = atom.getValue();
            String key = (String)atom.getKey();
            JanusGraphPredicate predicate = atom.getPredicate();
            if (value instanceof Number) {
                Preconditions.checkArgument(predicate instanceof Cmp, "Relation not supported on numeric types: " + predicate);
                Cmp numRel = (Cmp)predicate;
                switch (numRel) {
                    case EQUAL: {
                        return this.compat.term(key, value);
                    }
                    case NOT_EQUAL: {
                        return this.compat.boolMustNot(this.compat.term(key, value));
                    }
                    case LESS_THAN: {
                        return this.compat.lt(key, value);
                    }
                    case LESS_THAN_EQUAL: {
                        return this.compat.lte(key, value);
                    }
                    case GREATER_THAN: {
                        return this.compat.gt(key, value);
                    }
                    case GREATER_THAN_EQUAL: {
                        return this.compat.gte(key, value);
                    }
                }
                throw new IllegalArgumentException("Unexpected relation: " + numRel);
            }
            if (value instanceof String) {
                Mapping mapping = ElasticSearchIndex.getStringMapping(information.get(key));
                if (mapping == Mapping.TEXT && !Text.HAS_CONTAINS.contains(predicate) && !(predicate instanceof Cmp)) {
                    throw new IllegalArgumentException("Text mapped string values only support CONTAINS and Compare queries and not: " + predicate);
                }
                if (mapping == Mapping.STRING && Text.HAS_CONTAINS.contains(predicate)) {
                    throw new IllegalArgumentException("String mapped string values do not support CONTAINS queries: " + predicate);
                }
                String fieldName = mapping == Mapping.TEXTSTRING && !Text.HAS_CONTAINS.contains(predicate) && (!(predicate instanceof Cmp) || predicate == Cmp.EQUAL) ? ElasticSearchIndex.getDualMappingName(key) : key;
                if (predicate == Text.CONTAINS || predicate == Cmp.EQUAL) {
                    return this.compat.match(fieldName, value);
                }
                if (predicate == Text.CONTAINS_PREFIX) {
                    if (!ParameterType.TEXT_ANALYZER.hasParameter(information.get(key).getParameters())) {
                        value = ((String)value).toLowerCase();
                    }
                    return this.compat.prefix(fieldName, value);
                }
                if (predicate == Text.CONTAINS_REGEX) {
                    if (!ParameterType.TEXT_ANALYZER.hasParameter(information.get(key).getParameters())) {
                        value = ((String)value).toLowerCase();
                    }
                    return this.compat.regexp(fieldName, value);
                }
                if (predicate == Text.PREFIX) {
                    return this.compat.prefix(fieldName, value);
                }
                if (predicate == Text.REGEX) {
                    return this.compat.regexp(fieldName, value);
                }
                if (predicate == Cmp.NOT_EQUAL) {
                    return this.compat.boolMustNot(this.compat.match(fieldName, value));
                }
                if (predicate == Text.FUZZY || predicate == Text.CONTAINS_FUZZY) {
                    return this.compat.fuzzyMatch(fieldName, value);
                }
                if (predicate == Cmp.LESS_THAN) {
                    return this.compat.lt(fieldName, value);
                }
                if (predicate == Cmp.LESS_THAN_EQUAL) {
                    return this.compat.lte(fieldName, value);
                }
                if (predicate == Cmp.GREATER_THAN) {
                    return this.compat.gt(fieldName, value);
                }
                if (predicate == Cmp.GREATER_THAN_EQUAL) {
                    return this.compat.gte(fieldName, value);
                }
                throw new IllegalArgumentException("Predicate is not supported for string value: " + predicate);
            }
            if (value instanceof Geoshape && Mapping.getMapping(information.get(key)) == Mapping.DEFAULT) {
                Map<String, Object> query;
                Geoshape shape = (Geoshape)value;
                Preconditions.checkArgument(predicate instanceof Geo && predicate != Geo.CONTAINS, "Relation not supported on geopoint types: " + predicate);
                switch (shape.getType()) {
                    case CIRCLE: {
                        Geoshape.Point center = shape.getPoint();
                        query = this.compat.geoDistance(key, center.getLatitude(), center.getLongitude(), shape.getRadius());
                        break;
                    }
                    case BOX: {
                        Geoshape.Point southwest = shape.getPoint(0);
                        Geoshape.Point northeast = shape.getPoint(1);
                        query = this.compat.geoBoundingBox(key, southwest.getLatitude(), southwest.getLongitude(), northeast.getLatitude(), northeast.getLongitude());
                        break;
                    }
                    case POLYGON: {
                        List<List<Double>> points = IntStream.range(0, shape.size()).mapToObj(i -> ImmutableList.of(Double.valueOf(shape.getPoint(i).getLongitude()), Double.valueOf(shape.getPoint(i).getLatitude()))).collect(Collectors.toList());
                        query = this.compat.geoPolygon(key, points);
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Unsupported or invalid search shape type for geopoint: " + (Object)((Object)shape.getType()));
                    }
                }
                return predicate == Geo.DISJOINT ? this.compat.boolMustNot(query) : query;
            }
            if (value instanceof Geoshape) {
                ImmutableMap<String, Object> geo;
                Preconditions.checkArgument(predicate instanceof Geo, "Relation not supported on geoshape types: " + predicate);
                Geoshape shape = (Geoshape)value;
                switch (shape.getType()) {
                    case CIRCLE: {
                        Geoshape.Point center = shape.getPoint();
                        geo = ImmutableMap.of("type", "circle", "coordinates", ImmutableList.of(Double.valueOf(center.getLongitude()), Double.valueOf(center.getLatitude())), "radius", shape.getRadius() + "km");
                        break;
                    }
                    case BOX: {
                        Geoshape.Point southwest = shape.getPoint(0);
                        Geoshape.Point northeast = shape.getPoint(1);
                        geo = ImmutableMap.of("type", "envelope", "coordinates", ImmutableList.of(ImmutableList.of(Double.valueOf(southwest.getLongitude()), Double.valueOf(northeast.getLatitude())), ImmutableList.of(Double.valueOf(northeast.getLongitude()), Double.valueOf(southwest.getLatitude()))));
                        break;
                    }
                    case LINE: {
                        List lineCoords = IntStream.range(0, shape.size()).mapToObj(i -> ImmutableList.of(Double.valueOf(shape.getPoint(i).getLongitude()), Double.valueOf(shape.getPoint(i).getLatitude()))).collect(Collectors.toList());
                        geo = ImmutableMap.of("type", "linestring", "coordinates", lineCoords);
                        break;
                    }
                    case POLYGON: {
                        List polyCoords = IntStream.range(0, shape.size()).mapToObj(i -> ImmutableList.of(Double.valueOf(shape.getPoint(i).getLongitude()), Double.valueOf(shape.getPoint(i).getLatitude()))).collect(Collectors.toList());
                        geo = ImmutableMap.of("type", "polygon", "coordinates", ImmutableList.of(polyCoords));
                        break;
                    }
                    case POINT: {
                        geo = ImmutableMap.of("type", "point", "coordinates", ImmutableList.of(Double.valueOf(shape.getPoint().getLongitude()), Double.valueOf(shape.getPoint().getLatitude())));
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Unsupported or invalid search shape type: " + (Object)((Object)shape.getType()));
                    }
                }
                return this.compat.geoShape(key, geo, (Geo)predicate);
            }
            if (value instanceof Date || value instanceof Instant) {
                Preconditions.checkArgument(predicate instanceof Cmp, "Relation not supported on date types: " + predicate);
                Cmp numRel = (Cmp)predicate;
                if (value instanceof Instant) {
                    value = Date.from((Instant)value);
                }
                switch (numRel) {
                    case EQUAL: {
                        return this.compat.term(key, value);
                    }
                    case NOT_EQUAL: {
                        return this.compat.boolMustNot(this.compat.term(key, value));
                    }
                    case LESS_THAN: {
                        return this.compat.lt(key, value);
                    }
                    case LESS_THAN_EQUAL: {
                        return this.compat.lte(key, value);
                    }
                    case GREATER_THAN: {
                        return this.compat.gt(key, value);
                    }
                    case GREATER_THAN_EQUAL: {
                        return this.compat.gte(key, value);
                    }
                }
                throw new IllegalArgumentException("Unexpected relation: " + numRel);
            }
            if (value instanceof Boolean) {
                Cmp numRel = (Cmp)predicate;
                switch (numRel) {
                    case EQUAL: {
                        return this.compat.term(key, value);
                    }
                    case NOT_EQUAL: {
                        return this.compat.boolMustNot(this.compat.term(key, value));
                    }
                }
                throw new IllegalArgumentException("Boolean types only support EQUAL or NOT_EQUAL");
            }
            if (value instanceof UUID) {
                if (predicate == Cmp.EQUAL) {
                    return this.compat.term(key, value);
                }
                if (predicate == Cmp.NOT_EQUAL) {
                    return this.compat.boolMustNot(this.compat.term(key, value));
                }
                throw new IllegalArgumentException("Only equal or not equal is supported for UUIDs: " + predicate);
            }
            throw new IllegalArgumentException("Unsupported type: " + value);
        }
        if (condition instanceof Not) {
            return this.compat.boolMustNot(this.getFilter(((Not)condition).getChild(), information));
        }
        if (condition instanceof And) {
            List<Map<String, Object>> queries = StreamSupport.stream(condition.getChildren().spliterator(), false).map(c -> this.getFilter((Condition<?>)c, information)).collect(Collectors.toList());
            return this.compat.boolMust(queries);
        }
        if (condition instanceof Or) {
            List<Map<String, Object>> queries = StreamSupport.stream(condition.getChildren().spliterator(), false).map(c -> this.getFilter((Condition<?>)c, information)).collect(Collectors.toList());
            return this.compat.boolShould(queries);
        }
        throw new IllegalArgumentException("Invalid condition: " + condition);
    }

    @Override
    public Stream<String> query(IndexQuery query, KeyInformation.IndexRetriever informations, BaseTransaction tx) throws BackendException {
        ElasticSearchRequest sr = new ElasticSearchRequest();
        Map<String, Object> esQuery = this.getFilter(query.getCondition(), informations.get(query.getStore()));
        sr.setQuery(this.compat.prepareQuery(esQuery));
        if (!query.getOrder().isEmpty()) {
            this.addOrderToQuery(informations, sr, query.getOrder(), query.getStore());
        }
        sr.setFrom(0);
        if (query.hasLimit()) {
            sr.setSize(Math.min(query.getLimit(), this.batchSize));
        } else {
            sr.setSize(this.batchSize);
        }
        try {
            String indexStoreName = this.getIndexStoreName(query.getStore());
            String indexType = this.useMultitypeIndex ? query.getStore() : null;
            ElasticSearchResponse response = this.client.search(indexStoreName, indexType, this.compat.createRequestBody(sr, NULL_PARAMETERS), sr.getSize() >= this.batchSize);
            log.debug("First Executed query [{}] in {} ms", (Object)query.getCondition(), (Object)response.getTook());
            ElasticSearchScroll resultIterator = new ElasticSearchScroll(this.client, response, sr.getSize());
            Stream<RawQuery.Result<String>> toReturn = StreamSupport.stream(Spliterators.spliteratorUnknownSize(resultIterator, 16), false);
            return (query.hasLimit() ? toReturn.limit(query.getLimit()) : toReturn).map(RawQuery.Result::getResult);
        }
        catch (IOException | UncheckedIOException e) {
            throw new PermanentBackendException(e);
        }
    }

    private String convertToEsDataType(Class<?> dataType, Mapping mapping) {
        if (String.class.isAssignableFrom(dataType)) {
            return "string";
        }
        if (Integer.class.isAssignableFrom(dataType)) {
            return "integer";
        }
        if (Long.class.isAssignableFrom(dataType)) {
            return "long";
        }
        if (Float.class.isAssignableFrom(dataType)) {
            return "float";
        }
        if (Double.class.isAssignableFrom(dataType)) {
            return "double";
        }
        if (Boolean.class.isAssignableFrom(dataType)) {
            return "boolean";
        }
        if (Date.class.isAssignableFrom(dataType)) {
            return "date";
        }
        if (Instant.class.isAssignableFrom(dataType)) {
            return "date";
        }
        if (Geoshape.class.isAssignableFrom(dataType)) {
            return mapping == Mapping.DEFAULT ? "geo_point" : "geo_shape";
        }
        return null;
    }

    private ElasticSearchResponse runCommonQuery(RawQuery query, KeyInformation.IndexRetriever informations, BaseTransaction tx, int size, boolean useScroll) throws BackendException {
        ElasticSearchRequest sr = new ElasticSearchRequest();
        sr.setQuery(this.compat.queryString(query.getQuery()));
        if (!query.getOrders().isEmpty()) {
            this.addOrderToQuery(informations, sr, query.getOrders(), query.getStore());
        }
        sr.setFrom(0);
        sr.setSize(size);
        try {
            return this.client.search(this.getIndexStoreName(query.getStore()), this.useMultitypeIndex ? query.getStore() : null, this.compat.createRequestBody(sr, query.getParameters()), useScroll);
        }
        catch (IOException | UncheckedIOException e) {
            throw new PermanentBackendException(e);
        }
    }

    private void addOrderToQuery(KeyInformation.IndexRetriever informations, ElasticSearchRequest sr, List<IndexQuery.OrderEntry> orders, String store) {
        for (IndexQuery.OrderEntry orderEntry : orders) {
            String order = orderEntry.getOrder().name();
            KeyInformation information = informations.get(store).get(orderEntry.getKey());
            Mapping mapping = Mapping.getMapping(information);
            Class<?> datatype = orderEntry.getDatatype();
            sr.addSort(orderEntry.getKey(), order.toLowerCase(), this.convertToEsDataType(datatype, mapping));
        }
    }

    @Override
    public Stream<RawQuery.Result<String>> query(RawQuery query, KeyInformation.IndexRetriever information, BaseTransaction tx) throws BackendException {
        int size = query.hasLimit() ? Math.min(query.getLimit() + query.getOffset(), this.batchSize) : this.batchSize;
        ElasticSearchResponse response = this.runCommonQuery(query, information, tx, size, size >= this.batchSize);
        log.debug("First Executed query [{}] in {} ms", (Object)query.getQuery(), (Object)response.getTook());
        ElasticSearchScroll resultIterator = new ElasticSearchScroll(this.client, response, size);
        Stream<RawQuery.Result<String>> toReturn = StreamSupport.stream(Spliterators.spliteratorUnknownSize(resultIterator, 16), false).skip(query.getOffset());
        return query.hasLimit() ? toReturn.limit(query.getLimit()) : toReturn;
    }

    @Override
    public Long totals(RawQuery query, KeyInformation.IndexRetriever information, BaseTransaction tx) throws BackendException {
        int size = query.hasLimit() ? Math.min(query.getLimit() + query.getOffset(), this.batchSize) : this.batchSize;
        ElasticSearchResponse response = this.runCommonQuery(query, information, tx, size, false);
        log.debug("Executed query [{}] in {} ms", (Object)query.getQuery(), (Object)response.getTook());
        return response.getTotal();
    }

    @Override
    public boolean supports(KeyInformation information, JanusGraphPredicate janusgraphPredicate) {
        Class<?> dataType = information.getDataType();
        Mapping mapping = Mapping.getMapping(information);
        if (!(mapping == Mapping.DEFAULT || AttributeUtil.isString(dataType) || mapping == Mapping.PREFIX_TREE && AttributeUtil.isGeo(dataType))) {
            return false;
        }
        if (Number.class.isAssignableFrom(dataType)) {
            return janusgraphPredicate instanceof Cmp;
        }
        if (dataType == Geoshape.class) {
            switch (mapping) {
                case DEFAULT: {
                    return janusgraphPredicate instanceof Geo && janusgraphPredicate != Geo.CONTAINS;
                }
                case PREFIX_TREE: {
                    return janusgraphPredicate instanceof Geo;
                }
            }
        } else if (AttributeUtil.isString(dataType)) {
            switch (mapping) {
                case TEXT: 
                case DEFAULT: {
                    return janusgraphPredicate == Text.CONTAINS || janusgraphPredicate == Text.CONTAINS_PREFIX || janusgraphPredicate == Text.CONTAINS_REGEX || janusgraphPredicate == Text.CONTAINS_FUZZY;
                }
                case STRING: {
                    return janusgraphPredicate instanceof Cmp || janusgraphPredicate == Text.REGEX || janusgraphPredicate == Text.PREFIX || janusgraphPredicate == Text.FUZZY;
                }
                case TEXTSTRING: {
                    return janusgraphPredicate instanceof Text || janusgraphPredicate instanceof Cmp;
                }
            }
        } else {
            if (dataType == Date.class || dataType == Instant.class) {
                return janusgraphPredicate instanceof Cmp;
            }
            if (dataType == Boolean.class) {
                return janusgraphPredicate == Cmp.EQUAL || janusgraphPredicate == Cmp.NOT_EQUAL;
            }
            if (dataType == UUID.class) {
                return janusgraphPredicate == Cmp.EQUAL || janusgraphPredicate == Cmp.NOT_EQUAL;
            }
        }
        return false;
    }

    @Override
    public boolean supports(KeyInformation information) {
        Class<?> dataType = information.getDataType();
        Mapping mapping = Mapping.getMapping(information);
        if (Number.class.isAssignableFrom(dataType) || dataType == Date.class || dataType == Instant.class || dataType == Boolean.class || dataType == UUID.class) {
            return mapping == Mapping.DEFAULT;
        }
        if (AttributeUtil.isString(dataType)) {
            return mapping == Mapping.DEFAULT || mapping == Mapping.STRING || mapping == Mapping.TEXT || mapping == Mapping.TEXTSTRING;
        }
        if (AttributeUtil.isGeo(dataType)) {
            return mapping == Mapping.DEFAULT || mapping == Mapping.PREFIX_TREE;
        }
        return false;
    }

    @Override
    public String mapKey2Field(String key, KeyInformation information) {
        IndexProvider.checkKeyValidity(key);
        return key.replace(' ', '\u2022');
    }

    @Override
    public IndexFeatures getFeatures() {
        return this.compat.getIndexFeatures();
    }

    @Override
    public BaseTransactionConfigurable beginTransaction(BaseTransactionConfig config) throws BackendException {
        return new DefaultTransaction(config);
    }

    @Override
    public void close() throws BackendException {
        try {
            this.client.close();
        }
        catch (IOException e) {
            throw new PermanentBackendException(e);
        }
    }

    @Override
    public void clearStorage() throws BackendException {
        try {
            this.client.deleteIndex(this.indexName);
        }
        catch (Exception e) {
            throw new PermanentBackendException("Could not delete index " + this.indexName, e);
        }
        finally {
            this.close();
        }
    }

    @Override
    public boolean exists() throws BackendException {
        try {
            return this.client.indexExists(this.indexName);
        }
        catch (IOException e) {
            throw new PermanentBackendException("Could not check if index " + this.indexName + " exists", e);
        }
    }

    static {
        ObjectMapper mapper = new ObjectMapper();
        mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        mapWriter = mapper.writerWithView(Map.class);
        NULL_PARAMETERS = null;
    }
}

