/* SPDX-License-Identifier: Apache-2.0 */
/* Copyright Contributors to the ODPi Egeria project. */
package org.odpi.openmetadata.integrationservices.lineage.ffdc;

import org.odpi.openmetadata.frameworks.auditlog.messagesets.AuditLogMessageDefinition;
import org.odpi.openmetadata.frameworks.auditlog.messagesets.AuditLogMessageSet;
import org.odpi.openmetadata.repositoryservices.auditlog.OMRSAuditLogRecordSeverity;


/**
 * The LineageIntegratorAuditCode is used to define the message content for the OMRS Audit Log.
 *
 * The 5 fields in the enum are:
 * <ul>
 *     <li>Log Message Id - to uniquely identify the message</li>
 *     <li>Severity - is this an event, decision, action, error or exception</li>
 *     <li>Log Message Text - includes placeholder to allow additional values to be captured</li>
 *     <li>Additional Information - further parameters and data relating to the audit message (optional)</li>
 *     <li>SystemAction - describes the result of the situation</li>
 *     <li>UserAction - describes how a user should correct the situation</li>
 * </ul>
 */
public enum LineageIntegratorAuditCode implements AuditLogMessageSet
{
    CONTEXT_INITIALIZING("OMIS-LINEAGE-INTEGRATOR-0001",
                        OMRSAuditLogRecordSeverity.STARTUP,
                        "The lineage integrator context manager is being initialized for calls to server {0} on platform {1}",
                        "The Lineage Integrator OMIS is initializing its context manager.",
                        "Verify that the start up sequence goes on to initialize the context for each connector configured for this service."),

    CONNECTOR_CONTEXT_INITIALIZING("OMIS-LINEAGE-INTEGRATOR-0002",
                                   OMRSAuditLogRecordSeverity.STARTUP,
                                   "Creating context for integration connector {0} ({1}) connecting to third party technology {2} with permitted synchronization of {3} and service options of {4}",
                                   "A new context is created for an integration connector.  This acts as a client to the open metadata repositories " +
                                           "enabling the integration connector to synchronize open metadata with the third party technology's metadata",
                                   "Verify that this connector is being started with the correct configuration."),


    PERMITTED_SYNCHRONIZATION("OMIS-LINEAGE-INTEGRATOR-0003",
             OMRSAuditLogRecordSeverity.STARTUP,
             "The context for connector {0} has its permitted synchronization set to {1}",
             "The context is set up to ensure that the connector can only issue requests that support the permitted synchronization.  " +
                     "If the connector issues requests that are not permitted it is returned UserNotAuthorizedExceptions.",
             "Check that this permitted synchronized value is as expected.  If it is not," +
                     "change the configuration for this connector and restart the integration daemon."),

    ;


    AuditLogMessageDefinition messageDefinition;



    /**
     * The constructor for LineageIntegratorAuditCode expects to be passed one of the enumeration rows defined in
     * LineageIntegratorAuditCode above.   For example:
     *
     *     LineageIntegratorAuditCode   auditCode = LineageIntegratorAuditCode.SERVER_SHUTDOWN;
     *
     * This will expand out to the 4 parameters shown below.
     *
     * @param messageId - unique Id for the message
     * @param severity - severity of the message
     * @param message - text for the message
     * @param systemAction - description of the action taken by the system when the condition happened
     * @param userAction - instructions for resolving the situation, if any
     */
    LineageIntegratorAuditCode(String                     messageId,
                               OMRSAuditLogRecordSeverity severity,
                               String                     message,
                               String                     systemAction,
                               String                     userAction)
    {
        messageDefinition = new AuditLogMessageDefinition(messageId,
                                                          severity,
                                                          message,
                                                          systemAction,
                                                          userAction);
    }


    /**
     * Retrieve a message definition object for logging.  This method is used when there are no message inserts.
     *
     * @return message definition object.
     */
    @Override
    public AuditLogMessageDefinition getMessageDefinition()
    {
        return messageDefinition;
    }


    /**
     * Retrieve a message definition object for logging.  This method is used when there are values to be inserted into the message.
     *
     * @param params array of parameters (all strings).  They are inserted into the message according to the numbering in the message text.
     * @return message definition object.
     */
    @Override
    public AuditLogMessageDefinition getMessageDefinition(String ...params)
    {
        messageDefinition.setMessageParameters(params);
        return messageDefinition;
    }


    /**
     * toString() JSON-style
     *
     * @return string description
     */
    @Override
    public String toString()
    {
        return "LineageIntegratorAuditCode{" +
                "messageDefinition=" + messageDefinition +
                '}';
    }
}
