/*
 *  SPDX-License-Identifier: Apache-2.0
 *  Copyright Contributors to the ODPi Egeria project.
 */
package org.odpi.openmetadata.accessservices.securityofficer.server.admin.services;

import org.odpi.openmetadata.accessservices.securityofficer.api.ffdc.errorcode.SecurityOfficerErrorCode;
import org.odpi.openmetadata.accessservices.securityofficer.api.ffdc.exceptions.NewInstanceException;
import org.odpi.openmetadata.accessservices.securityofficer.api.ffdc.exceptions.PropertyServerException;
import org.odpi.openmetadata.accessservices.securityofficer.server.admin.publisher.SecurityOfficerPublisher;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.OMRSMetadataCollection;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.repositoryconnector.OMRSRepositoryConnector;

/**
 * SecurityOfficerServicesInstance caches references to OMRS objects for a specific server.
 * It is also responsible for registering itself in the instance map.
 */
public class SecurityOfficerServicesInstance {
    private OMRSRepositoryConnector repositoryConnector;
    private OMRSMetadataCollection metadataCollection;
    private SecurityOfficerPublisher securityOfficerPublisher;
    private String serverName;


    /**
     * Set up the local repository connector that will service the REST Calls.
     *
     * @param repositoryConnector      link to the repository responsible for servicing the REST calls.
     * @param securityOfficerPublisher
     * @throws NewInstanceException a problem occurred during initialization
     */
    public SecurityOfficerServicesInstance(OMRSRepositoryConnector repositoryConnector, SecurityOfficerPublisher securityOfficerPublisher) throws NewInstanceException {
        final String methodName = "new ServiceInstance";

        if (securityOfficerPublisher != null) {
            this.securityOfficerPublisher = securityOfficerPublisher;
        }

        if (repositoryConnector != null) {
            try {
                this.repositoryConnector = repositoryConnector;
                this.serverName = repositoryConnector.getServerName();
                this.metadataCollection = repositoryConnector.getMetadataCollection();

                SecurityOfficerServicesInstanceMap.setNewInstanceForJVM(serverName, this);
            } catch (Exception error) {
                SecurityOfficerErrorCode errorCode = SecurityOfficerErrorCode.OMRS_NOT_INITIALIZED;
                String errorMessage = errorCode.getErrorMessageId() + errorCode.getFormattedErrorMessage(methodName);

                throw new NewInstanceException(errorCode.getHttpErrorCode(),
                        this.getClass().getName(),
                        methodName,
                        errorMessage,
                        errorCode.getSystemAction(),
                        errorCode.getUserAction());

            }
        } else {
            SecurityOfficerErrorCode errorCode = SecurityOfficerErrorCode.OMRS_NOT_INITIALIZED;
            String errorMessage = errorCode.getErrorMessageId() + errorCode.getFormattedErrorMessage(methodName);

            throw new NewInstanceException(errorCode.getHttpErrorCode(),
                    this.getClass().getName(),
                    methodName,
                    errorMessage,
                    errorCode.getSystemAction(),
                    errorCode.getUserAction());

        }
    }


    /**
     * Return the server name.
     *
     * @return serverName name of the server for this instance
     * @throws NewInstanceException a problem occurred during initialization
     */
    public String getServerName() throws NewInstanceException {
        final String methodName = "getServerName";

        if (serverName != null) {
            return serverName;
        } else {
            SecurityOfficerErrorCode errorCode = SecurityOfficerErrorCode.OMRS_NOT_AVAILABLE;
            String errorMessage = errorCode.getErrorMessageId() + errorCode.getFormattedErrorMessage(methodName);

            throw new NewInstanceException(errorCode.getHttpErrorCode(),
                    this.getClass().getName(),
                    methodName,
                    errorMessage,
                    errorCode.getSystemAction(),
                    errorCode.getUserAction());
        }
    }

    /**
     * Return the repository connector for this server.
     *
     * @return OMRSRepositoryConnector object
     * @throws PropertyServerException the instance has not been initialized successfully
     */
    OMRSRepositoryConnector getRepositoryConnector() throws PropertyServerException {
        final String methodName = "getRepositoryConnector";

        if ((repositoryConnector != null) && (metadataCollection != null) && (repositoryConnector.isActive())) {
            return repositoryConnector;
        } else {
            SecurityOfficerErrorCode errorCode = SecurityOfficerErrorCode.OMRS_NOT_AVAILABLE;
            String errorMessage = errorCode.getErrorMessageId() + errorCode.getFormattedErrorMessage(methodName);

            throw new PropertyServerException(errorCode.getHttpErrorCode(),
                    this.getClass().getName(),
                    methodName,
                    errorMessage,
                    errorCode.getSystemAction(),
                    errorCode.getUserAction());
        }
    }

    /**
     * Return the Security Officer Output Publisher
     *
     * @return SecurityOfficerPublisher - this instance is able to publish events of the Security Officer Output Topic
     */
    SecurityOfficerPublisher getSecurityOfficerPublisher() {
        return securityOfficerPublisher;
    }

    /**
     * Unregister this instance from the instance map.
     */
    public void shutdown() {
        SecurityOfficerServicesInstanceMap.removeInstanceForJVM(serverName);
    }
}

