/* SPDX-License-Identifier: Apache-2.0 */
/* Copyright Contributors to the ODPi Egeria project. */
package org.odpi.openmetadata.accessservices.subjectarea.client.nodes.categories;

import org.odpi.openmetadata.accessservices.subjectarea.client.SubjectAreaRestClient;
import org.odpi.openmetadata.accessservices.subjectarea.client.nodes.AbstractSubjectAreaNode;
import org.odpi.openmetadata.accessservices.subjectarea.client.nodes.SubjectAreaNodeClient;
import org.odpi.openmetadata.accessservices.subjectarea.properties.objects.category.Category;
import org.odpi.openmetadata.accessservices.subjectarea.properties.objects.common.FindRequest;
import org.odpi.openmetadata.accessservices.subjectarea.properties.objects.term.Term;
import org.odpi.openmetadata.accessservices.subjectarea.responses.SubjectAreaOMASAPIResponse;
import org.odpi.openmetadata.accessservices.subjectarea.utils.QueryBuilder;
import org.odpi.openmetadata.accessservices.subjectarea.utils.QueryParams;
import org.odpi.openmetadata.commonservices.ffdc.rest.GenericResponse;
import org.odpi.openmetadata.frameworks.connectors.ffdc.InvalidParameterException;
import org.odpi.openmetadata.frameworks.connectors.ffdc.PropertyServerException;
import org.odpi.openmetadata.frameworks.connectors.ffdc.UserNotAuthorizedException;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.core.ResolvableType;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@SubjectAreaNodeClient
public class SubjectAreaCategoryClient<C extends Category> extends AbstractSubjectAreaNode<C> {
    public SubjectAreaCategoryClient(SubjectAreaRestClient client)
    {
        super(client, SUBJECT_AREA_BASE_URL + "categories");
    }
    /**
     * Get the Category's children categories
     *
     * @param userId      unique identifier for requesting user, under which the request is performed.
     * @param guid        unique identifier of the Category
     * @param findRequest specification of the search
     * @return list of Category children
     * @throws PropertyServerException    something went wrong with the REST call stack.
     * @throws UserNotAuthorizedException the requesting user is not authorized to issue this request.
     * @throws InvalidParameterException  one of the parameters is null or invalid.
     */

    public List<Category> getCategoryChildren(String userId, String guid, FindRequest findRequest) throws InvalidParameterException, PropertyServerException, UserNotAuthorizedException {
      return getCategoryChildren(userId,guid, findRequest, false, true);
    }
    /**
     * Get the Category's children categories
     *
     * @param userId      unique identifier for requesting user, under which the request is performed.
     * @param guid        unique identifier of the Category
     * @param findRequest specification of the search
     * @param exactValue should the result match exactly?
     * @param ignoreCase should the match be case insensitive?
     * @return list of Category children
     * @throws PropertyServerException    something went wrong with the REST call stack.
     * @throws UserNotAuthorizedException the requesting user is not authorized to issue this request.
     * @throws InvalidParameterException  one of the parameters is null or invalid.
     */

    public List<Category> getCategoryChildren(String userId, String guid, FindRequest findRequest, boolean exactValue, boolean ignoreCase) throws InvalidParameterException, PropertyServerException, UserNotAuthorizedException {
        final String urnTemplate = BASE_URL + "/%s/categories";
        final String methodInfo = getMethodInfo(" getCategoryChildren");
        QueryParams queryParams = new QueryParams()
                .setExactValue(exactValue)
                .setIgnoreCase(ignoreCase);
        QueryBuilder query = client.createFindQuery(methodInfo, findRequest, queryParams);

        String urlTemplate = urnTemplate + query.toString();
        ResolvableType resolvableType = ResolvableType.forClassWithGenerics(SubjectAreaOMASAPIResponse.class, Category.class);
        ParameterizedTypeReference<GenericResponse<Category>> type = ParameterizedTypeReference.forType(resolvableType.getType());
        GenericResponse<Category> response = client.getByIdRESTCall(userId ,guid, methodInfo, type, urlTemplate);
        return response.results();
    }
    /**
     * Get the terms owned by this Category
     *
     * @param userId      unique identifier for requesting user, under which the request is performed.
     * @param guid        unique identifier of the object to which the found objects should relate.
     * @param findRequest information object for find calls. This include pageSize to limit the number of elements returned.
     * @return list of Terms
     * @throws PropertyServerException    something went wrong with the REST call stack.
     * @throws UserNotAuthorizedException the requesting user is not authorized to issue this request.
     * @throws InvalidParameterException  one of the parameters is null or invalid.
     */

    public List<Term> getTerms(String userId, String guid, FindRequest findRequest) throws InvalidParameterException, PropertyServerException, UserNotAuthorizedException {

        return getTerms(userId, guid, findRequest, false,true);
    }
    /**
     * Get the terms owned by this Category
     *
     * @param userId      unique identifier for requesting user, under which the request is performed.
     * @param guid        unique identifier of the object to which the found objects should relate.
     * @param findRequest information object for find calls. This include pageSize to limit the number of elements returned.
     * @param exactValue should the result match exactly?
     * @param ignoreCase should the match be case insensitive?
     * @return list of Terms
     * @throws PropertyServerException    something went wrong with the REST call stack.
     * @throws UserNotAuthorizedException the requesting user is not authorized to issue this request.
     * @throws InvalidParameterException  one of the parameters is null or invalid.
     */

    public List<Term> getTerms(String userId, String guid, FindRequest findRequest, boolean exactValue, boolean ignoreCase) throws InvalidParameterException, PropertyServerException, UserNotAuthorizedException {
        final String urnTemplate = BASE_URL + "/%s/terms";
        final String methodInfo = getMethodInfo("getTerms");
        QueryParams queryParams = new QueryParams()
                .setExactValue(exactValue)
                .setIgnoreCase(ignoreCase);
        QueryBuilder query = client.createFindQuery(methodInfo, findRequest, queryParams);
        String urlTemplate = urnTemplate + query.toString();

        ResolvableType resolvableType = ResolvableType.forClassWithGenerics(SubjectAreaOMASAPIResponse.class, Term.class);
        ParameterizedTypeReference<GenericResponse<Term>> type = ParameterizedTypeReference.forType(resolvableType.getType());
        GenericResponse<Term> response = client.getByIdRESTCall(userId ,guid, methodInfo, type, urlTemplate);
        return response.results();
    }
}
