/* SPDX-License-Identifier: Apache-2.0 */
/* Copyright Contributors to the ODPi Egeria project. */
package org.odpi.openmetadata.accessservices.subjectarea.server.mappers.relationships;

import org.odpi.openmetadata.accessservices.subjectarea.properties.enums.TermRelationshipStatus;
import org.odpi.openmetadata.accessservices.subjectarea.properties.objects.graph.Line;
import org.odpi.openmetadata.accessservices.subjectarea.properties.relationships.TermTYPEDBYRelationship;
import org.odpi.openmetadata.accessservices.subjectarea.utilities.OMRSAPIHelper;
import org.odpi.openmetadata.accessservices.subjectarea.utilities.SubjectAreaUtils;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.properties.instances.EnumPropertyValue;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.properties.instances.InstanceProperties;
import org.odpi.openmetadata.repositoryservices.connectors.stores.metadatacollectionstore.properties.instances.Relationship;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Mapping methods to map between the termTYPEDBYRelationship and the equivalent omrs Relationship.
 */
public class TermTYPEDBYRelationshipMapper extends LineMapper
{
    private static final Logger log = LoggerFactory.getLogger( TermTYPEDBYRelationshipMapper.class);
    private static final String className = TermTYPEDBYRelationshipMapper.class.getName();
    public static final String TERM_TYPEDBY_RELATIONSHIP = "TermTYPEDBYRelationship";

    public TermTYPEDBYRelationshipMapper(OMRSAPIHelper omrsapiHelper) {
        super(omrsapiHelper);
    }

    /**
     * Map the supplied Line to omrs InstanceProperties.
     * @param line supplied line
     * @param instanceProperties equivalent instance properties to the Line
     */
    @Override
    protected void mapLineToInstanceProperties(Line line, InstanceProperties instanceProperties) {
        TermTYPEDBYRelationship termTYPEDBYRelationship = (TermTYPEDBYRelationship)line;
        if (termTYPEDBYRelationship.getDescription()!=null) {
            SubjectAreaUtils.setStringPropertyInInstanceProperties(instanceProperties, termTYPEDBYRelationship.getDescription(), "description");
        }
        if (termTYPEDBYRelationship.getSteward()!=null) {
            SubjectAreaUtils.setStringPropertyInInstanceProperties(instanceProperties, termTYPEDBYRelationship.getSteward(), "steward");
        }
        if (termTYPEDBYRelationship.getSource()!=null) {
            SubjectAreaUtils.setStringPropertyInInstanceProperties(instanceProperties, termTYPEDBYRelationship.getSource(), "source");
        }
        if (termTYPEDBYRelationship.getStatus()!=null) {
            EnumPropertyValue enumPropertyValue = new EnumPropertyValue();
            enumPropertyValue.setOrdinal(termTYPEDBYRelationship.getStatus().getOrdinal());
            instanceProperties.setProperty("status",enumPropertyValue);
        }
    }
    /**
     * Map a primitive omrs property to the termTYPEDBYRelationship object.
     * @param line the glossary to be updated
     * @param propertyName the omrs property name
     * @param value the omrs primitive property value
     * @return true if the propertyName was recognised and mapped to the Line, otherwise false
     */
    @Override
    protected boolean mapPrimitiveToLine(Line line, String propertyName, Object value) {
        String stringValue = (String) value;
       TermTYPEDBYRelationship termTYPEDBYRelationship = (TermTYPEDBYRelationship) line;
        boolean foundProperty = false;
        if (propertyName.equals("description")) {
            termTYPEDBYRelationship.setDescription(stringValue);
            foundProperty = true;
        }
        if (propertyName.equals("steward")) {
            termTYPEDBYRelationship.setSteward(stringValue);
            foundProperty = true;
        }
        if (propertyName.equals("source")) {
            termTYPEDBYRelationship.setSource(stringValue);
            foundProperty = true;
        }
        return foundProperty;
    }
    @Override
    protected boolean mapEnumToLine(Line line, String propertyName, EnumPropertyValue enumPropertyValue)
    {
        TermTYPEDBYRelationship termTYPEDBYRelationship = (TermTYPEDBYRelationship) line;
        boolean foundProperty = false;
        if (propertyName.equals("status")) {
            TermRelationshipStatus status = TermRelationshipStatus.valueOf(enumPropertyValue.getSymbolicName());
            termTYPEDBYRelationship.setStatus(status);
            foundProperty = true;
        }
        return foundProperty;
    }

    /**
     * Get proxy1 guid.
     * The proxy has omrs type GlossaryTerm
     * @param line line
     * @return guid for entity proxy 1
     */
    @Override
    protected String getProxy1Guid(Line line)
    {
        TermTYPEDBYRelationship termTYPEDBYRelationship = (TermTYPEDBYRelationship) line;
        return termTYPEDBYRelationship.getAttributeGuid();
    }

    /**
     * Get proxy2 guid
     * The proxy has omrs type GlossaryTerm
     * @param line for this Line
     * @return guid for entity proxy 2
     */
    @Override
    protected String getProxy2Guid(Line line)
    {
        TermTYPEDBYRelationship termTYPEDBYRelationship = (TermTYPEDBYRelationship) line;
        return termTYPEDBYRelationship.getTypeGuid();
    }

    /**
     * Get the relationship type def guid.
     * @param relationship the relationship associated with the typedef whose guid is returned.
     * @return guid of the typedef
     */
    @Override
    protected String getRelationshipTypeDefGuid(Relationship relationship)
    {
        return repositoryHelper.getTypeDefByName(omrsapiHelper.getServiceName(), TERM_TYPEDBY_RELATIONSHIP).getGUID();
    }
    @Override
    protected String getTypeName() {
        return  TERM_TYPEDBY_RELATIONSHIP;
    }
    @Override
    protected Line getLineInstance() {
        return new TermTYPEDBYRelationship();
    }
    @Override
    protected void setEnd1GuidInLine(Line line, String guid){
        TermTYPEDBYRelationship termTYPEDBYRelationship = (TermTYPEDBYRelationship)line;
        termTYPEDBYRelationship.setAttributeGuid(guid);
    }
    @Override
    protected void setEnd2GuidInLine(Line line, String guid) {
        TermTYPEDBYRelationship termTYPEDBYRelationship = (TermTYPEDBYRelationship)line;
        termTYPEDBYRelationship.setTypeGuid(guid);
    }
}
