/**
@license
Copyright (c) 2020 Vaadin Ltd.
This program is available under Apache License Version 2.0, available at https://vaadin.com/license/
*/
const DirHelper = {
  /**
  * Get the scroll type in the current browser view.
  *
  * @return {String} the scroll type. Possible values are `default|reverse|negative`
  */
  detectScrollType: function () {
    if (!window.document.body) {
      return 'indeterminate';
    }

    const dummy = window.document.createElement('div');
    dummy.textContent = 'ABCD';
    dummy.dir = 'rtl';
    dummy.style.fontSize = '14px';
    dummy.style.width = '4px';
    dummy.style.height = '1px';
    dummy.style.position = 'absolute';
    dummy.style.top = '-1000px';
    dummy.style.overflow = 'scroll';
    document.body.appendChild(dummy);
    let cachedType = 'reverse';

    if (dummy.scrollLeft > 0) {
      cachedType = 'default';
    } else {
      dummy.scrollLeft = 1;

      if (dummy.scrollLeft === 0) {
        cachedType = 'negative';
      }
    }

    document.body.removeChild(dummy);
    return cachedType;
  },

  /**
  * Get the scrollLeft value of the element relative to the direction
  *
  * @param {String} scrollType type of the scroll detected with `detectScrollType`
  * @param {String} direction current direction of the element
  * @param {Node} element
  * @return {Number} the scrollLeft value.
  */
  getNormalizedScrollLeft: function (scrollType, direction, element) {
    const {
      scrollLeft
    } = element;

    if (direction !== 'rtl' || !scrollType) {
      return scrollLeft;
    }

    switch (scrollType) {
      case 'negative':
        return element.scrollWidth - element.clientWidth + scrollLeft;

      case 'reverse':
        return element.scrollWidth - element.clientWidth - scrollLeft;
    }

    return scrollLeft;
  },

  /**
  * Set the scrollLeft value of the element relative to the direction
  *
  * @param {String} scrollType type of the scroll detected with `detectScrollType`
  * @param {String} direction current direction of the element
  * @param {Node} element
  * @param {Number} scrollLeft the scrollLeft value to be set
  */
  setNormalizedScrollLeft: function (scrollType, direction, element, scrollLeft) {
    if (direction !== 'rtl' || !scrollType) {
      element.scrollLeft = scrollLeft;
      return;
    }

    switch (scrollType) {
      case 'negative':
        element.scrollLeft = element.clientWidth - element.scrollWidth + scrollLeft;
        break;

      case 'reverse':
        element.scrollLeft = element.scrollWidth - element.clientWidth - scrollLeft;
        break;

      default:
        element.scrollLeft = scrollLeft;
        break;
    }
  }
};
export { DirHelper };