/**
@license
Copyright (c) 2016 The Polymer Project Authors. All rights reserved.
This code may only be used under the BSD style license found at
http://polymer.github.io/LICENSE.txt The complete set of authors may be found at
http://polymer.github.io/AUTHORS.txt The complete set of contributors may be
found at http://polymer.github.io/CONTRIBUTORS.txt Code distributed by Google as
part of the polymer project is also subject to an additional IP rights grant
found at http://polymer.github.io/PATENTS.txt
*/

/**
app-grid is a helper class useful for creating responsive, fluid grid layouts
using custom properties. Because custom properties can be defined inside a
`@media` rule, you can customize the grid layout for different responsive
breakpoints.

Example:

Import `app-grid-style.html` and include `app-grid-style` in the style of an
element's definition. Then, add the class `app-grid` to a container such as `ul`
or `div`:

```html
<template>
  <style include="app-grid-style">
    :host {
      --app-grid-columns: 3;
      --app-grid-item-height: 100px;
    }

    ul {
      padding: 0;
      list-style: none;
    }

    .item {
      background-color: white;
    }
  </style>
  <ul class="app-grid">
    <li class="item">1</li>
    <li class="item">2</li>
    <li class="item">3</li>
  </ul>
</template>
```
In the example above, the grid  will take 3 columns per row.

### Expandible items

In many cases, it's useful to expand an item more than 1 column. To achieve this
type of layout, you can specify the number of columns the item should expand to
by setting the custom property
`--app-grid-expandible-item-columns`. To indicate which item should expand,
apply the mixin
`--app-grid-expandible-item` to a rule with a selector to the item. For example:

<pre><code>
&lt;template>
  &lt;style include="app-grid-style">
    :host {
      --app-grid-columns: 3;
      --app-grid-item-height: 100px;
      --app-grid-expandible-item-columns: 3;
    }

    /* Only the first item should expand *\/
    .item:first-child {
      &#64;apply --app-grid-expandible-item;
    }
  &lt;/style>
&lt;/template>
</code></pre>

### Preserving the aspect ratio

When the size of a grid item should preserve the aspect ratio, you can add the
`has-aspect-ratio` attribute to the element with the class `.app-grid`. Now,
every item element becomes a wrapper around the item content. For example:

```html
<template>
  <style include="app-grid-style">
    :host {
      --app-grid-columns: 3;
      /* 50% the width of the item is equivalent to 2:1 aspect ratio*\/
      --app-grid-item-height: 50%;
    }

    .item {
      background-color: white;
    }
  </style>
  <ul class="app-grid" has-aspect-ratio>
    <li class="item">
      <div>item 1</div>
    </li>
    <li class="item">
      <div>item 2</div>
    </li>
    <li class="item">
      <div>item 3</div>
    </li>
  </ul>
</template>
```

### Styling

Custom property                               | Description                                                | Default
----------------------------------------------|------------------------------------------------------------|------------------
`--app-grid-columns`                          | The number of columns per row.                             | 1
`--app-grid-gutter`                           | The space between two items.                               | 0px
`--app-grid-item-height`                      | The height of the items.                                   | auto
`--app-grid-expandible-item-columns`          | The number of columns an expandible item should expand to. | 1

@pseudoElement app-grid
@demo app-grid/demo/index.html
*/
import "../../polymer/polymer-legacy.js";
const $_documentContainer = document.createElement('template');
$_documentContainer.setAttribute('style', 'display: none;');
$_documentContainer.innerHTML = `<dom-module id="app-grid-style">
  <template>
    <style>
      :host {
        /**
         * The width for the expandible item is:
         * ((100% - subPixelAdjustment) / columns * itemColumns - gutter
         *
         * - subPixelAdjustment: 0.1px (Required for IE 11)
         * - gutter: var(--app-grid-gutter)
         * - columns: var(--app-grid-columns)
         * - itemColumn: var(--app-grid-expandible-item-columns)
         */
        --app-grid-expandible-item: {
          -webkit-flex-basis: calc((100% - 0.1px) / var(--app-grid-columns, 1) * var(--app-grid-expandible-item-columns, 1) - var(--app-grid-gutter, 0px)) !important;
          flex-basis: calc((100% - 0.1px) / var(--app-grid-columns, 1) * var(--app-grid-expandible-item-columns, 1) - var(--app-grid-gutter, 0px)) !important;
          max-width: calc((100% - 0.1px) / var(--app-grid-columns, 1) * var(--app-grid-expandible-item-columns, 1) - var(--app-grid-gutter, 0px)) !important;
        };
      }

      .app-grid {
        display: -ms-flexbox;
        display: -webkit-flex;
        display: flex;

        -ms-flex-direction: row;
        -webkit-flex-direction: row;
        flex-direction: row;

        -ms-flex-wrap: wrap;
        -webkit-flex-wrap: wrap;
        flex-wrap: wrap;

        padding-top: var(--app-grid-gutter, 0px);
        padding-left: var(--app-grid-gutter, 0px);
        box-sizing: border-box;
      }

      .app-grid > * {
        /* Required for IE 10 */
        -ms-flex: 1 1 100%;
        -webkit-flex: 1;
        flex: 1;

        /* The width for an item is: (100% - subPixelAdjustment - gutter * columns) / columns */
        -webkit-flex-basis: calc((100% - 0.1px - (var(--app-grid-gutter, 0px) * var(--app-grid-columns, 1))) / var(--app-grid-columns, 1));
        flex-basis: calc((100% - 0.1px - (var(--app-grid-gutter, 0px) * var(--app-grid-columns, 1))) / var(--app-grid-columns, 1));

        max-width: calc((100% - 0.1px - (var(--app-grid-gutter, 0px) * var(--app-grid-columns, 1))) / var(--app-grid-columns, 1));
        margin-bottom: var(--app-grid-gutter, 0px);
        margin-right: var(--app-grid-gutter, 0px);
        height: var(--app-grid-item-height);
        box-sizing: border-box;
      }

      .app-grid[has-aspect-ratio] > * {
        position: relative;
      }

      .app-grid[has-aspect-ratio] > *::before {
        display: block;
        content: "";
        padding-top: var(--app-grid-item-height, 100%);
      }

      .app-grid[has-aspect-ratio] > * > * {
        position: absolute;
        top: 0;
        right: 0;
        bottom: 0;
        left: 0;
      }
    </style>
  </template>
</dom-module>`;
document.head.appendChild($_documentContainer.content);