/* -*-             c-basic-offset: 4; indent-tabs-mode: nil; -*-  //------100-columns-wide------>|*/
// for license please see accompanying LICENSE.txt file (available also at http://www.xmlpull.org/)
package org.xmlpull.infoset;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Vector;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;
import org.xmlpull.v1.XmlSerializer;

//import org.xmlpull.v1.builder.impl.XmlInfosetBuilderImpl;

//  builder.setUseAllTokens(true)  //comments, PIs, doctype
//setPrettyPrint

/**
 * This is abstract factory that is pluggable to allow buildings of <a
 * href="http://www.w3.org/TR/xml-infoset/">XML Information Set </a>
 * representation with <b>any</b> XML parsing API.
 *
 * @version $Revision: 1.2 $
 * @author <a href="http://www.extreme.indiana.edu/~aslom/">Aleksander Slominski
 *         </a>
 */
public abstract class XmlInfosetBuilder {
    public static final String FEATURE_READ_ONLY = "org.xmlpull.infoset.feature.read.only";
    public static final String FEATURE_BUILD_COMMENTS = "org.xmlpull.infoset.feature.comments";
    public static final String FEATURE_BUILD_PROCESSING_INSTRUCTIONS = "org.xmlpull.infoset.feature.pis";
    public static final String FEATURE_WRAP_CHARACTERS = "org.xmlpull.infoset.feature.wrap.chars";
    public static final String FEATURE_VIEWS = "org.xmlpull.infoset.feature.views";
    public static final String FEATURE_BUILD_DOM2 = "org.xmlpull.infoset.feature.dom2";
    
    
    /** used as default class to server as context class in newInstance() */
    final static Class referenceContextClass = XmlInfosetBuilder.class;
    
    /**
     * Name of the system property that should containing a comma separated list
     * of XML Infoset builder implementation class
     * names (value: "org.xmlpull.infoset.XmlInfosetBuilder").
     */
    
    public static final String PROPERTY_NAME = "org.xmlpull.infoset.XmlInfosetBuilder";
    
    private static final String RESOURCE_NAME = "/META-INF/services/" + PROPERTY_NAME;
    
    
    /**
     * Create a new instance of a XML Infoset builder that can be used to create
     * XML Infoset artifacts (see other methods in this factory for
     * description).
     *
     * @return a new instance of a XmlInfosetBuilder, as returned by newInstance
     *         (null, null);
     */
    public static XmlInfosetBuilder newInstance() throws XmlBuilderException {
        return newInstance(null, null);
    }
    
    public static XmlInfosetBuilder newInstance(String classNames, Class context)
    throws XmlBuilderException {
        
        if(context == null) {
            // NOTE: make sure context uses the same class loader as API classes
            // this is the best we can do without having access to context
            // classloader in J2ME
            // if API is in the same classloader as implementation then this
            // will work
            context = referenceContextClass;
        }
        
        String classNamesLocation = null;
        
        if(classNames == null || classNames.length() == 0
           || "DEFAULT".equals(classNames)) {
            String resourceContent = "<MISSING>";
            try {
                InputStream is = context.getResourceAsStream(RESOURCE_NAME);
                
                if(is == null) {
                    throw new XmlBuilderException(
                        "resource not found: "
                        + RESOURCE_NAME
                        + " make sure that builder implementing XML Infoset API is available");
                }
                final StringBuffer sb = new StringBuffer();
                
                while(true) {
                    final int ch = is.read();
                    if(ch < 0) {
                        break;
                    } else if(ch > ' ') {
                        sb.append((char) ch);
                    }
                }
                is.close();
                
                resourceContent = sb.toString();
            } catch(Exception e) {
                throw new XmlBuilderException(
                    "Could not create XML Infoset of " + RESOURCE_NAME
                    + " (that  had '" + resourceContent + "'):" + e, e);
            }
            classNamesLocation = "resource " + RESOURCE_NAME
            + " that contained '" + resourceContent + "'";
            classNames = resourceContent;
        } else {
            classNamesLocation = "parameter classNames to newInstance() that contained '"
            + classNames + "'";
        }
        
        XmlInfosetBuilder factory = null;
        int pos = 0;
        
        while(pos < classNames.length()) {
            int cut = classNames.indexOf(',', pos);
            
            if(cut == -1)
                cut = classNames.length();
            final String name = classNames.substring(pos, cut);
            
            Class candidate = null;
            Object instance = null;
            
            try {
                candidate = Class.forName(name);
                // necessary because of J2ME .class issue
                instance = candidate.newInstance();
            } catch(Exception e) {
            }
            
            if(candidate != null) {
                boolean recognized = false;
                if(instance instanceof XmlInfosetBuilder) {
                    if(factory == null) {
                        factory = (XmlInfosetBuilder) instance;
                    }
                    recognized = true;
                }
                if(!recognized) {
                    throw new XmlBuilderException("incompatible class: "
                                                  + name);
                }
            }
            pos = cut + 1;
        }
        
        if(factory == null) {
            throw new XmlBuilderException("no XML Infoset builder could be created out of "
                                          + classNamesLocation);
        }
        return factory;
    }
    
    
    /**
     * Request that feature identified by name is set to given value.
     * It is dafe to assume that unset features are by default false.
     *
     * @param    featureName         a  String
     * @param    value               a  boolean
     *
     * @exception   XmlBuilderException
     *
     */
    public abstract void setFeature(String featureName, boolean value) throws XmlBuilderException;
    
    /**
     * Create a new document.
     */
    public XmlDocument newDocument() throws XmlBuilderException {
        return newDocument(null, null, null);
    }
    
    /**
     * Create a new document with given XML prolog.
     *
     * @param version
     *            a String
     * @param standalone
     *            a Boolean
     * @param characterEncoding
     *            a String
     *
     * @return a XmlDocument
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlDocument newDocument(String version, Boolean standalone,
                                            String characterEncoding) throws XmlBuilderException;
    
    /**
     * Create XML fragment that is not associated with any document.
     *
     * @param elementName
     *            name of element
     *
     * @return a XmlElement
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlElement newFragment(String elementName)
    throws XmlBuilderException;
    
    /**
     * Create XML fragment that is not associated with any document.
     *
     * @param elementNamespace
     *            namespace of element
     * @param elementName
     *            name of element
     *
     * @return a XmlElement
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlElement newFragment(String elementNamespace,
                                           String elementName) throws XmlBuilderException;
    
    /**
     * Create XML fragment that is not associated with any document.
     *
     * @param elementNamespace
     *            a XmlNamespace
     * @param elementName
     *            a String
     *
     * @return a XmlElement
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlElement newFragment(XmlNamespace elementNamespace,
                                           String elementName) throws XmlBuilderException;
    
    /**
     * Create a new namespace that is not associated with any XML document.
     *
     * @param namespaceName
     *            a String
     *
     * @return a XmlNamespace
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlNamespace newNamespace(String namespaceName)
    throws XmlBuilderException;
    
    /**
     * Create a new namespace that is not associated with any XML document.
     *
     * @param prefix
     *            a String
     * @param namespaceName
     *            a String
     *
     * @return a XmlNamespace
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlNamespace newNamespace(String prefix,
                                              String namespaceName) throws XmlBuilderException;
    
    // --- parsing
    
    // public abstract XmlElement newFragment(String elementNamespace, String
    // elementName, XmlNamespace[] context);
    // public abstract XmlElement parse(XmlPullParser sourceForNode,boolean
    // addAllNamespaces);
    
    /**
     * Parse input stream to create XML document.
     *
     * @param is
     *            an InputStream
     *
     * @return a XmlDocument
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlDocument parseInputStream(InputStream is)
    throws XmlBuilderException;
    
    /**
     * Parse input stream to create XML document using specified encoding.
     *
     * @param is
     *            an InputStream
     * @param encoding
     *            a String
     *
     * @return a XmlDocument
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlDocument parseInputStream(InputStream is, String encoding)
    throws XmlBuilderException;
    
    /**
     * Parse reader to create XML document.
     *
     * @param reader
     *            a Reader
     *
     * @return a XmlDocument
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlDocument parseReader(Reader reader)
    throws XmlBuilderException;
    
    /**
     * Parse XML contained in String to create XML document.
     *
     * @param xmlAsString
     *            a String containing XML document to parse
     *
     * @return a XmlDocument
     *
     * @exception XmlBuilderException
     *
     */
    public XmlDocument parseString(String xmlAsString)
    throws XmlBuilderException {
        return parseReader(new StringReader(xmlAsString));
    }
    
    /**
     * Parse input from URL location to create XML document.
     *
     * @param locationUrl
     *            a String
     *
     * @return a XmlDocument
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlDocument parseLocation(String locationUrl)
    throws XmlBuilderException;
    
    /**
     * Parse input stream to create XML fragment.
     *
     * @param is
     *            an InputStream
     *
     * @return a XmlElement
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlElement parseFragmentFromInputStream(InputStream is)
    throws XmlBuilderException;
    
    /**
     * Parse input stream to create XML fragment using specified encoding.
     *
     * @param is
     *            an InputStream
     * @param encoding
     *            a String
     *
     * @return a XmlElement
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlElement parseFragmentFromInputStream(InputStream is,
                                                             String encoding) throws XmlBuilderException;
    
    /**
     * Parse reader to create XML fragment.
     *
     * @param reader
     *            a Reader
     *
     * @return a XmlElement
     *
     * @exception XmlBuilderException
     *
     */
    public abstract XmlElement parseFragmentFromReader(Reader reader)
    throws XmlBuilderException;
    
    /**
     * Parse input stream to create XML fragment.
     *
     * @param xmlAsString
     *            string containing XML fragment (well-formed)
     *
     * @return a XmlElement
     *
     * @exception XmlBuilderException
     *
     */
    public XmlElement parseFragmentFromString(String xmlAsString)
    throws XmlBuilderException {
        return parseFragmentFromReader(new StringReader(xmlAsString));
    }
    
    // --- serialization
    
    /**
     * Serialize item using default UTF8 encoding.
     *
     * @see serializeItem
     */
    public void serializeToOutputStream(Object item, // XmlContainer node,
                                        OutputStream os) throws XmlBuilderException {
        serializeToOutputStream(item, os, "UTF8");
    }
    
    /**
     * Serialize item to given output stream using given character encoding.
     *
     * @param item
     *            an Object
     * @param os
     *            an OutputStream
     * @param encoding
     *            a String
     *
     * @exception XmlBuilderException
     *
     * @see serializeItem
     *
     */
    public abstract void serializeToOutputStream(Object item, // XmlContainer // node,
                                                 OutputStream os, String encoding) throws XmlBuilderException;
    
    /**
     * Serialize item to given writer.
     *
     * @param item
     *            an Object
     * @param writer
     *            a Writer
     *
     * @exception XmlBuilderException
     *
     */
    public abstract void serializeToWriter(Object item, Writer writer)
    throws XmlBuilderException;

    /**
     * Serialize item to given writer.
     *
     * @param item
     *            an Object
     * @param writer
     *            a Writer
     *
     * @exception XmlBuilderException
     *
     */
    public abstract void serializeToWriter(Object item, Writer writer, boolean pretty)
    throws XmlBuilderException;
    
    /**
     * Convert item into String representing XML content.
     *
     * @param item
     *            an Object
     *
     * @return a String
     *
     * @exception XmlBuilderException
     *
     */
    public String serializeToString(Object item) throws XmlBuilderException {
        StringWriter sw = new StringWriter();
        serializeToWriter(item, sw);
        return sw.toString();
    }
    
    public String serializeToStringPretty(Object item) throws XmlBuilderException {
        StringWriter sw = new StringWriter();
        serializeToWriter(item, sw, true);
        return sw.toString();
    }
}

