/**
 * Copyright 2011-2018 Fraunhofer-Gesellschaft zur Förderung der angewandten Wissenschaften e.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ogema.driver.generic_xbee;

import java.util.Hashtable;
import java.util.Map;

import org.apache.felix.service.command.Descriptor;
import org.json.JSONArray;
import org.json.JSONObject;
import org.ogema.core.channelmanager.ChannelConfiguration;
import org.ogema.core.channelmanager.NoSuchDriverException;
import org.ogema.core.channelmanager.driverspi.ChannelLocator;
import org.ogema.core.channelmanager.driverspi.NoSuchInterfaceException;
import org.ogema.driverconfig.HLDriverInterface;
import org.osgi.framework.BundleContext;

public class ShellCommands implements HLDriverInterface {
	private BundleContext context = null;
	private GenericXbeeZbDriver driver = null;

	public ShellCommands(GenericXbeeZbDriver driver, BundleContext context) {
		Hashtable<String, Object> props = new Hashtable<String, Object>();
		props.put("osgi.command.scope", "xbhl");
		props.put("osgi.command.function", new String[] { "showCreatedChannels", "createChannel", "deleteChannel",
				"readChannel", "deviceScan" });
		this.context = context;
		this.driver = driver;
		this.context.registerService(this.getClass().getName(), this, props);
		context.registerService(HLDriverInterface.class, this, null);
	}

	public void showCreatedChannels() {
		for (Map.Entry<String, ChannelConfiguration> mapEntry : driver.channelMap.entrySet()) {
			System.out.println(mapEntry.getKey() + ": " + mapEntry.getValue().getChannelLocator().toString());
		}
	}

	@Override
	@Descriptor("Creates a new channel.")
	public void createChannel(
			@Descriptor("The interface ID/Port name.") String interfaceId,
			@Descriptor("The device address in form of 16Bit:EndpointID e. g. 0000:00.") String deviceAddress,
			@Descriptor("The channel address in form of ClusterId:Type:Id e. g. 0000:Attribute:0000.") String channelAddress,
			@Descriptor("The period in which the ChannelManager should try to request a new value. 0 for commands.") long timeout,
			@Descriptor("A name for the resource.") String resourceName, String deviceId) {
		GenericXbeeZbConfig config = new GenericXbeeZbConfig();
		config.interfaceId = interfaceId;
		config.deviceAddress = deviceAddress.toUpperCase();
		config.channelAddress = channelAddress.toUpperCase();
		config.timeout = timeout;
		config.resourceName = resourceName;
		config.deviceParameters = "";
		System.out.println(config.interfaceId + " " + config.deviceAddress + " " + config.channelAddress);
		driver.resourceAvailable(config);
	}

	@Descriptor("Deletes a channel.")
	public void deleteChannel(
			@Descriptor("The interface ID/Port name.") String interfaceId,
			@Descriptor("The device address in form of 16Bit:EndpointID e. g. 0000:00.") String deviceAddress,
			@Descriptor("The channel address in form of ClusterId:Type:Id e. g. 0000:Attribute:0000.") String channelAddress) {
		GenericXbeeZbConfig config = new GenericXbeeZbConfig();
		config.interfaceId = interfaceId;
		config.deviceAddress = deviceAddress.toUpperCase();
		config.channelAddress = channelAddress.toUpperCase();
		config.deviceParameters = "";
		driver.resourceUnavailable(config);
	}

	@Descriptor("Read from a channel.")
	public JSONObject readChannel(
			@Descriptor("The interface ID/Port name.") String interfaceId,
			@Descriptor("The device address in form of 16Bit:EndpointID e. g. 0000:00.") String deviceAddress,
			@Descriptor("The channel address in form of ClusterId:Type:Id e. g. 0000:Attribute:0000.") String channelAddress) {
		return updateValues(driver.getDevices().get(
				"xbee-driver:" + interfaceId + ":" + deviceAddress.toUpperCase() + ":" + channelAddress.toUpperCase()));
	}

	public void deviceScan(@Descriptor("The interface ID/Port name.") String interfaceId) throws Throwable {
		try {
			if (driver.appManager != null) // do it only if the HLD app is running
				driver.appManager.getChannelAccess().discoverDevices("xbee-driver", interfaceId, null, driver);
		} catch (Throwable e) {
			throw e;
		}
	}

	@Override
	public void writeChannel(String interfaceId, String deviceAddress, String channelAddress, String writeValue) {
		// TODO Auto-generated method stub

	}

	@Override
	public JSONArray showCreatedChannels(String deviceAddress) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public String whichID() {
		return "generic-xbee-zb";
	}

	JSONObject updateValues(GenericXbeeZbDevice dev) {
		return dev.packValuesAsJSON();
	}

}
