/*
 * Copyright 2016 William Oemler, Blueprint Medicines
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.oncoblocks.centromere.web.security;

import org.springframework.security.core.userdetails.UserDetails;

/**
 * Standard method interface for creating and validating user {@link TokenDetails}
 * 
 * @author woemler
 */
public interface TokenOperations {

	/**
	 * Generates a string token from the submitted {@link UserDetails}
	 * 
	 * @param userDetails {@link UserDetails}
	 * @return string token representation
	 */
	String createToken(UserDetails userDetails);

	/**
	 * Creates the hash of user credentials and token expiration timestamp.
	 * 
	 * @param userDetails {@link UserDetails}
	 * @param expires timestamp (in milliseconds) when the token expires.
	 * @return string representation of the hash
	 */
	String computeSignature(UserDetails userDetails, long expires);

	/**
	 * Returns the username portion of a submitted authentication token
	 * 
	 * @param authToken token generated by {@link org.oncoblocks.centromere.web.security.TokenOperations#createToken}
	 * @return username
	 */
	String getUserNameFromToken(String authToken);

	/**
	 * Checks that the submitted token is valid and returns a {@code boolean} verdict
	 * 
	 * @param authToken token generated by {@link org.oncoblocks.centromere.web.security.TokenOperations#createToken}
	 * @param userDetails {@link UserDetails}
	 * @return boolean result of validation
	 */
	boolean validateToken(String authToken, UserDetails userDetails);
}
