/*
 * Decompiled with CFR 0.152.
 */
package org.onlab.packet;

import com.google.common.net.InetAddresses;
import com.google.common.primitives.UnsignedBytes;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Objects;
import org.onlab.packet.Ip4Address;
import org.onlab.packet.Ip4Prefix;
import org.onlab.packet.Ip6Address;
import org.onlab.packet.Ip6Prefix;
import org.onlab.packet.IpPrefix;

public class IpAddress
implements Comparable<IpAddress> {
    private static final int BIT_MASK = 255;
    public static final int INET_BYTE_LENGTH = 4;
    public static final int INET_BIT_LENGTH = 32;
    public static final int INET6_BYTE_LENGTH = 16;
    public static final int INET6_BIT_LENGTH = 128;
    private final Version version;
    private final byte[] octets;

    protected IpAddress(Version version, byte[] value) {
        IpAddress.checkArguments(version, value, 0);
        this.version = version;
        switch (version) {
            case INET: {
                this.octets = Arrays.copyOf(value, 4);
                break;
            }
            case INET6: {
                this.octets = Arrays.copyOf(value, 16);
                break;
            }
            default: {
                this.octets = null;
            }
        }
    }

    protected IpAddress() {
        this.version = null;
        this.octets = null;
    }

    public Version version() {
        return this.version;
    }

    public boolean isIp4() {
        return this.version() == Ip4Address.VERSION;
    }

    public boolean isIp6() {
        return this.version() == Ip6Address.VERSION;
    }

    public Ip4Address getIp4Address() {
        if (!this.isIp4()) {
            return null;
        }
        if (this instanceof Ip4Address) {
            return (Ip4Address)this;
        }
        return Ip4Address.valueOf(this.octets);
    }

    public Ip6Address getIp6Address() {
        if (!this.isIp6()) {
            return null;
        }
        if (this instanceof Ip6Address) {
            return (Ip6Address)this;
        }
        return Ip6Address.valueOf(this.octets);
    }

    public byte[] toOctets() {
        return Arrays.copyOf(this.octets, this.octets.length);
    }

    public InetAddress toInetAddress() {
        try {
            return InetAddress.getByAddress(this.octets);
        }
        catch (UnknownHostException e) {
            return null;
        }
    }

    public static int byteLength(Version version) {
        switch (version) {
            case INET: {
                return 4;
            }
            case INET6: {
                return 16;
            }
        }
        String msg = "Invalid IP version " + version;
        throw new IllegalArgumentException(msg);
    }

    public static IpAddress valueOf(int value) {
        byte[] bytes = ByteBuffer.allocate(4).putInt(value).array();
        return new IpAddress(Version.INET, bytes);
    }

    public static IpAddress valueOf(Version version, byte[] value) {
        return new IpAddress(version, value);
    }

    public static IpAddress valueOf(Version version, byte[] value, int offset) {
        IpAddress.checkArguments(version, value, offset);
        byte[] bc = Arrays.copyOfRange(value, offset, value.length);
        return IpAddress.valueOf(version, bc);
    }

    public static IpAddress valueOf(InetAddress inetAddress) {
        byte[] bytes = inetAddress.getAddress();
        if (inetAddress instanceof Inet4Address) {
            return new IpAddress(Version.INET, bytes);
        }
        if (inetAddress instanceof Inet6Address) {
            return new IpAddress(Version.INET6, bytes);
        }
        if (bytes.length == 4) {
            return new IpAddress(Version.INET, bytes);
        }
        if (bytes.length == 16) {
            return new IpAddress(Version.INET6, bytes);
        }
        String msg = "Unrecognized IP version address string: " + inetAddress.toString();
        throw new IllegalArgumentException(msg);
    }

    public static IpAddress valueOf(String value) {
        InetAddress inetAddress = null;
        try {
            inetAddress = InetAddresses.forString((String)value);
        }
        catch (IllegalArgumentException e) {
            String msg = "Invalid IP address string: " + value;
            throw new IllegalArgumentException(msg);
        }
        return IpAddress.valueOf(inetAddress);
    }

    public static IpAddress makeMaskPrefix(Version version, int prefixLength) {
        byte[] mask = IpAddress.makeMaskPrefixArray(version, prefixLength);
        return new IpAddress(version, mask);
    }

    public static IpAddress makeMaskedAddress(IpAddress address, int prefixLength) {
        if (address instanceof Ip4Address) {
            Ip4Address ip4a = (Ip4Address)address;
            return Ip4Address.makeMaskedAddress(ip4a, prefixLength);
        }
        if (address instanceof Ip6Address) {
            Ip6Address ip6a = (Ip6Address)address;
            return Ip6Address.makeMaskedAddress(ip6a, prefixLength);
        }
        byte[] net = IpAddress.makeMaskedAddressArray(address, prefixLength);
        return IpAddress.valueOf(address.version(), net);
    }

    public boolean isZero() {
        for (byte b : this.octets) {
            if (b == 0) continue;
            return false;
        }
        return true;
    }

    public boolean isSelfAssigned() {
        return this.isIp4() && this.octets[0] == -87 && this.octets[1] == -2;
    }

    public boolean isMulticast() {
        return this.isIp4() ? Ip4Prefix.IPV4_MULTICAST_PREFIX.contains(this.getIp4Address()) : Ip6Prefix.IPV6_MULTICAST_PREFIX.contains(this.getIp6Address());
    }

    public boolean isLinkLocal() {
        return this.isIp4() ? Ip4Prefix.IPV4_LINK_LOCAL_PREFIX.contains(this.getIp4Address()) : Ip6Prefix.IPV6_LINK_LOCAL_PREFIX.contains(this.getIp6Address());
    }

    @Override
    public int compareTo(IpAddress o) {
        if (this.version != o.version) {
            return this.version.compareTo(o.version);
        }
        for (int i = 0; i < this.octets.length; ++i) {
            if (this.octets[i] == o.octets[i]) continue;
            return UnsignedBytes.compare((byte)this.octets[i], (byte)o.octets[i]);
        }
        return 0;
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.version, Arrays.hashCode(this.octets)});
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || !(obj instanceof IpAddress)) {
            return false;
        }
        IpAddress other = (IpAddress)obj;
        return this.version == other.version && Arrays.equals(this.octets, other.octets);
    }

    public String toString() {
        switch (this.version) {
            case INET: {
                return String.format("%d.%d.%d.%d", this.octets[0] & 0xFF, this.octets[1] & 0xFF, this.octets[2] & 0xFF, this.octets[3] & 0xFF);
            }
        }
        return this.ipv6ToStringHelper();
    }

    public IpPrefix toIpPrefix() {
        if (this.isIp4()) {
            return IpPrefix.valueOf(new IpAddress(Version.INET, this.octets), 32);
        }
        return IpPrefix.valueOf(new IpAddress(Version.INET6, this.octets), 128);
    }

    private static String addressName(Version version) {
        switch (version) {
            case INET: {
                return "IPv4";
            }
            case INET6: {
                return "IPv6";
            }
        }
        return "UnknownIP(" + version + ")";
    }

    static void checkArguments(Version version, byte[] value, int offset) {
        int addrByteLength = IpAddress.byteLength(version);
        if (offset < 0 || offset + addrByteLength > value.length) {
            String msg = value.length < addrByteLength ? "Invalid " + IpAddress.addressName(version) + " address array: array length: " + value.length + ". Must be at least " + addrByteLength : "Invalid " + IpAddress.addressName(version) + " address array: array offset: " + offset + ". Must be in the interval [0, " + (value.length - addrByteLength) + "]";
            throw new IllegalArgumentException(msg);
        }
    }

    static byte[] makeMaskPrefixArray(Version version, int prefixLength) {
        int i;
        int addrByteLength = IpAddress.byteLength(version);
        int addrBitLength = addrByteLength * 8;
        if (prefixLength < 0 || prefixLength > addrBitLength) {
            String msg = "Invalid IP prefix length: " + prefixLength + ". Must be in the interval [0, " + addrBitLength + "].";
            throw new IllegalArgumentException(msg);
        }
        int maskBytes = prefixLength / 8;
        int maskBits = prefixLength % 8;
        byte[] mask = new byte[addrByteLength];
        for (i = 0; i < maskBytes; ++i) {
            mask[i] = -1;
        }
        for (i = maskBytes; i < addrByteLength; ++i) {
            mask[i] = 0;
        }
        if (maskBits > 0) {
            mask[maskBytes] = (byte)(255 << 8 - maskBits);
        }
        return mask;
    }

    static byte[] makeMaskedAddressArray(IpAddress addr, int prefixLength) {
        byte[] mask = IpAddress.makeMaskPrefixArray(addr.version(), prefixLength);
        byte[] net = new byte[mask.length];
        for (int i = 0; i < net.length; ++i) {
            net[i] = (byte)(addr.octets[i] & mask[i]);
        }
        return net;
    }

    private String ipv6ToStringHelper() {
        StringBuilder buff = new StringBuilder();
        buff.append(String.format("%x:%x:%x:%x:%x:%x:%x:%x", (this.octets[0] & 0xFF) << 8 | this.octets[1] & 0xFF, (this.octets[2] & 0xFF) << 8 | this.octets[3] & 0xFF, (this.octets[4] & 0xFF) << 8 | this.octets[5] & 0xFF, (this.octets[6] & 0xFF) << 8 | this.octets[7] & 0xFF, (this.octets[8] & 0xFF) << 8 | this.octets[9] & 0xFF, (this.octets[10] & 0xFF) << 8 | this.octets[11] & 0xFF, (this.octets[12] & 0xFF) << 8 | this.octets[13] & 0xFF, (this.octets[14] & 0xFF) << 8 | this.octets[15] & 0xFF));
        int longestSeqStart = 0;
        int longestSeqLen = 0;
        int currSeqStart = 0;
        int currSeqLen = 0;
        for (int index = 0; index < buff.length(); ++index) {
            if (buff.charAt(index) == ':') {
                if (currSeqLen == 0 || buff.charAt(index + 1) != '0') continue;
                ++currSeqLen;
                continue;
            }
            if (buff.charAt(index) == '0' && (index == 0 || buff.charAt(index - 1) == ':')) {
                if (currSeqLen == 0) {
                    currSeqStart = index;
                }
                ++currSeqLen;
                continue;
            }
            if (currSeqLen > longestSeqLen) {
                longestSeqStart = currSeqStart;
                longestSeqLen = currSeqLen;
            }
            currSeqLen = 0;
        }
        if (currSeqLen > longestSeqLen) {
            longestSeqLen = currSeqLen;
            longestSeqStart = currSeqStart;
        }
        if (longestSeqLen > 1) {
            if (buff.length() == longestSeqStart + longestSeqLen) {
                buff.append(':');
            }
            buff.delete(longestSeqStart, longestSeqStart + longestSeqLen);
            if (longestSeqStart == 0) {
                buff.insert(0, ':');
            }
        }
        return buff.toString();
    }

    public static enum Version {
        INET,
        INET6;

    }
}

