/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.copycat.server.storage;

import io.atomix.catalyst.buffer.Buffer;
import io.atomix.catalyst.buffer.FileBuffer;
import io.atomix.catalyst.buffer.MappedBuffer;
import io.atomix.catalyst.buffer.SlicedBuffer;
import io.atomix.catalyst.serializer.Serializer;
import io.atomix.catalyst.util.Assert;
import io.atomix.copycat.server.storage.OffsetCleaner;
import io.atomix.copycat.server.storage.OffsetIndex;
import io.atomix.copycat.server.storage.SegmentDescriptor;
import io.atomix.copycat.server.storage.SegmentManager;
import io.atomix.copycat.server.storage.TermIndex;
import io.atomix.copycat.server.storage.entry.Entry;
import java.util.function.Predicate;

public class Segment
implements AutoCloseable {
    private final SegmentDescriptor descriptor;
    private final Serializer serializer;
    private final Buffer buffer;
    private final OffsetIndex offsetIndex;
    private final OffsetCleaner cleaner;
    private final TermIndex termIndex = new TermIndex();
    private final SegmentManager manager;
    private long skip = 0L;
    private boolean open = true;

    Segment(Buffer buffer, SegmentDescriptor descriptor, OffsetIndex offsetIndex, OffsetCleaner cleaner, Serializer serializer, SegmentManager manager) {
        this.serializer = Assert.notNull(serializer, "serializer");
        this.buffer = Assert.notNull(buffer, "buffer");
        this.descriptor = Assert.notNull(descriptor, "descriptor");
        this.offsetIndex = Assert.notNull(offsetIndex, "offsetIndex");
        this.cleaner = Assert.notNull(cleaner, "cleaner");
        this.manager = Assert.notNull(manager, "manager");
        long position = buffer.mark().position();
        int length = buffer.readUnsignedShort();
        while (length != 0) {
            long offset = buffer.readLong();
            if (buffer.readBoolean()) {
                this.termIndex.index(offset, buffer.readLong());
            }
            offsetIndex.index(offset, position);
            position = buffer.skip(length).position();
            length = buffer.mark().readUnsignedShort();
        }
        buffer.reset();
    }

    public SegmentDescriptor descriptor() {
        return this.descriptor;
    }

    public boolean isOpen() {
        return this.open;
    }

    public boolean isEmpty() {
        return this.offsetIndex.size() > 0 ? this.offsetIndex.lastOffset() + 1L + this.skip == 0L : this.skip == 0L;
    }

    public boolean isCompacted() {
        return this.descriptor.version() > 1L;
    }

    public boolean isFull() {
        return this.size() >= this.descriptor.maxSegmentSize() || this.offsetIndex.size() >= this.descriptor.maxEntries();
    }

    public long size() {
        return this.buffer.offset() + this.buffer.position();
    }

    public long length() {
        return !this.isEmpty() ? this.offsetIndex.lastOffset() + 1L + this.skip : 0L;
    }

    public int count() {
        return this.offsetIndex.size();
    }

    long index() {
        return this.descriptor.index();
    }

    public long firstIndex() {
        this.assertSegmentOpen();
        return !this.isEmpty() ? this.descriptor.index() : 0L;
    }

    public long lastIndex() {
        this.assertSegmentOpen();
        return !this.isEmpty() ? this.offsetIndex.lastOffset() + this.descriptor.index() + this.skip : this.descriptor.index() - 1L;
    }

    public long nextIndex() {
        return !this.isEmpty() ? this.lastIndex() + 1L : this.descriptor.index() + this.skip;
    }

    public long offset(long index) {
        return this.offsetIndex.find(this.relativeOffset(index));
    }

    private long relativeOffset(long index) {
        return index - this.descriptor.index();
    }

    private void checkRange(long index) {
        Assert.indexNot(this.isEmpty(), "segment is empty", new Object[0]);
        Assert.indexNot(index < this.firstIndex(), index + " is less than the first index in the segment", new Object[0]);
        Assert.indexNot(index > this.lastIndex(), index + " is greater than the last index in the segment", new Object[0]);
    }

    public long append(Entry entry) {
        Assert.notNull(entry, "entry");
        Assert.stateNot(this.isFull(), "segment is full", new Object[0]);
        long index = this.nextIndex();
        Assert.index(index == entry.getIndex(), "inconsistent index: %s", entry.getIndex());
        long offset = this.relativeOffset(index);
        long term = entry.getTerm();
        long lastTerm = this.termIndex.term();
        Assert.arg(term > 0L && term >= lastTerm, "term must be monotonically increasing", new Object[0]);
        long position = this.buffer.mark().position();
        boolean skipTerm = term == lastTerm;
        int headerLength = 11 + (skipTerm ? 0 : 8);
        this.serializer.writeObject(entry, this.buffer.skip(headerLength));
        int length = (int)(this.buffer.position() - (position + (long)headerLength));
        entry.setSize(length);
        this.buffer.reset().writeUnsignedShort(length).writeLong(offset);
        if (skipTerm) {
            this.buffer.writeBoolean(false).skip(length);
        } else {
            this.buffer.writeBoolean(true).writeLong(entry.getTerm()).skip(length);
        }
        this.offsetIndex.index(offset, position);
        if (term > lastTerm) {
            this.termIndex.index(offset, term);
        }
        this.skip = 0L;
        return index;
    }

    public long term(long index) {
        this.assertSegmentOpen();
        this.checkRange(index);
        long offset = this.relativeOffset(index);
        return this.termIndex.lookup(offset);
    }

    public synchronized <T extends Entry> T get(long index) {
        this.assertSegmentOpen();
        this.checkRange(index);
        long offset = this.relativeOffset(index);
        long position = this.offsetIndex.position(offset);
        if (position != -1L) {
            int length = this.buffer.readUnsignedShort(position);
            long entryOffset = this.buffer.readLong(position + 2L);
            Assert.state(entryOffset == offset, "inconsistent index: %s", index);
            boolean skipTerm = !this.buffer.readBoolean(position + 2L + 8L);
            try (Buffer value = this.buffer.slice(position + 2L + 8L + 1L + (long)(skipTerm ? 0 : 8), length);){
                Entry entry = (Entry)this.serializer.readObject(value);
                ((Entry)((Entry)entry.setIndex(index)).setTerm(this.termIndex.lookup(offset))).setSize(length);
                Entry entry2 = entry;
                return (T)entry2;
            }
        }
        return null;
    }

    boolean validIndex(long index) {
        this.assertSegmentOpen();
        return !this.isEmpty() && index >= this.firstIndex() && index <= this.lastIndex();
    }

    public boolean contains(long index) {
        this.assertSegmentOpen();
        if (!this.validIndex(index)) {
            return false;
        }
        long offset = this.relativeOffset(index);
        return this.offsetIndex.contains(offset);
    }

    public boolean clean(long index) {
        this.assertSegmentOpen();
        long offset = this.offsetIndex.find(this.relativeOffset(index));
        return offset != -1L && this.cleaner.clean(offset);
    }

    public boolean isClean(long index) {
        this.assertSegmentOpen();
        return this.cleaner.isClean(this.offsetIndex.find(this.relativeOffset(index)));
    }

    public long cleanCount() {
        this.assertSegmentOpen();
        return this.cleaner.count();
    }

    public Predicate<Long> cleanPredicate() {
        return new OffsetCleaner(this.cleaner.bits().copy());
    }

    public Segment skip(long entries) {
        this.assertSegmentOpen();
        this.skip += entries;
        return this;
    }

    public Segment truncate(long index) {
        this.assertSegmentOpen();
        Assert.index(index >= this.manager.commitIndex(), "cannot truncate committed index", new Object[0]);
        long offset = this.relativeOffset(index);
        long lastOffset = this.offsetIndex.lastOffset();
        long diff = Math.abs(lastOffset - offset);
        this.skip = Math.max(this.skip - diff, 0L);
        if (offset < lastOffset) {
            long position = this.offsetIndex.truncate(offset);
            ((Buffer)this.buffer.position(position).zero(position)).flush();
            this.termIndex.truncate(offset);
        }
        return this;
    }

    public Segment flush() {
        this.buffer.flush();
        this.offsetIndex.flush();
        return this;
    }

    @Override
    public void close() {
        this.buffer.close();
        this.offsetIndex.close();
        this.cleaner.close();
        this.descriptor.close();
        this.open = false;
    }

    public void delete() {
        Buffer buffer;
        Buffer buffer2 = buffer = this.buffer instanceof SlicedBuffer ? ((SlicedBuffer)this.buffer).root() : this.buffer;
        if (buffer instanceof FileBuffer) {
            ((FileBuffer)buffer).delete();
        } else if (buffer instanceof MappedBuffer) {
            ((MappedBuffer)buffer).delete();
        }
        this.offsetIndex.delete();
    }

    public String toString() {
        return String.format("Segment[id=%d, version=%d, index=%d, length=%d]", this.descriptor.id(), this.descriptor.version(), this.firstIndex(), this.length());
    }

    private void assertSegmentOpen() {
        Assert.state(this.isOpen(), "segment not open", new Object[0]);
    }
}

