/*
 * Decompiled with CFR 0.152.
 */
package net.kuujo.copycat.log;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import net.kuujo.copycat.log.AbstractLogSegment;
import net.kuujo.copycat.log.FileLogManager;
import net.kuujo.copycat.log.LogException;
import net.kuujo.copycat.log.LogManager;

public class FileLogSegment
extends AbstractLogSegment {
    private final FileLogManager log;
    private final File logFile;
    private final File indexFile;
    private final File metadataFile;
    private long timestamp;
    private FileChannel logFileChannel;
    private FileChannel indexFileChannel;
    private Long firstIndex;
    private Long lastIndex;
    private final ByteBuffer indexBuffer = ByteBuffer.allocate(8);

    FileLogSegment(FileLogManager log, long id, long firstIndex) {
        super(id, firstIndex);
        this.log = log;
        this.logFile = new File(log.base.getParentFile(), String.format("%s-%d.log", log.base.getName(), id));
        this.indexFile = new File(log.base.getParentFile(), String.format("%s-%d.index", log.base.getName(), id));
        this.metadataFile = new File(log.base.getParentFile(), String.format("%s-%d.metadata", log.base.getName(), id));
    }

    @Override
    public LogManager log() {
        return this.log;
    }

    @Override
    public long timestamp() {
        this.assertIsOpen();
        return this.timestamp;
    }

    @Override
    public void open() throws IOException {
        this.assertIsNotOpen();
        if (!this.logFile.getParentFile().exists()) {
            this.logFile.getParentFile().mkdirs();
        }
        if (!this.metadataFile.exists()) {
            this.timestamp = System.currentTimeMillis();
            try (RandomAccessFile metaFile = new RandomAccessFile(this.metadataFile, "rw");){
                metaFile.writeLong(((AbstractLogSegment)this).firstIndex);
                metaFile.writeLong(this.timestamp);
            }
        }
        try (RandomAccessFile metaFile = new RandomAccessFile(this.metadataFile, "r");){
            if (metaFile.readLong() != ((AbstractLogSegment)this).firstIndex) {
                throw new LogException("Segment metadata out of sync", new Object[0]);
            }
            this.timestamp = metaFile.readLong();
        }
        this.logFileChannel = FileChannel.open(this.logFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        this.logFileChannel.position(this.logFileChannel.size());
        this.indexFileChannel = FileChannel.open(this.indexFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        this.indexFileChannel.position(this.indexFileChannel.size());
        if (this.indexFileChannel.size() > 0L) {
            this.firstIndex = ((AbstractLogSegment)this).firstIndex;
            this.lastIndex = this.firstIndex + this.indexFileChannel.size() / 8L - 1L;
        }
    }

    @Override
    public boolean isEmpty() {
        this.assertIsOpen();
        return this.firstIndex == null;
    }

    @Override
    public boolean isOpen() {
        return this.logFileChannel != null && this.indexFileChannel != null;
    }

    @Override
    public long size() {
        this.assertIsOpen();
        try {
            return this.logFileChannel.size();
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    @Override
    public long entryCount() {
        this.assertIsOpen();
        return this.firstIndex != null ? this.lastIndex - this.firstIndex + 1L : 0L;
    }

    private long nextIndex() {
        if (this.firstIndex == null) {
            this.lastIndex = this.firstIndex = Long.valueOf(((AbstractLogSegment)this).firstIndex);
            return this.firstIndex;
        }
        this.lastIndex = this.lastIndex + 1L;
        return this.lastIndex;
    }

    @Override
    public long appendEntry(ByteBuffer entry) {
        this.assertIsOpen();
        long index = this.nextIndex();
        try {
            entry.rewind();
            long position = this.logFileChannel.position();
            this.logFileChannel.write(entry);
            this.storePosition(index, position);
        }
        catch (IOException e) {
            throw new LogException(e);
        }
        return index;
    }

    private void storePosition(long index, long position) {
        try {
            ByteBuffer buffer = ByteBuffer.allocate(8).putLong(position);
            buffer.flip();
            this.indexFileChannel.write(buffer, (index - this.firstIndex) * 8L);
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    private long findPosition(long index) {
        try {
            if (this.firstIndex == null || index <= this.firstIndex) {
                return 0L;
            }
            if (this.lastIndex == null || index > this.lastIndex) {
                return this.logFileChannel.size();
            }
            this.indexFileChannel.read(this.indexBuffer, (index - this.firstIndex) * 8L);
            this.indexBuffer.flip();
            long position = this.indexBuffer.getLong();
            this.indexBuffer.clear();
            return position;
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    @Override
    public Long firstIndex() {
        this.assertIsOpen();
        return this.firstIndex;
    }

    @Override
    public Long lastIndex() {
        this.assertIsOpen();
        return this.lastIndex;
    }

    @Override
    public boolean containsIndex(long index) {
        this.assertIsOpen();
        return this.firstIndex != null && this.lastIndex != null && this.firstIndex <= index && index <= this.lastIndex;
    }

    @Override
    public ByteBuffer getEntry(long index) {
        this.assertIsOpen();
        this.assertContainsIndex(index);
        try {
            long startPosition = this.findPosition(index);
            long endPosition = this.findPosition(index + 1L);
            ByteBuffer buffer = ByteBuffer.allocate((int)(endPosition - startPosition));
            this.logFileChannel.read(buffer, startPosition);
            buffer.flip();
            return buffer;
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    @Override
    public void removeAfter(long index) {
        this.assertIsOpen();
        if (this.containsIndex(index + 1L)) {
            try {
                this.logFileChannel.truncate(this.findPosition(index + 1L));
                this.indexFileChannel.truncate((index + 1L - this.firstIndex) * 8L);
                if (index >= this.firstIndex) {
                    this.lastIndex = index;
                } else {
                    this.lastIndex = null;
                    this.firstIndex = null;
                }
            }
            catch (IOException e) {
                throw new LogException(e);
            }
        }
    }

    @Override
    public void flush() {
        try {
            this.logFileChannel.force(false);
            this.indexFileChannel.force(false);
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    @Override
    public void close() throws IOException {
        this.assertIsOpen();
        this.logFileChannel.close();
        this.logFileChannel = null;
        this.indexFileChannel.close();
        this.indexFileChannel = null;
    }

    @Override
    public boolean isClosed() {
        return this.logFileChannel == null;
    }

    @Override
    public void delete() {
        this.logFile.delete();
        this.indexFile.delete();
        this.metadataFile.delete();
    }
}

