/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.onosproject.store.primitives;

import com.google.common.base.Throwables;
import org.onosproject.store.service.AsyncAtomicCounterMap;
import org.onosproject.store.service.AtomicCounterMap;
import org.onosproject.store.service.ConsistentMapException;
import org.onosproject.store.service.Synchronous;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * Default implementation of {@code AtomicCounterMap}.
 *
 * @param <K> map key type
 */
public class DefaultAtomicCounterMap<K> extends Synchronous<AsyncAtomicCounterMap<K>> implements AtomicCounterMap<K> {

    private final AsyncAtomicCounterMap<K> asyncCounterMap;
    private final long operationTimeoutMillis;

    public DefaultAtomicCounterMap(AsyncAtomicCounterMap<K> asyncCounterMap, long operationTimeoutMillis) {
        super(asyncCounterMap);
        this.asyncCounterMap = asyncCounterMap;
        this.operationTimeoutMillis = operationTimeoutMillis;
    }

    @Override
    public long incrementAndGet(K key) {
        return complete(asyncCounterMap.incrementAndGet(key));
    }

    @Override
    public long decrementAndGet(K key) {
        return complete(asyncCounterMap.decrementAndGet(key));
    }

    @Override
    public long getAndIncrement(K key) {
        return complete(asyncCounterMap.getAndIncrement(key));
    }

    @Override
    public long getAndDecrement(K key) {
        return complete(asyncCounterMap.getAndDecrement(key));
    }

    @Override
    public long addAndGet(K key, long delta) {
        return complete(asyncCounterMap.addAndGet(key, delta));
    }

    @Override
    public long getAndAdd(K key, long delta) {
        return complete(asyncCounterMap.getAndAdd(key, delta));
    }

    @Override
    public long get(K key) {
        return complete(asyncCounterMap.get(key));
    }

    @Override
    public long put(K key, long newValue) {
        return complete(asyncCounterMap.put(key, newValue));
    }

    @Override
    public long putIfAbsent(K key, long newValue) {
        return complete(asyncCounterMap.putIfAbsent(key, newValue));
    }

    @Override
    public boolean replace(K key, long expectedOldValue, long newValue) {
        return complete(asyncCounterMap.replace(key, expectedOldValue, newValue));
    }

    @Override
    public long remove(K key) {
        return complete(asyncCounterMap.remove(key));
    }

    @Override
    public boolean remove(K key, long value) {
        return complete(asyncCounterMap.remove(key, value));
    }

    @Override
    public int size() {
        return complete(asyncCounterMap.size());
    }

    @Override
    public boolean isEmpty() {
        return complete(asyncCounterMap.isEmpty());
    }

    @Override
    public void clear() {
        complete(asyncCounterMap.clear());
    }

    private <T> T complete(CompletableFuture<T> future) {
        try {
            return future.get(operationTimeoutMillis, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ConsistentMapException.Interrupted();
        } catch (TimeoutException e) {
            throw new ConsistentMapException.Timeout(name());
        } catch (ExecutionException e) {
            Throwables.throwIfUnchecked(e.getCause());
            throw new ConsistentMapException(e.getCause());
        }
    }
}
