/*
 * Copyright 2014-2015 Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.onosproject.net.device;

import org.onosproject.net.Device;
import org.onosproject.net.DeviceId;
import org.onosproject.net.MastershipRole;
import org.onosproject.net.Port;
import org.onosproject.net.PortNumber;

import java.util.List;

/**
 * Service for interacting with the inventory of infrastructure devices.
 */
public interface DeviceService {

    /**
     * Returns the number of infrastructure devices known to the system.
     *
     * @return number of infrastructure devices
     */
    int getDeviceCount();

    /**
     * Returns a collection of the currently known infrastructure
     * devices.
     *
     * @return collection of devices
     */
    Iterable<Device> getDevices();

    /**
     * Returns an iterable collection of all devices
     * currently available to the system.
     *
     * @return device collection
     */
    Iterable<Device> getAvailableDevices();

    /**
     * Returns the device with the specified identifier.
     *
     * @param deviceId device identifier
     * @return device or null if one with the given identifier is not known
     */
    Device getDevice(DeviceId deviceId);

    /**
     * Returns the current mastership role for the specified device.
     *
     * @param deviceId device identifier
     * @return designated mastership role
     */
    //XXX do we want this method here when MastershipService already does?
    MastershipRole getRole(DeviceId deviceId);


    /**
     * Returns the list of ports associated with the device.
     *
     * @param deviceId device identifier
     * @return list of ports
     */
    List<Port> getPorts(DeviceId deviceId);

    /**
     * Returns the list of port statistics associated with the device.
     *
     * @param deviceId device identitifer
     * @return list of port statistics
     */
    List<PortStatistics> getPortStatistics(DeviceId deviceId);

    /**
     * Returns the port with the specified number and hosted by the given device.
     *
     * @param deviceId   device identifier
     * @param portNumber port number
     * @return device port
     */
    Port getPort(DeviceId deviceId, PortNumber portNumber);

    /**
     * Indicates whether or not the device is presently online and available.
     *
     * @param deviceId device identifier
     * @return true if the device is available
     */
    boolean isAvailable(DeviceId deviceId);

    /**
     * Adds the specified device listener.
     *
     * @param listener device listener
     */
    void addListener(DeviceListener listener);

    /**
     * Removes the specified device listener.
     *
     * @param listener device listener
     */
    void removeListener(DeviceListener listener);

}
