/*
 * Decompiled with CFR 0.152.
 */
package org.onlab.packet;

import com.google.common.base.Preconditions;
import com.google.common.net.InetAddresses;
import com.google.common.primitives.UnsignedBytes;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Objects;
import org.onlab.packet.Ip4Address;
import org.onlab.packet.Ip6Address;

public class IpAddress
implements Comparable<IpAddress> {
    public static final int INET_BYTE_LENGTH = 4;
    public static final int INET_BIT_LENGTH = 32;
    public static final int INET6_BYTE_LENGTH = 16;
    public static final int INET6_BIT_LENGTH = 128;
    private final Version version;
    private final byte[] octets;

    protected IpAddress(Version version, byte[] value) {
        IpAddress.checkArguments(version, value, 0);
        this.version = version;
        switch (version) {
            case INET: {
                this.octets = Arrays.copyOf(value, 4);
                break;
            }
            case INET6: {
                this.octets = Arrays.copyOf(value, 16);
                break;
            }
            default: {
                this.octets = null;
            }
        }
    }

    public Version version() {
        return this.version;
    }

    public Ip4Address getIp4Address() {
        if (this.version() != Ip4Address.VERSION) {
            return null;
        }
        if (this instanceof Ip4Address) {
            return (Ip4Address)this;
        }
        return Ip4Address.valueOf(this.octets);
    }

    public Ip6Address getIp6Address() {
        if (this.version() != Ip6Address.VERSION) {
            return null;
        }
        if (this instanceof Ip6Address) {
            return (Ip6Address)this;
        }
        return Ip6Address.valueOf(this.octets);
    }

    public byte[] toOctets() {
        return Arrays.copyOf(this.octets, this.octets.length);
    }

    public static int byteLength(Version version) {
        switch (version) {
            case INET: {
                return 4;
            }
            case INET6: {
                return 16;
            }
        }
        String msg = "Invalid IP version " + (Object)((Object)version);
        throw new IllegalArgumentException(msg);
    }

    public static IpAddress valueOf(int value) {
        byte[] bytes = ByteBuffer.allocate(4).putInt(value).array();
        return new IpAddress(Version.INET, bytes);
    }

    public static IpAddress valueOf(Version version, byte[] value) {
        return new IpAddress(version, value);
    }

    public static IpAddress valueOf(Version version, byte[] value, int offset) {
        IpAddress.checkArguments(version, value, offset);
        byte[] bc = Arrays.copyOfRange(value, offset, value.length);
        return IpAddress.valueOf(version, bc);
    }

    public static IpAddress valueOf(InetAddress inetAddress) {
        byte[] bytes = inetAddress.getAddress();
        if (inetAddress instanceof Inet4Address) {
            return new IpAddress(Version.INET, bytes);
        }
        if (inetAddress instanceof Inet6Address) {
            return new IpAddress(Version.INET6, bytes);
        }
        if (bytes.length == 4) {
            return new IpAddress(Version.INET, bytes);
        }
        if (bytes.length == 16) {
            return new IpAddress(Version.INET6, bytes);
        }
        String msg = "Unrecognized IP version address string: " + inetAddress.toString();
        throw new IllegalArgumentException(msg);
    }

    public static IpAddress valueOf(String value) {
        InetAddress inetAddress = null;
        try {
            inetAddress = InetAddresses.forString(value);
        }
        catch (IllegalArgumentException e) {
            String msg = "Invalid IP address string: " + value;
            throw new IllegalArgumentException(msg);
        }
        return IpAddress.valueOf(inetAddress);
    }

    public static IpAddress makeMaskPrefix(Version version, int prefixLength) {
        byte[] mask = IpAddress.makeMaskPrefixArray(version, prefixLength);
        return new IpAddress(version, mask);
    }

    public static IpAddress makeMaskedAddress(IpAddress address, int prefixLength) {
        if (address instanceof Ip4Address) {
            Ip4Address ip4a = (Ip4Address)address;
            return Ip4Address.makeMaskedAddress(ip4a, prefixLength);
        }
        if (address instanceof Ip6Address) {
            Ip6Address ip6a = (Ip6Address)address;
            return Ip6Address.makeMaskedAddress(ip6a, prefixLength);
        }
        byte[] net = IpAddress.makeMaskedAddressArray(address, prefixLength);
        return IpAddress.valueOf(address.version(), net);
    }

    @Override
    public int compareTo(IpAddress o) {
        if (this.version != o.version) {
            return this.version.compareTo(o.version);
        }
        for (int i = 0; i < this.octets.length; ++i) {
            if (this.octets[i] == o.octets[i]) continue;
            return UnsignedBytes.compare(this.octets[i], o.octets[i]);
        }
        return 0;
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.version, Arrays.hashCode(this.octets)});
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || !(obj instanceof IpAddress)) {
            return false;
        }
        IpAddress other = (IpAddress)obj;
        return this.version == other.version && Arrays.equals(this.octets, other.octets);
    }

    public String toString() {
        InetAddress inetAddr = null;
        try {
            inetAddr = InetAddress.getByAddress(this.octets);
        }
        catch (UnknownHostException e) {
            Preconditions.checkState(false, "Internal error: Ip6Address.toString()");
            return "[Invalid IP Address]";
        }
        return InetAddresses.toAddrString(inetAddr);
    }

    private static String addressName(Version version) {
        switch (version) {
            case INET: {
                return "IPv4";
            }
            case INET6: {
                return "IPv6";
            }
        }
        return "UnknownIP(" + (Object)((Object)version) + ")";
    }

    static void checkArguments(Version version, byte[] value, int offset) {
        int addrByteLength = IpAddress.byteLength(version);
        if (offset < 0 || offset + addrByteLength > value.length) {
            String msg = value.length < addrByteLength ? "Invalid " + IpAddress.addressName(version) + " address array: array length: " + value.length + ". Must be at least " + addrByteLength : "Invalid " + IpAddress.addressName(version) + " address array: array offset: " + offset + ". Must be in the interval [0, " + (value.length - addrByteLength) + "]";
            throw new IllegalArgumentException(msg);
        }
    }

    static byte[] makeMaskPrefixArray(Version version, int prefixLength) {
        int i;
        int addrByteLength = IpAddress.byteLength(version);
        int addrBitLength = addrByteLength * 8;
        if (prefixLength < 0 || prefixLength > addrBitLength) {
            String msg = "Invalid IP prefix length: " + prefixLength + ". Must be in the interval [0, " + addrBitLength + "].";
            throw new IllegalArgumentException(msg);
        }
        int maskBytes = prefixLength / 8;
        int maskBits = prefixLength % 8;
        byte[] mask = new byte[addrByteLength];
        for (i = 0; i < maskBytes; ++i) {
            mask[i] = -1;
        }
        for (i = maskBytes; i < addrByteLength; ++i) {
            mask[i] = 0;
        }
        if (maskBits > 0) {
            mask[maskBytes] = (byte)(255 << 8 - maskBits);
        }
        return mask;
    }

    static byte[] makeMaskedAddressArray(IpAddress addr, int prefixLength) {
        byte[] mask = IpAddress.makeMaskPrefixArray(addr.version(), prefixLength);
        byte[] net = new byte[mask.length];
        for (int i = 0; i < net.length; ++i) {
            net[i] = (byte)(addr.octets[i] & mask[i]);
        }
        return net;
    }

    public static enum Version {
        INET,
        INET6;

    }
}

