/*
 * Copyright 2014-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.onosproject.store.serializers;

import java.util.Map.Entry;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Serializer;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;

/**
* Kryo Serializer for {@link ImmutableMap}.
*/
public class ImmutableMapSerializer extends Serializer<ImmutableMap<?, ?>> {

    /**
     * Creates {@link ImmutableMap} serializer instance.
     */
    public ImmutableMapSerializer() {
        // non-null, immutable
        super(false, true);
    }

    @Override
    public void write(Kryo kryo, Output output, ImmutableMap<?, ?> object) {
        output.writeInt(object.size());
        for (Entry<?, ?> e : object.entrySet()) {
            kryo.writeClassAndObject(output, e.getKey());
            kryo.writeClassAndObject(output, e.getValue());
        }
    }

    @Override
    public ImmutableMap<?, ?> read(Kryo kryo, Input input,
                                    Class<ImmutableMap<?, ?>> type) {
        final int size = input.readInt();
        switch (size) {
        case 0:
            return ImmutableMap.of();
        case 1:
            return ImmutableMap.of(kryo.readClassAndObject(input),
                                   kryo.readClassAndObject(input));

        default:
            Builder<Object, Object> builder = ImmutableMap.builder();
            for (int i = 0; i < size; ++i) {
                builder.put(kryo.readClassAndObject(input),
                            kryo.readClassAndObject(input));
            }
            return builder.build();
        }
    }
}
