/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.onosproject.incubator.net.l2monitoring.soam.delay;

import java.time.Duration;
import java.util.Collection;

import org.onosproject.incubator.net.l2monitoring.soam.SoamId;

/**
 * Delay Measurement from ITU Y.1731 Chapter 8.2, MEF 17, MEF 36.1 and MEF 39.
 *
 * This represents the result returned and includes the parameters used to
 * configure the DM when it was created
 */
public interface DelayMeasurementEntry extends DelayMeasurementCreate {
    /**
     * This uniquely identifies a scheduled measurement.
     * It is automatically generated by the server on creation of a new measurement
     * @return the id
     */
    SoamId dmId();

    /**
     * The current status of the DM session.
     * A value of 'active' indicates the current DM session is active,
     * i.e. the current time lies between the start time and the stop time, and
     * enabled is true. A value of 'not-active' indicates the current DM session
     * is not active, i.e. it has not started yet, has stopped upon reaching the
     * stop time, or is disabled
     * @return the status
     */
    SessionStatus sessionStatus();

    /**
     * The two-way frame delay calculated by this MEP from the last received SOAM PDU.
     * This object is undefined is measurement-type is dm1-transmitted or dm1-received
     * @return The delay as a java duration
     */
    Duration frameDelayTwoWay();

    /**
     * The frame delay in the forward direction calculated by this MEP from the last received SOAM PDU.
     * The value of this object may not be accurate in the absence of sufficiently
     * precise clock synchronization.
     * This object is undefined is measurement-type is dm1-transmitted
     * @return The delay as a java duration
     */
    Duration frameDelayForward();

    /**
     * The frame delay in the backward direction calculated by this MEP from the last received SOAM PDU.
     * The value of this object may not be accurate in the absence of sufficiently
     * precise clock synchronization.
     * This object is undefined is measurement-type is dm1-transmitted or dm1-received
     * @return The delay as a java duration
     */
    Duration frameDelayBackward();

    /**
     * The last two-way inter-frame delay interval calculated by this MEP.
     * The value of this object is undefined when measurement-type is dm1-transmitted or dm1-received
     * @return The delay as a java duration
     */
    Duration interFrameDelayVariationTwoWay();

    /**
     * The last one-way inter-frame delay interval in the forward direction calculated by this MEP.
     * The value of this object is undefined when measurement-type is dm1-transmitted
     * @return The delay as a java duration
     */
    Duration interFrameDelayVariationForward();

    /**
     * The last one-way inter-frame delay interval in the backward direction calculated by this MEP.
     * The value of this object is undefined when measurement-type is
     * dm1-transmitted or dm1-received
     * @return The delay as a java duration
     */
    Duration interFrameDelayVariationBackward();

    /**
     * The results for the current Measurement Interval in a SOAM Delay Measurement session.
     * gathered during the interval indicated by measurement-interval.
     * @return The current set of results
     */
    DelayMeasurementStatCurrent currentResult();

    /**
     * The results for history Measurement Intervals in a SOAM Delay Measurement session.
     * @return A collection of history results
     */
    Collection<DelayMeasurementStatHistory> historicalResults();

    /**
     * Builder for {@link org.onosproject.incubator.net.l2monitoring.soam.delay.DelayMeasurementEntry}.
     */
    public interface DmEntryBuilder extends DmCreateBuilder {
        DmEntryBuilder sessionStatus(SessionStatus sessionStatus);

        DmEntryBuilder frameDelayTwoWay(Duration frameDelayTwoWay);

        DmEntryBuilder frameDelayForward(Duration frameDelayForward);

        DmEntryBuilder frameDelayBackward(Duration frameDelayBackward);

        DmEntryBuilder interFrameDelayVariationTwoWay(Duration interFrameDelayVariationTwoWay);

        DmEntryBuilder interFrameDelayVariationForward(Duration interFrameDelayVariationForward);

        DmEntryBuilder interFrameDelayVariationBackward(Duration interFrameDelayVariationBackward);

        DmEntryBuilder currentResult(DelayMeasurementStatCurrent currentResult);

        DmEntryBuilder addToHistoricalResults(
                DelayMeasurementStatHistory historicalResult);

        DelayMeasurementEntry build();
    }

    /**
     * Session Status options.
     */
    public enum SessionStatus {
        ACTIVE,
        NOT_ACTIVE;
    }
}
