/*
 * Decompiled with CFR 0.152.
 */
package org.onosproject.store.service.impl;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.InflaterInputStream;
import net.kuujo.copycat.Command;
import net.kuujo.copycat.Query;
import net.kuujo.copycat.StateMachine;
import org.onlab.util.Tools;
import org.onosproject.store.cluster.messaging.MessageSubject;
import org.onosproject.store.service.BatchReadRequest;
import org.onosproject.store.service.BatchWriteRequest;
import org.onosproject.store.service.ReadRequest;
import org.onosproject.store.service.ReadResult;
import org.onosproject.store.service.ReadStatus;
import org.onosproject.store.service.VersionedValue;
import org.onosproject.store.service.WriteRequest;
import org.onosproject.store.service.WriteResult;
import org.onosproject.store.service.WriteStatus;
import org.onosproject.store.service.impl.ClusterMessagingProtocol;
import org.onosproject.store.service.impl.DatabaseUpdateEventListener;
import org.onosproject.store.service.impl.SnapshotException;
import org.onosproject.store.service.impl.TableModificationEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseStateMachine
implements StateMachine {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final ExecutorService updatesExecutor = Executors.newSingleThreadExecutor(Tools.namedThreads((String)"database-statemachine-updates"));
    public static final MessageSubject DATABASE_UPDATE_EVENTS = new MessageSubject("database-update-events");
    private final Set<DatabaseUpdateEventListener> listeners = Sets.newIdentityHashSet();
    private State state = new State();
    private boolean compressSnapshot = true;

    @Command
    public boolean createTable(String tableName) {
        TableMetadata metadata = new TableMetadata(tableName);
        return this.createTable(metadata);
    }

    @Command
    public boolean createTable(String tableName, Integer ttlMillis) {
        TableMetadata metadata = new TableMetadata(tableName, ttlMillis);
        return this.createTable(metadata);
    }

    private boolean createTable(final TableMetadata metadata) {
        Map<String, VersionedValue> existingTable = this.state.getTable(metadata.tableName());
        if (existingTable != null) {
            return false;
        }
        this.state.createTable(metadata);
        this.updatesExecutor.submit(new Runnable(){

            @Override
            public void run() {
                for (DatabaseUpdateEventListener listener : DatabaseStateMachine.this.listeners) {
                    listener.tableCreated(metadata);
                }
            }
        });
        return true;
    }

    @Command
    public boolean dropTable(final String tableName) {
        if (this.state.removeTable(tableName)) {
            this.updatesExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    for (DatabaseUpdateEventListener listener : DatabaseStateMachine.this.listeners) {
                        listener.tableDeleted(tableName);
                    }
                }
            });
            return true;
        }
        return false;
    }

    @Command
    public boolean dropAllTables() {
        final Set<String> tableNames = this.state.getTableNames();
        this.state.removeAllTables();
        this.updatesExecutor.submit(new Runnable(){

            @Override
            public void run() {
                for (DatabaseUpdateEventListener listener : DatabaseStateMachine.this.listeners) {
                    for (String tableName : tableNames) {
                        listener.tableDeleted(tableName);
                    }
                }
            }
        });
        return true;
    }

    @Query
    public Set<String> listTables() {
        return ImmutableSet.copyOf(this.state.getTableNames());
    }

    @Query
    public List<ReadResult> read(BatchReadRequest batchRequest) {
        ArrayList<ReadResult> results = new ArrayList<ReadResult>(batchRequest.batchSize());
        for (ReadRequest request : batchRequest.getAsList()) {
            Map<String, VersionedValue> table = this.state.getTable(request.tableName());
            if (table == null) {
                results.add(new ReadResult(ReadStatus.NO_SUCH_TABLE, request.tableName(), request.key(), null));
                continue;
            }
            VersionedValue value = VersionedValue.copy((VersionedValue)table.get(request.key()));
            results.add(new ReadResult(ReadStatus.OK, request.tableName(), request.key(), value));
        }
        return results;
    }

    @Query
    public Map<String, VersionedValue> getAll(String tableName) {
        return ImmutableMap.copyOf(this.state.getTable(tableName));
    }

    WriteStatus checkIfApplicable(WriteRequest request, VersionedValue value) {
        switch (request.type()) {
            case PUT: {
                return WriteStatus.OK;
            }
            case PUT_IF_ABSENT: {
                if (value == null) {
                    return WriteStatus.OK;
                }
                return WriteStatus.PRECONDITION_VIOLATION;
            }
            case PUT_IF_VALUE: 
            case REMOVE_IF_VALUE: {
                if (value != null && Arrays.equals(value.value(), request.oldValue())) {
                    return WriteStatus.OK;
                }
                return WriteStatus.PRECONDITION_VIOLATION;
            }
            case PUT_IF_VERSION: 
            case REMOVE_IF_VERSION: {
                if (value != null && request.previousVersion() == value.version()) {
                    return WriteStatus.OK;
                }
                return WriteStatus.PRECONDITION_VIOLATION;
            }
            case REMOVE: {
                return WriteStatus.OK;
            }
        }
        this.log.error("Should never reach here {}", (Object)request);
        return WriteStatus.ABORTED;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Command
    public List<WriteResult> write(BatchWriteRequest batchRequest) {
        boolean abort = false;
        ArrayList<WriteResult> results = new ArrayList<WriteResult>(batchRequest.batchSize());
        for (WriteRequest request : batchRequest.getAsList()) {
            Map<String, VersionedValue> table = this.state.getTable(request.tableName());
            if (table == null) {
                results.add(new WriteResult(WriteStatus.NO_SUCH_TABLE, null));
                abort = true;
                continue;
            }
            VersionedValue value = table.get(request.key());
            WriteStatus result = this.checkIfApplicable(request, value);
            results.add(new WriteResult(result, value));
            if (result == WriteStatus.OK) continue;
            abort = true;
        }
        if (abort) {
            for (int i = 0; i < results.size(); ++i) {
                if (((WriteResult)results.get(i)).status() != WriteStatus.OK) continue;
                results.set(i, new WriteResult(WriteStatus.ABORTED, null));
            }
            return results;
        }
        final LinkedList tableModificationEvents = Lists.newLinkedList();
        for (WriteRequest request : batchRequest.getAsList()) {
            Map<String, VersionedValue> table = this.state.getTable(request.tableName());
            TableModificationEvent tableModificationEvent = null;
            Map<String, VersionedValue> map = table;
            synchronized (map) {
                switch (request.type()) {
                    case PUT: 
                    case PUT_IF_ABSENT: 
                    case PUT_IF_VALUE: 
                    case PUT_IF_VERSION: {
                        VersionedValue newValue = new VersionedValue(request.newValue(), this.state.nextVersion());
                        VersionedValue previousValue = table.put(request.key(), newValue);
                        WriteResult putResult = new WriteResult(WriteStatus.OK, previousValue);
                        results.add(putResult);
                        tableModificationEvent = previousValue == null ? TableModificationEvent.rowAdded(request.tableName(), request.key(), newValue) : TableModificationEvent.rowUpdated(request.tableName(), request.key(), newValue);
                        break;
                    }
                    case REMOVE_IF_VALUE: 
                    case REMOVE_IF_VERSION: 
                    case REMOVE: {
                        VersionedValue removedValue = table.remove(request.key());
                        WriteResult removeResult = new WriteResult(WriteStatus.OK, removedValue);
                        results.add(removeResult);
                        if (removedValue == null) break;
                        tableModificationEvent = TableModificationEvent.rowDeleted(request.tableName(), request.key(), removedValue);
                        break;
                    }
                    default: {
                        this.log.error("Invalid WriteRequest type {}", (Object)request.type());
                    }
                }
            }
            if (tableModificationEvent == null) continue;
            tableModificationEvents.add(tableModificationEvent);
        }
        this.updatesExecutor.submit(new Runnable(){

            @Override
            public void run() {
                for (DatabaseUpdateEventListener listener : DatabaseStateMachine.this.listeners) {
                    for (TableModificationEvent tableModificationEvent : tableModificationEvents) {
                        DatabaseStateMachine.this.log.trace("Publishing table modification event: {}", (Object)tableModificationEvent);
                        listener.tableModified(tableModificationEvent);
                    }
                }
            }
        });
        return results;
    }

    public byte[] takeSnapshot() {
        try {
            if (this.compressSnapshot) {
                byte[] input = ClusterMessagingProtocol.DB_SERIALIZER.encode((Object)this.state);
                ByteArrayOutputStream comp = new ByteArrayOutputStream(input.length);
                DeflaterOutputStream compressor = new DeflaterOutputStream(comp);
                compressor.write(input, 0, input.length);
                compressor.close();
                return comp.toByteArray();
            }
            return ClusterMessagingProtocol.DB_SERIALIZER.encode((Object)this.state);
        }
        catch (Exception e) {
            this.log.error("Failed to take snapshot", (Throwable)e);
            throw new SnapshotException(e);
        }
    }

    public void installSnapshot(byte[] data) {
        try {
            if (this.compressSnapshot) {
                ByteArrayInputStream in = new ByteArrayInputStream(data);
                InflaterInputStream decompressor = new InflaterInputStream(in);
                this.state = (State)ClusterMessagingProtocol.DB_SERIALIZER.decode((InputStream)decompressor);
            } else {
                this.state = (State)ClusterMessagingProtocol.DB_SERIALIZER.decode(data);
            }
            this.updatesExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    for (DatabaseUpdateEventListener listener : DatabaseStateMachine.this.listeners) {
                        listener.snapshotInstalled(DatabaseStateMachine.this.state);
                    }
                }
            });
        }
        catch (Exception e) {
            this.log.error("Failed to install from snapshot", (Throwable)e);
            throw new SnapshotException(e);
        }
    }

    public void addEventListener(DatabaseUpdateEventListener listener) {
        this.listeners.add(listener);
    }

    public void removeEventListener(DatabaseUpdateEventListener listener) {
        this.listeners.remove(listener);
    }

    public static class TableMetadata {
        private final String tableName;
        private final boolean expireOldEntries;
        private final int ttlMillis;

        public TableMetadata(String tableName) {
            this.tableName = tableName;
            this.expireOldEntries = false;
            this.ttlMillis = Integer.MAX_VALUE;
        }

        public TableMetadata(String tableName, int ttlMillis) {
            this.tableName = tableName;
            this.expireOldEntries = true;
            this.ttlMillis = ttlMillis;
        }

        public String tableName() {
            return this.tableName;
        }

        public boolean expireOldEntries() {
            return this.expireOldEntries;
        }

        public int ttlMillis() {
            return this.ttlMillis;
        }

        public String toString() {
            return MoreObjects.toStringHelper(this.getClass()).add("tableName", (Object)this.tableName).add("expireOldEntries", this.expireOldEntries).add("ttlMillis", this.ttlMillis).toString();
        }
    }

    public static class State {
        private final Map<String, TableMetadata> tableMetadata = Maps.newHashMap();
        private final Map<String, Map<String, VersionedValue>> tableData = Maps.newHashMap();
        private long versionCounter = 1L;

        Map<String, VersionedValue> getTable(String tableName) {
            return this.tableData.get(tableName);
        }

        void createTable(TableMetadata metadata) {
            this.tableMetadata.put(metadata.tableName, metadata);
            this.tableData.put(metadata.tableName, Maps.newHashMap());
        }

        TableMetadata getTableMetadata(String tableName) {
            return this.tableMetadata.get(tableName);
        }

        long nextVersion() {
            return this.versionCounter++;
        }

        Set<String> getTableNames() {
            return ImmutableSet.copyOf(this.tableMetadata.keySet());
        }

        boolean removeTable(String tableName) {
            if (!this.tableMetadata.containsKey(tableName)) {
                return false;
            }
            this.tableMetadata.remove(tableName);
            this.tableData.remove(tableName);
            return true;
        }

        void removeAllTables() {
            this.tableMetadata.clear();
            this.tableData.clear();
        }
    }
}

