/*
 * Decompiled with CFR 0.152.
 */
package org.onosproject.store.cluster.impl;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.apache.felix.scr.annotations.Activate;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Deactivate;
import org.apache.felix.scr.annotations.Modified;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.joda.time.DateTime;
import org.onlab.packet.IpAddress;
import org.onlab.util.KryoNamespace;
import org.onlab.util.Tools;
import org.onosproject.cfg.ComponentConfigService;
import org.onosproject.cfg.ConfigProperty;
import org.onosproject.cluster.ClusterEvent;
import org.onosproject.cluster.ClusterMetadataService;
import org.onosproject.cluster.ClusterStore;
import org.onosproject.cluster.ClusterStoreDelegate;
import org.onosproject.cluster.ControllerNode;
import org.onosproject.cluster.DefaultControllerNode;
import org.onosproject.cluster.NodeId;
import org.onosproject.event.Event;
import org.onosproject.store.AbstractStore;
import org.onosproject.store.cluster.impl.PhiAccrualFailureDetector;
import org.onosproject.store.cluster.messaging.Endpoint;
import org.onosproject.store.cluster.messaging.MessagingService;
import org.onosproject.store.serializers.KryoNamespaces;
import org.onosproject.store.serializers.StoreSerializer;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(immediate=true)
@Service
public class DistributedClusterStore
extends AbstractStore<ClusterEvent, ClusterStoreDelegate>
implements ClusterStore {
    private static final Logger log = LoggerFactory.getLogger(DistributedClusterStore.class);
    public static final String HEARTBEAT_MESSAGE = "onos-cluster-heartbeat";
    private static final int DEFAULT_HEARTBEAT_INTERVAL = 100;
    @Property(name="heartbeatInterval", intValue={100}, label="Interval time to send heartbeat to other controller nodes (millisecond)")
    private int heartbeatInterval = 100;
    private static final int DEFAULT_PHI_FAILURE_THRESHOLD = 10;
    @Property(name="phiFailureThreshold", intValue={10}, label="the value of Phi threshold to detect accrual failure")
    private int phiFailureThreshold = 10;
    private static final StoreSerializer SERIALIZER = StoreSerializer.using((KryoNamespace)KryoNamespace.newBuilder().register(KryoNamespaces.API).nextId(500).register(new Class[]{HeartbeatMessage.class}).build("ClusterStore"));
    private static final String INSTANCE_ID_NULL = "Instance ID cannot be null";
    private final Map<NodeId, ControllerNode> allNodes = Maps.newConcurrentMap();
    private final Map<NodeId, ControllerNode.State> nodeStates = Maps.newConcurrentMap();
    private final Map<NodeId, DateTime> nodeStateLastUpdatedTimes = Maps.newConcurrentMap();
    private ScheduledExecutorService heartBeatSender = Executors.newSingleThreadScheduledExecutor(Tools.groupedThreads((String)"onos/cluster/membership", (String)"heartbeat-sender", (Logger)log));
    private ExecutorService heartBeatMessageHandler = Executors.newSingleThreadExecutor(Tools.groupedThreads((String)"onos/cluster/membership", (String)"heartbeat-receiver", (Logger)log));
    private PhiAccrualFailureDetector failureDetector;
    private ControllerNode localNode;
    @Reference(cardinality=ReferenceCardinality.MANDATORY_UNARY)
    protected ClusterMetadataService clusterMetadataService;
    @Reference(cardinality=ReferenceCardinality.MANDATORY_UNARY)
    protected MessagingService messagingService;
    @Reference(cardinality=ReferenceCardinality.OPTIONAL_UNARY, bind="bindComponentConfigService", unbind="unbindComponentConfigService", policy=ReferencePolicy.DYNAMIC)
    protected ComponentConfigService cfgService;

    protected void bindComponentConfigService(ComponentConfigService service) {
        if (this.cfgService == null) {
            this.cfgService = service;
            this.cfgService.registerProperties(((Object)((Object)this)).getClass());
            this.readComponentConfiguration();
        }
    }

    protected void unbindComponentConfigService(ComponentConfigService service) {
        if (this.cfgService == service) {
            this.cfgService.unregisterProperties(((Object)((Object)this)).getClass(), false);
            this.cfgService = null;
        }
    }

    @Activate
    public void activate() {
        this.localNode = this.clusterMetadataService.getLocalNode();
        this.messagingService.registerHandler(HEARTBEAT_MESSAGE, (BiConsumer)new HeartbeatMessageHandler(), (Executor)this.heartBeatMessageHandler);
        this.failureDetector = new PhiAccrualFailureDetector();
        this.heartBeatSender.scheduleWithFixedDelay(this::heartbeat, 0L, this.heartbeatInterval, TimeUnit.MILLISECONDS);
        log.info("Started");
    }

    @Deactivate
    public void deactivate() {
        this.messagingService.unregisterHandler(HEARTBEAT_MESSAGE);
        this.heartBeatSender.shutdownNow();
        this.heartBeatMessageHandler.shutdownNow();
        log.info("Stopped");
    }

    @Modified
    public void modified(ComponentContext context) {
        this.readComponentConfiguration();
    }

    public void setDelegate(ClusterStoreDelegate delegate) {
        Preconditions.checkNotNull((Object)delegate, (Object)"Delegate cannot be null");
        this.delegate = delegate;
    }

    public void unsetDelegate(ClusterStoreDelegate delegate) {
        this.delegate = null;
    }

    public boolean hasDelegate() {
        return this.delegate != null;
    }

    public ControllerNode getLocalNode() {
        return this.localNode;
    }

    public Set<ControllerNode> getNodes() {
        return ImmutableSet.copyOf(this.allNodes.values());
    }

    public ControllerNode getNode(NodeId nodeId) {
        Preconditions.checkNotNull((Object)nodeId, (Object)INSTANCE_ID_NULL);
        return this.allNodes.get(nodeId);
    }

    public ControllerNode.State getState(NodeId nodeId) {
        Preconditions.checkNotNull((Object)nodeId, (Object)INSTANCE_ID_NULL);
        return (ControllerNode.State)MoreObjects.firstNonNull((Object)this.nodeStates.get(nodeId), (Object)ControllerNode.State.INACTIVE);
    }

    public void markFullyStarted(boolean started) {
        this.updateState(this.localNode.id(), started ? ControllerNode.State.READY : ControllerNode.State.ACTIVE);
    }

    public ControllerNode addNode(NodeId nodeId, IpAddress ip, int tcpPort) {
        Preconditions.checkNotNull((Object)nodeId, (Object)INSTANCE_ID_NULL);
        DefaultControllerNode node = new DefaultControllerNode(nodeId, ip, tcpPort);
        this.addNode((ControllerNode)node);
        return node;
    }

    public void removeNode(NodeId nodeId) {
        Preconditions.checkNotNull((Object)nodeId, (Object)INSTANCE_ID_NULL);
        ControllerNode node = this.allNodes.remove(nodeId);
        if (node != null) {
            this.nodeStates.remove(nodeId);
            this.notifyDelegate((Event)new ClusterEvent(ClusterEvent.Type.INSTANCE_REMOVED, node));
        }
    }

    private void addNode(ControllerNode node) {
        this.allNodes.put(node.id(), node);
        this.updateState(node.id(), node.equals(this.localNode) ? ControllerNode.State.ACTIVE : ControllerNode.State.INACTIVE);
        this.notifyDelegate((Event)new ClusterEvent(ClusterEvent.Type.INSTANCE_ADDED, node));
    }

    private void updateState(NodeId nodeId, ControllerNode.State newState) {
        ControllerNode.State currentState = this.nodeStates.get(nodeId);
        if (!Objects.equals(currentState, newState)) {
            this.nodeStates.put(nodeId, newState);
            this.nodeStateLastUpdatedTimes.put(nodeId, DateTime.now());
            this.notifyStateChange(nodeId, currentState, newState);
        }
    }

    private void heartbeat() {
        try {
            Set<ControllerNode> peers = this.allNodes.values().stream().filter(node -> !node.id().equals((Object)this.localNode.id())).collect(Collectors.toSet());
            ControllerNode.State state = this.nodeStates.get(this.localNode.id());
            byte[] hbMessagePayload = SERIALIZER.encode((Object)new HeartbeatMessage(this.localNode, state));
            peers.forEach(node -> {
                this.heartbeatToPeer(hbMessagePayload, (ControllerNode)node);
                ControllerNode.State currentState = this.nodeStates.get(node.id());
                double phi = this.failureDetector.phi(node.id());
                if (phi >= (double)this.phiFailureThreshold) {
                    if (currentState.isActive()) {
                        this.updateState(node.id(), ControllerNode.State.INACTIVE);
                    }
                } else if (currentState == ControllerNode.State.INACTIVE) {
                    this.updateState(node.id(), ControllerNode.State.ACTIVE);
                }
            });
        }
        catch (Exception e) {
            log.debug("Failed to send heartbeat", (Throwable)e);
        }
    }

    private void notifyStateChange(NodeId nodeId, ControllerNode.State oldState, ControllerNode.State newState) {
        if (oldState != newState) {
            ControllerNode node = this.allNodes.get(nodeId);
            if (node == null) {
                log.debug("Could not find node {} in the cluster, ignoring state change", (Object)nodeId);
                return;
            }
            ClusterEvent.Type type = newState == ControllerNode.State.READY ? ClusterEvent.Type.INSTANCE_READY : (newState == ControllerNode.State.ACTIVE ? ClusterEvent.Type.INSTANCE_ACTIVATED : ClusterEvent.Type.INSTANCE_DEACTIVATED);
            this.notifyDelegate((Event)new ClusterEvent(type, node));
        }
    }

    private void heartbeatToPeer(byte[] messagePayload, ControllerNode peer) {
        Endpoint remoteEp = new Endpoint(peer.ip(), peer.tcpPort());
        this.messagingService.sendAsync(remoteEp, HEARTBEAT_MESSAGE, messagePayload).whenComplete((result, error) -> {
            if (error != null) {
                log.trace("Sending heartbeat to {} failed", (Object)remoteEp, error);
            }
        });
    }

    public DateTime getLastUpdated(NodeId nodeId) {
        return this.nodeStateLastUpdatedTimes.get(nodeId);
    }

    private void readComponentConfiguration() {
        Set configProperties = this.cfgService.getProperties(((Object)((Object)this)).getClass().getName());
        for (ConfigProperty property : configProperties) {
            String s;
            if ("heartbeatInterval".equals(property.name())) {
                s = property.value();
                if (s == null) {
                    this.setHeartbeatInterval(100);
                    log.info("Heartbeat interval time is not configured, default value is {}", (Object)100);
                } else {
                    int newHeartbeatInterval;
                    int n = newHeartbeatInterval = Strings.isNullOrEmpty((String)s) ? 100 : Integer.parseInt(s.trim());
                    if (newHeartbeatInterval > 0 && this.heartbeatInterval != newHeartbeatInterval) {
                        this.heartbeatInterval = newHeartbeatInterval;
                        this.restartHeartbeatSender();
                    }
                    log.info("Configured. Heartbeat interval time is configured to {}", (Object)this.heartbeatInterval);
                }
            }
            if (!"phiFailureThreshold".equals(property.name())) continue;
            s = property.value();
            if (s == null) {
                this.setPhiFailureThreshold(10);
                log.info("Phi failure threshold is not configured, default value is {}", (Object)10);
                continue;
            }
            int newPhiFailureThreshold = Strings.isNullOrEmpty((String)s) ? 100 : Integer.parseInt(s.trim());
            this.setPhiFailureThreshold(newPhiFailureThreshold);
            log.info("Configured. Phi failure threshold is configured to {}", (Object)this.phiFailureThreshold);
        }
    }

    private void setHeartbeatInterval(int interval) {
        try {
            Preconditions.checkArgument((interval > 0 ? 1 : 0) != 0, (Object)"Interval must be greater than zero");
            this.heartbeatInterval = interval;
        }
        catch (IllegalArgumentException e) {
            log.warn(e.getMessage());
            this.heartbeatInterval = 100;
        }
    }

    private void setPhiFailureThreshold(int threshold) {
        this.phiFailureThreshold = threshold;
    }

    private void restartHeartbeatSender() {
        try {
            ScheduledExecutorService prevSender = this.heartBeatSender;
            this.heartBeatSender = Executors.newSingleThreadScheduledExecutor(Tools.groupedThreads((String)"onos/cluster/membership", (String)"heartbeat-sender-%d", (Logger)log));
            this.heartBeatSender.scheduleWithFixedDelay(this::heartbeat, 0L, this.heartbeatInterval, TimeUnit.MILLISECONDS);
            prevSender.shutdown();
        }
        catch (Exception e) {
            log.warn(e.getMessage());
        }
    }

    protected void bindClusterMetadataService(ClusterMetadataService clusterMetadataService) {
        this.clusterMetadataService = clusterMetadataService;
    }

    protected void unbindClusterMetadataService(ClusterMetadataService clusterMetadataService) {
        if (this.clusterMetadataService == clusterMetadataService) {
            this.clusterMetadataService = null;
        }
    }

    protected void bindMessagingService(MessagingService messagingService) {
        this.messagingService = messagingService;
    }

    protected void unbindMessagingService(MessagingService messagingService) {
        if (this.messagingService == messagingService) {
            this.messagingService = null;
        }
    }

    private static class HeartbeatMessage {
        private ControllerNode source;
        private ControllerNode.State state;

        public HeartbeatMessage(ControllerNode source, ControllerNode.State state) {
            this.source = source;
            this.state = state != null ? state : ControllerNode.State.ACTIVE;
        }

        public ControllerNode source() {
            return this.source;
        }
    }

    private class HeartbeatMessageHandler
    implements BiConsumer<Endpoint, byte[]> {
        private HeartbeatMessageHandler() {
        }

        @Override
        public void accept(Endpoint sender, byte[] message) {
            HeartbeatMessage hb = (HeartbeatMessage)SERIALIZER.decode(message);
            if (DistributedClusterStore.this.clusterMetadataService.getClusterMetadata().getNodes().contains(hb.source())) {
                DistributedClusterStore.this.failureDetector.report(hb.source().id());
                DistributedClusterStore.this.updateState(hb.source().id(), hb.state);
            }
        }
    }
}

