/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.urn.ietf.params.xml.ns.yang.yrt.ietf.te.topology.rev20160317.yrtietftetopology
            .informationsourceattributes.informationsourcestate;

import com.google.common.base.MoreObjects;
import java.util.BitSet;
import java.util.Objects;
import org.onosproject.yang.model.ModelObject;

/**
 * Represents the implementation of topology.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultTopology extends ModelObject implements Topology {
    protected Object providerIdRef;
    protected Object clientIdRef;
    protected Object teTopologyIdRef;
    protected Object networkIdRef;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public Object providerIdRef() {
        return providerIdRef;
    }

    @Override
    public Object clientIdRef() {
        return clientIdRef;
    }

    @Override
    public Object teTopologyIdRef() {
        return teTopologyIdRef;
    }

    @Override
    public Object networkIdRef() {
        return networkIdRef;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void providerIdRef(Object providerIdRef) {
        valueLeafFlags.set(LeafIdentifier.PROVIDERIDREF.getLeafIndex());
        this.providerIdRef = providerIdRef;
    }

    @Override
    public void clientIdRef(Object clientIdRef) {
        valueLeafFlags.set(LeafIdentifier.CLIENTIDREF.getLeafIndex());
        this.clientIdRef = clientIdRef;
    }

    @Override
    public void teTopologyIdRef(Object teTopologyIdRef) {
        valueLeafFlags.set(LeafIdentifier.TETOPOLOGYIDREF.getLeafIndex());
        this.teTopologyIdRef = teTopologyIdRef;
    }

    @Override
    public void networkIdRef(Object networkIdRef) {
        valueLeafFlags.set(LeafIdentifier.NETWORKIDREF.getLeafIndex());
        this.networkIdRef = networkIdRef;
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerIdRef, clientIdRef, teTopologyIdRef, networkIdRef, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultTopology) {
            DefaultTopology other = (DefaultTopology) obj;
            return
                Objects.equals(providerIdRef, other.providerIdRef) &&
                Objects.equals(clientIdRef, other.clientIdRef) &&
                Objects.equals(teTopologyIdRef, other.teTopologyIdRef) &&
                Objects.equals(networkIdRef, other.networkIdRef) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(getClass())
            .omitNullValues()
            .add("providerIdRef", providerIdRef)
            .add("clientIdRef", clientIdRef)
            .add("teTopologyIdRef", teTopologyIdRef)
            .add("networkIdRef", networkIdRef)
            .add("valueLeafFlags", valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultTopology.
     */
    public DefaultTopology() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}
