/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.urn.ietf.params.xml.ns.yang.yrt.ietf.te.types.rev20160320.yrtietftetypes
            .tunnelbidirassocproperties.bidirectional;

import com.google.common.base.MoreObjects;
import java.util.BitSet;
import java.util.Objects;
import org.onosproject.yang.gen.v1.urn.ietf.params.xml.ns.yang.yrt.ietf.inet.types.rev20130715.yrtietfinettypes
            .IpAddress;
import org.onosproject.yang.gen.v1.urn.ietf.params.xml.ns.yang.yrt.ietf.te.types.rev20160320.yrtietftetypes
            .BidirAssociationType;
import org.onosproject.yang.gen.v1.urn.ietf.params.xml.ns.yang.yrt.ietf.te.types.rev20160320.yrtietftetypes
            .BidirProvisioningMode;
import org.onosproject.yang.model.ModelObject;

/**
 * Represents the implementation of association.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultAssociation extends ModelObject implements Association {
    protected int id;
    protected IpAddress source;
    protected IpAddress globalSource;
    protected Class<? extends BidirAssociationType> type;
    protected Class<? extends BidirProvisioningMode> provisioing;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public int id() {
        return id;
    }

    @Override
    public IpAddress source() {
        return source;
    }

    @Override
    public IpAddress globalSource() {
        return globalSource;
    }

    @Override
    public Class<? extends BidirAssociationType> type() {
        return type;
    }

    @Override
    public Class<? extends BidirProvisioningMode> provisioing() {
        return provisioing;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void id(int id) {
        valueLeafFlags.set(LeafIdentifier.ID.getLeafIndex());
        this.id = id;
    }

    @Override
    public void source(IpAddress source) {
        valueLeafFlags.set(LeafIdentifier.SOURCE.getLeafIndex());
        this.source = source;
    }

    @Override
    public void globalSource(IpAddress globalSource) {
        valueLeafFlags.set(LeafIdentifier.GLOBALSOURCE.getLeafIndex());
        this.globalSource = globalSource;
    }

    @Override
    public void type(Class<? extends BidirAssociationType> type) {
        valueLeafFlags.set(LeafIdentifier.TYPE.getLeafIndex());
        this.type = type;
    }

    @Override
    public void provisioing(Class<? extends BidirProvisioningMode> provisioing) {
        valueLeafFlags.set(LeafIdentifier.PROVISIOING.getLeafIndex());
        this.provisioing = provisioing;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, source, globalSource, type, provisioing, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultAssociation) {
            DefaultAssociation other = (DefaultAssociation) obj;
            return
                Objects.equals(id, other.id) &&
                Objects.equals(source, other.source) &&
                Objects.equals(globalSource, other.globalSource) &&
                Objects.equals(type, other.type) &&
                Objects.equals(provisioing, other.provisioing) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(getClass())
            .omitNullValues()
            .add("id", id)
            .add("source", source)
            .add("globalSource", globalSource)
            .add("type", type)
            .add("provisioing", provisioing)
            .add("valueLeafFlags", valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultAssociation.
     */
    public DefaultAssociation() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}
