/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.samplenamespace;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.samplenamespace.sample.L1;
import org.onosproject.yang.gen.v1.samplenamespace.sample.L2;
import org.onosproject.yang.gen.v1.samplenamespace.sample.Top;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of sample.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class SampleOpParam extends InnerModelObject implements Sample {
    protected String leaf5;
    protected List<String> leaf6;
    protected BitSet valueLeafFlags = new BitSet();
    protected Top top;
    protected List<L1> l1;
    protected List<L2> l2;

    @Override
    public String leaf5() {
        return leaf5;
    }

    @Override
    public List<String> leaf6() {
        return leaf6;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public Top top() {
        return top;
    }

    @Override
    public List<L1> l1() {
        return l1;
    }

    @Override
    public List<L2> l2() {
        return l2;
    }

    @Override
    public void leaf5(String leaf5) {
        valueLeafFlags.set(LeafIdentifier.LEAF5.getLeafIndex());
        this.leaf5 = leaf5;
    }

    @Override
    public void leaf6(List<String> leaf6) {
        valueLeafFlags.set(LeafIdentifier.LEAF6.getLeafIndex());
        this.leaf6 = leaf6;
    }

    @Override
    public void top(Top top) {
        this.top = top;
    }

    @Override
    public void l1(List<L1> l1) {
        this.l1 = l1;
    }

    @Override
    public void l2(List<L2> l2) {
        this.l2 = l2;
    }

    @Override
    public void addToLeaf6(String addTo) {
        if (leaf6 == null) {
            leaf6 = new ArrayList<>();
        }
        leaf6.add(addTo);
    }


    @Override
    public void addToL1(L1 addTo) {
        if (l1 == null) {
            l1 = new ArrayList<>();
        }
        l1.add(addTo);
    }


    @Override
    public void addToL2(L2 addTo) {
        if (l2 == null) {
            l2 = new ArrayList<>();
        }
        l2.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(leaf5, leaf6, valueLeafFlags, top, l1, l2);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof SampleOpParam) {
            SampleOpParam other = (SampleOpParam) obj;
            return
                Objects.equals(leaf5, other.leaf5) &&
                Objects.equals(leaf6, other.leaf6) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.equals(top, other.top) &&
                Objects.equals(l1, other.l1) &&
                Objects.equals(l2, other.l2);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("leaf5=" + leaf5)
            .add("leaf6=" + leaf6)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("top=" + top)
            .add("l1=" + l1)
            .add("l2=" + l2)
            .toString();
    }

    /**
     * Creates an instance of sampleOpParam.
     */
    public SampleOpParam() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }


    @Override
    public void addAugmentation(InnerModelObject obj) {
    }

    @Override
    public void removeAugmentation(InnerModelObject obj) {
    }

    @Override
    public Map<Class<? extends InnerModelObject>, InnerModelObject> augmentations() {

        return null;
    }

    @Override
    public <T extends InnerModelObject> T augmentation(Class<T> c) {

        return null;
    }
}
