/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.l3vpn.svc.rev20160730.ietfl3vpnsvc
            .customerlocationinfo;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of location.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultLocation extends InnerModelObject implements Location {
    protected String address;
    protected String zipCode;
    protected String state;
    protected String city;
    protected String countryCode;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public String address() {
        return address;
    }

    @Override
    public String zipCode() {
        return zipCode;
    }

    @Override
    public String state() {
        return state;
    }

    @Override
    public String city() {
        return city;
    }

    @Override
    public String countryCode() {
        return countryCode;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void address(String address) {
        valueLeafFlags.set(LeafIdentifier.ADDRESS.getLeafIndex());
        this.address = address;
    }

    @Override
    public void zipCode(String zipCode) {
        valueLeafFlags.set(LeafIdentifier.ZIPCODE.getLeafIndex());
        this.zipCode = zipCode;
    }

    @Override
    public void state(String state) {
        valueLeafFlags.set(LeafIdentifier.STATE.getLeafIndex());
        this.state = state;
    }

    @Override
    public void city(String city) {
        valueLeafFlags.set(LeafIdentifier.CITY.getLeafIndex());
        this.city = city;
    }

    @Override
    public void countryCode(String countryCode) {
        valueLeafFlags.set(LeafIdentifier.COUNTRYCODE.getLeafIndex());
        this.countryCode = countryCode;
    }

    @Override
    public int hashCode() {
        return Objects.hash(address, zipCode, state, city, countryCode, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultLocation) {
            DefaultLocation other = (DefaultLocation) obj;
            return
                Objects.equals(address, other.address) &&
                Objects.equals(zipCode, other.zipCode) &&
                Objects.equals(state, other.state) &&
                Objects.equals(city, other.city) &&
                Objects.equals(countryCode, other.countryCode) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("address=" + address)
            .add("zipCode=" + zipCode)
            .add("state=" + state)
            .add("city=" + city)
            .add("countryCode=" + countryCode)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultLocation.
     */
    public DefaultLocation() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}
