/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.l3vpn.svc.rev20160730.ietfl3vpnsvc.siterouting
            .routingprotocols.routingprotocol;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.l3vpn.svc.rev20160730.ietfl3vpnsvc.AddressFamily;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of bgp.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultBgp extends InnerModelObject implements Bgp {
    protected long autonomousSystem;
    protected List<Class<? extends AddressFamily>> addressFamily;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public long autonomousSystem() {
        return autonomousSystem;
    }

    @Override
    public List<Class<? extends AddressFamily>> addressFamily() {
        return addressFamily;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void autonomousSystem(long autonomousSystem) {
        valueLeafFlags.set(LeafIdentifier.AUTONOMOUSSYSTEM.getLeafIndex());
        this.autonomousSystem = autonomousSystem;
    }

    @Override
    public void addressFamily(List<Class<? extends AddressFamily>> addressFamily) {
        valueLeafFlags.set(LeafIdentifier.ADDRESSFAMILY.getLeafIndex());
        this.addressFamily = addressFamily;
    }

    @Override
    public void addToAddressFamily(Class<? extends AddressFamily> addTo) {
        if (addressFamily == null) {
            addressFamily = new ArrayList<>();
        }
        addressFamily.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(autonomousSystem, addressFamily, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultBgp) {
            DefaultBgp other = (DefaultBgp) obj;
            return
                Objects.equals(autonomousSystem, other.autonomousSystem) &&
                Objects.equals(addressFamily, other.addressFamily) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("autonomousSystem=" + autonomousSystem)
            .add("addressFamily=" + addressFamily)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultBgp.
     */
    public DefaultBgp() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}
