/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteservicemulticast;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteservicemulticast.multicast.MulticastSiteTypeEnum;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteservicemulticast.multicast.MulticastTransportProtocol;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteservicemulticast.multicast.ProtocolTypeEnum;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of multicast.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultMulticast extends InnerModelObject implements Multicast {
    protected MulticastSiteTypeEnum multicastSiteType;
    protected ProtocolTypeEnum protocolType;
    protected BitSet valueLeafFlags = new BitSet();
    protected MulticastTransportProtocol multicastTransportProtocol;

    @Override
    public MulticastSiteTypeEnum multicastSiteType() {
        return multicastSiteType;
    }

    @Override
    public ProtocolTypeEnum protocolType() {
        return protocolType;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public MulticastTransportProtocol multicastTransportProtocol() {
        return multicastTransportProtocol;
    }

    @Override
    public void multicastSiteType(MulticastSiteTypeEnum multicastSiteType) {
        valueLeafFlags.set(LeafIdentifier.MULTICASTSITETYPE.getLeafIndex());
        this.multicastSiteType = multicastSiteType;
    }

    @Override
    public void protocolType(ProtocolTypeEnum protocolType) {
        valueLeafFlags.set(LeafIdentifier.PROTOCOLTYPE.getLeafIndex());
        this.protocolType = protocolType;
    }

    @Override
    public void multicastTransportProtocol(MulticastTransportProtocol multicastTransportProtocol) {
        this.multicastTransportProtocol = multicastTransportProtocol;
    }

    @Override
    public int hashCode() {
        return Objects.hash(multicastSiteType, protocolType, valueLeafFlags, multicastTransportProtocol);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultMulticast) {
            DefaultMulticast other = (DefaultMulticast) obj;
            return
                Objects.equals(multicastSiteType, other.multicastSiteType) &&
                Objects.equals(protocolType, other.protocolType) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.equals(multicastTransportProtocol, other.multicastTransportProtocol);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("multicastSiteType=" + multicastSiteType)
            .add("protocolType=" + protocolType)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("multicastTransportProtocol=" + multicastTransportProtocol)
            .toString();
    }


    /**
     * Creates an instance of defaultMulticast.
     */
    public DefaultMulticast() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}