/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteservicemulticast.multicast;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of multicastTransportProtocol.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultMulticastTransportProtocol extends InnerModelObject implements MulticastTransportProtocol {
    protected boolean ipv4;
    protected boolean ipv6;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public boolean ipv4() {
        return ipv4;
    }

    @Override
    public boolean ipv6() {
        return ipv6;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void ipv4(boolean ipv4) {
        valueLeafFlags.set(LeafIdentifier.IPV4.getLeafIndex());
        this.ipv4 = ipv4;
    }

    @Override
    public void ipv6(boolean ipv6) {
        valueLeafFlags.set(LeafIdentifier.IPV6.getLeafIndex());
        this.ipv6 = ipv6;
    }

    @Override
    public int hashCode() {
        return Objects.hash(ipv4, ipv6, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultMulticastTransportProtocol) {
            DefaultMulticastTransportProtocol other = (DefaultMulticastTransportProtocol) obj;
            return
                Objects.equals(ipv4, other.ipv4) &&
                Objects.equals(ipv6, other.ipv6) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("ipv4=" + ipv4)
            .add("ipv6=" + ipv6)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultMulticastTransportProtocol.
     */
    public DefaultMulticastTransportProtocol() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}