/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siterouting.routingprotocols.routingprotocol;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.AddressFamily;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of vrrp.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultVrrp extends InnerModelObject implements Vrrp {
    protected List<Class<? extends AddressFamily>> addressFamily;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public List<Class<? extends AddressFamily>> addressFamily() {
        return addressFamily;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void addressFamily(List<Class<? extends AddressFamily>> addressFamily) {
        valueLeafFlags.set(LeafIdentifier.ADDRESSFAMILY.getLeafIndex());
        this.addressFamily = addressFamily;
    }

    @Override
    public void addToAddressFamily(Class<? extends AddressFamily> addTo) {
        if (addressFamily == null) {
            addressFamily = new ArrayList<>();
        }
        addressFamily.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(addressFamily, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultVrrp) {
            DefaultVrrp other = (DefaultVrrp) obj;
            return
                Objects.equals(addressFamily, other.addressFamily) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("addressFamily=" + addressFamily)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultVrrp.
     */
    public DefaultVrrp() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}