/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.modeldatatoresourcedata.rev20160826.modeldatatoresourcedata.firstlevel;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.modeldatatoresourcedata.rev20160826.modeldatatoresourcedata.Uri;
import org.onosproject.yang.gen.v1.modeldatatoresourcedata.rev20160826.modeldatatoresourcedata.firstlevel.listinlist.List2;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of listInList.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultListInList extends InnerModelObject 
        implements ListInList, MultiInstanceObject<ListInListKeys> {
    protected Uri listKey;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<List2> list2;

    @Override
    public Uri listKey() {
        return listKey;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<List2> list2() {
        return list2;
    }

    @Override
    public void listKey(Uri listKey) {
        valueLeafFlags.set(LeafIdentifier.LISTKEY.getLeafIndex());
        this.listKey = listKey;
    }

    @Override
    public void list2(List<List2> list2) {
        this.list2 = list2;
    }

    @Override
    public void addToList2(List2 addTo) {
        if (list2 == null) {
            list2 = new ArrayList<>();
        }
        list2.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(listKey, valueLeafFlags, list2);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultListInList) {
            DefaultListInList other = (DefaultListInList) obj;
            return
                Objects.equals(listKey, other.listKey) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.equals(list2, other.list2);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("listKey=" + listKey)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("list2=" + list2)
            .toString();
    }


    /**
     * Creates an instance of defaultListInList.
     */
    public DefaultListInList() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}