/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yangautoprefixlist.rev20160624;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yangautoprefixlist.rev20160624.list2.L2;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of list2.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class List2OpParam extends InnerModelObject implements List2 {
    protected List<String> leaf2;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<L2> l2;

    @Override
    public List<String> leaf2() {
        return leaf2;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<L2> l2() {
        return l2;
    }

    @Override
    public void leaf2(List<String> leaf2) {
        valueLeafFlags.set(LeafIdentifier.LEAF2.getLeafIndex());
        this.leaf2 = leaf2;
    }

    @Override
    public void l2(List<L2> l2) {
        this.l2 = l2;
    }

    @Override
    public void addToLeaf2(String addTo) {
        if (leaf2 == null) {
            leaf2 = new ArrayList<>();
        }
        leaf2.add(addTo);
    }


    @Override
    public void addToL2(L2 addTo) {
        if (l2 == null) {
            l2 = new ArrayList<>();
        }
        l2.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(leaf2, valueLeafFlags, l2);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof List2OpParam) {
            List2OpParam other = (List2OpParam) obj;
            return
                Objects.equals(leaf2, other.leaf2) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.equals(l2, other.l2);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("leaf2=" + leaf2)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("l2=" + l2)
            .toString();
    }

    /**
     * Creates an instance of list2OpParam.
     */
    public List2OpParam() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }


    @Override
    public void addAugmentation(InnerModelObject obj) {
    }

    @Override
    public void removeAugmentation(InnerModelObject obj) {
    }

    @Override
    public Map<Class<? extends InnerModelObject>, InnerModelObject> augmentations() {

        return null;
    }

    @Override
    public <T extends InnerModelObject> T augmentation(Class<T> c) {

        return null;
    }
}