/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.l3vpnsvcext.rev20160730.l3vpnsvcext.bgpprofile;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of password.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultPassword extends InnerModelObject implements Password {
    protected boolean passwordEnabled;
    protected String passwordValue;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public boolean passwordEnabled() {
        return passwordEnabled;
    }

    @Override
    public String passwordValue() {
        return passwordValue;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void passwordEnabled(boolean passwordEnabled) {
        valueLeafFlags.set(LeafIdentifier.PASSWORDENABLED.getLeafIndex());
        this.passwordEnabled = passwordEnabled;
    }

    @Override
    public void passwordValue(String passwordValue) {
        valueLeafFlags.set(LeafIdentifier.PASSWORDVALUE.getLeafIndex());
        this.passwordValue = passwordValue;
    }

    @Override
    public int hashCode() {
        return Objects.hash(passwordEnabled, passwordValue, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultPassword) {
            DefaultPassword other = (DefaultPassword) obj;
            return
                Objects.equals(passwordEnabled, other.passwordEnabled) &&
                Objects.equals(passwordValue, other.passwordValue) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("passwordEnabled=" + passwordEnabled)
            .add("passwordValue=" + passwordValue)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultPassword.
     */
    public DefaultPassword() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}