/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ymsiptopology.rev20140101.ymsiptopology.node;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ymsiptopology.rev20140101.ymsiptopology.node.augmentedtoponode.AugRoute;
import org.onosproject.yang.gen.v1.ymsiptopology.rev20140101.ymsiptopology.node.augmentedtoponode.AugRoutec;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of augmentedTopoNode.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultAugmentedTopoNode extends InnerModelObject implements AugmentedTopoNode {
    protected String routerId;
    protected String routerIp;
    protected String routerPath;
    protected List<Short> value;
    protected BitSet valueLeafFlags = new BitSet();
    protected AugRoute augRoute;
    protected List<AugRoutec> augRoutec;

    @Override
    public String routerId() {
        return routerId;
    }

    @Override
    public String routerIp() {
        return routerIp;
    }

    @Override
    public String routerPath() {
        return routerPath;
    }

    @Override
    public List<Short> value() {
        return value;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public AugRoute augRoute() {
        return augRoute;
    }

    @Override
    public List<AugRoutec> augRoutec() {
        return augRoutec;
    }

    @Override
    public void routerId(String routerId) {
        valueLeafFlags.set(LeafIdentifier.ROUTERID.getLeafIndex());
        this.routerId = routerId;
    }

    @Override
    public void routerIp(String routerIp) {
        valueLeafFlags.set(LeafIdentifier.ROUTERIP.getLeafIndex());
        this.routerIp = routerIp;
    }

    @Override
    public void routerPath(String routerPath) {
        valueLeafFlags.set(LeafIdentifier.ROUTERPATH.getLeafIndex());
        this.routerPath = routerPath;
    }

    @Override
    public void value(List<Short> value) {
        valueLeafFlags.set(LeafIdentifier.VALUE.getLeafIndex());
        this.value = value;
    }

    @Override
    public void augRoute(AugRoute augRoute) {
        this.augRoute = augRoute;
    }

    @Override
    public void augRoutec(List<AugRoutec> augRoutec) {
        this.augRoutec = augRoutec;
    }

    @Override
    public void addToValue(Short addTo) {
        if (value == null) {
            value = new ArrayList<>();
        }
        value.add(addTo);
    }


    @Override
    public void addToAugRoutec(AugRoutec addTo) {
        if (augRoutec == null) {
            augRoutec = new ArrayList<>();
        }
        augRoutec.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(routerId, routerIp, routerPath, value, valueLeafFlags, augRoute, augRoutec);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultAugmentedTopoNode) {
            DefaultAugmentedTopoNode other = (DefaultAugmentedTopoNode) obj;
            return
                Objects.equals(routerId, other.routerId) &&
                Objects.equals(routerIp, other.routerIp) &&
                Objects.equals(routerPath, other.routerPath) &&
                Objects.equals(value, other.value) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.equals(augRoute, other.augRoute) &&
                Objects.equals(augRoutec, other.augRoutec);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("routerId=" + routerId)
            .add("routerIp=" + routerIp)
            .add("routerPath=" + routerPath)
            .add("value=" + value)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("augRoute=" + augRoute)
            .add("augRoutec=" + augRoutec)
            .toString();
    }


    /**
     * Creates an instance of defaultAugmentedTopoNode.
     */
    public DefaultAugmentedTopoNode() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}