/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteattachmentbearer;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteattachmentbearer.bearer.RequestedType;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of bearer.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultBearer extends InnerModelObject implements Bearer {
    protected boolean alwaysOn;
    protected String bearerReference;
    protected BitSet valueLeafFlags = new BitSet();
    protected RequestedType requestedType;

    @Override
    public boolean alwaysOn() {
        return alwaysOn;
    }

    @Override
    public String bearerReference() {
        return bearerReference;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public RequestedType requestedType() {
        return requestedType;
    }

    @Override
    public void alwaysOn(boolean alwaysOn) {
        valueLeafFlags.set(LeafIdentifier.ALWAYSON.getLeafIndex());
        this.alwaysOn = alwaysOn;
    }

    @Override
    public void bearerReference(String bearerReference) {
        valueLeafFlags.set(LeafIdentifier.BEARERREFERENCE.getLeafIndex());
        this.bearerReference = bearerReference;
    }

    @Override
    public void requestedType(RequestedType requestedType) {
        this.requestedType = requestedType;
    }

    @Override
    public int hashCode() {
        return Objects.hash(alwaysOn, bearerReference, valueLeafFlags, requestedType);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultBearer) {
            DefaultBearer other = (DefaultBearer) obj;
            return
                Objects.deepEquals(alwaysOn, other.alwaysOn) &&
                Objects.deepEquals(bearerReference, other.bearerReference) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(requestedType, other.requestedType);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("alwaysOn=" + alwaysOn)
            .add("bearerReference=" + bearerReference)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("requestedType=" + requestedType)
            .toString();
    }


    /**
     * Creates an instance of defaultBearer.
     */
    public DefaultBearer() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}