/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteattachmentipconnection.ipconnection;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.AddressAllocationType;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteattachmentipconnection.ipconnection.ipv6.Addresses;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of ipv6.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultIpv6 extends InnerModelObject implements Ipv6 {
    protected Class<? extends AddressAllocationType> addressAllocationType;
    protected short numberOfDynamicAddress;
    protected BitSet valueLeafFlags = new BitSet();
    protected Addresses addresses;

    @Override
    public Class<? extends AddressAllocationType> addressAllocationType() {
        return addressAllocationType;
    }

    @Override
    public short numberOfDynamicAddress() {
        return numberOfDynamicAddress;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public Addresses addresses() {
        return addresses;
    }

    @Override
    public void addressAllocationType(Class<? extends AddressAllocationType> addressAllocationType) {
        valueLeafFlags.set(LeafIdentifier.ADDRESSALLOCATIONTYPE.getLeafIndex());
        this.addressAllocationType = addressAllocationType;
    }

    @Override
    public void numberOfDynamicAddress(short numberOfDynamicAddress) {
        valueLeafFlags.set(LeafIdentifier.NUMBEROFDYNAMICADDRESS.getLeafIndex());
        this.numberOfDynamicAddress = numberOfDynamicAddress;
    }

    @Override
    public void addresses(Addresses addresses) {
        this.addresses = addresses;
    }

    @Override
    public int hashCode() {
        return Objects.hash(addressAllocationType, numberOfDynamicAddress, valueLeafFlags, addresses);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultIpv6) {
            DefaultIpv6 other = (DefaultIpv6) obj;
            return
                Objects.deepEquals(addressAllocationType, other.addressAllocationType) &&
                Objects.deepEquals(numberOfDynamicAddress, other.numberOfDynamicAddress) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(addresses, other.addresses);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("addressAllocationType=" + addressAllocationType)
            .add("numberOfDynamicAddress=" + numberOfDynamicAddress)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("addresses=" + addresses)
            .toString();
    }


    /**
     * Creates an instance of defaultIpv6.
     */
    public DefaultIpv6() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}