/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siterouting.routingprotocols.routingprotocol.ospf.shamlinks;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.SvcId;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of shamLink.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultShamLink extends InnerModelObject 
        implements ShamLink, MultiInstanceObject<ShamLinkKeys> {
    protected SvcId targetSite;
    protected int metric;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public SvcId targetSite() {
        return targetSite;
    }

    @Override
    public int metric() {
        return metric;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void targetSite(SvcId targetSite) {
        valueLeafFlags.set(LeafIdentifier.TARGETSITE.getLeafIndex());
        this.targetSite = targetSite;
    }

    @Override
    public void metric(int metric) {
        valueLeafFlags.set(LeafIdentifier.METRIC.getLeafIndex());
        this.metric = metric;
    }

    @Override
    public int hashCode() {
        return Objects.hash(targetSite, metric, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultShamLink) {
            DefaultShamLink other = (DefaultShamLink) obj;
            return
                Objects.deepEquals(targetSite, other.targetSite) &&
                Objects.deepEquals(metric, other.metric) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("targetSite=" + targetSite)
            .add("metric=" + metric)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultShamLink.
     */
    public DefaultShamLink() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}