/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.l3vpnsvcext.rev20160730.l3vpnsvcext.rdrtprofile.siterole.customcase;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.l3vpnsvcext.rev20160730.l3vpnsvcext.RouteDistinguisher;
import org.onosproject.yang.gen.v1.l3vpnsvcext.rev20160730.l3vpnsvcext.rdrtprofile.siterole.customcase.custom.ExportRt;
import org.onosproject.yang.gen.v1.l3vpnsvcext.rev20160730.l3vpnsvcext.rdrtprofile.siterole.customcase.custom.ImportRt;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of custom.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultCustom extends InnerModelObject implements Custom {
    protected RouteDistinguisher rd;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<ImportRt> importRt;
    protected List<ExportRt> exportRt;

    @Override
    public RouteDistinguisher rd() {
        return rd;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<ImportRt> importRt() {
        return importRt;
    }

    @Override
    public List<ExportRt> exportRt() {
        return exportRt;
    }

    @Override
    public void rd(RouteDistinguisher rd) {
        valueLeafFlags.set(LeafIdentifier.RD.getLeafIndex());
        this.rd = rd;
    }

    @Override
    public void importRt(List<ImportRt> importRt) {
        this.importRt = importRt;
    }

    @Override
    public void exportRt(List<ExportRt> exportRt) {
        this.exportRt = exportRt;
    }

    @Override
    public void addToImportRt(ImportRt addTo) {
        if (importRt == null) {
            importRt = new ArrayList<>();
        }
        importRt.add(addTo);
    }


    @Override
    public void addToExportRt(ExportRt addTo) {
        if (exportRt == null) {
            exportRt = new ArrayList<>();
        }
        exportRt.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(rd, valueLeafFlags, importRt, exportRt);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultCustom) {
            DefaultCustom other = (DefaultCustom) obj;
            return
                Objects.deepEquals(rd, other.rd) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(importRt, other.importRt) &&
                Objects.deepEquals(exportRt, other.exportRt);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("rd=" + rd)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("importRt=" + importRt)
            .add("exportRt=" + exportRt)
            .toString();
    }


    /**
     * Creates an instance of defaultCustom.
     */
    public DefaultCustom() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}