/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ymstopology.rev20140101.ymstopology.node;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ymstopology.rev20140101.ymstopology.node.terminationpoints.TerminationPoint;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of terminationPoints.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultTerminationPoints extends InnerModelObject implements TerminationPoints {
    protected short numberOfTp;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<TerminationPoint> terminationPoint;

    @Override
    public short numberOfTp() {
        return numberOfTp;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<TerminationPoint> terminationPoint() {
        return terminationPoint;
    }

    @Override
    public void numberOfTp(short numberOfTp) {
        valueLeafFlags.set(LeafIdentifier.NUMBEROFTP.getLeafIndex());
        this.numberOfTp = numberOfTp;
    }

    @Override
    public void terminationPoint(List<TerminationPoint> terminationPoint) {
        this.terminationPoint = terminationPoint;
    }

    @Override
    public void addToTerminationPoint(TerminationPoint addTo) {
        if (terminationPoint == null) {
            terminationPoint = new ArrayList<>();
        }
        terminationPoint.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(numberOfTp, valueLeafFlags, terminationPoint);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultTerminationPoints) {
            DefaultTerminationPoints other = (DefaultTerminationPoints) obj;
            return
                Objects.deepEquals(numberOfTp, other.numberOfTp) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(terminationPoint, other.terminationPoint);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("numberOfTp=" + numberOfTp)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("terminationPoint=" + terminationPoint)
            .toString();
    }


    /**
     * Creates an instance of defaultTerminationPoints.
     */
    public DefaultTerminationPoints() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}