/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtietftetopology.rev20160317.yrtietftetopology.informationsourceattributes;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietftetopology.rev20160317.yrtietftetopology.informationsourceattributes.informationsourcestate.Topology;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of informationSourceState.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultInformationSourceState extends InnerModelObject implements InformationSourceState {
    protected int credibilityPreference;
    protected String routingInstance;
    protected BitSet valueLeafFlags = new BitSet();
    protected Topology topology;

    @Override
    public int credibilityPreference() {
        return credibilityPreference;
    }

    @Override
    public String routingInstance() {
        return routingInstance;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public Topology topology() {
        return topology;
    }

    @Override
    public void credibilityPreference(int credibilityPreference) {
        valueLeafFlags.set(LeafIdentifier.CREDIBILITYPREFERENCE.getLeafIndex());
        this.credibilityPreference = credibilityPreference;
    }

    @Override
    public void routingInstance(String routingInstance) {
        valueLeafFlags.set(LeafIdentifier.ROUTINGINSTANCE.getLeafIndex());
        this.routingInstance = routingInstance;
    }

    @Override
    public void topology(Topology topology) {
        this.topology = topology;
    }

    @Override
    public int hashCode() {
        return Objects.hash(credibilityPreference, routingInstance, valueLeafFlags, topology);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultInformationSourceState) {
            DefaultInformationSourceState other = (DefaultInformationSourceState) obj;
            return
                Objects.deepEquals(credibilityPreference, other.credibilityPreference) &&
                Objects.deepEquals(routingInstance, other.routingInstance) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(topology, other.topology);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("credibilityPreference=" + credibilityPreference)
            .add("routingInstance=" + routingInstance)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("topology=" + topology)
            .toString();
    }


    /**
     * Creates an instance of defaultInformationSourceState.
     */
    public DefaultInformationSourceState() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}