/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.subscriptions.subscription.receivers.receiver;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietfyangtypes.rev20130715.yrtietfyangtypes.Counter64;
import org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.SubscriptionStatusType;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of augmentedReceiver.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultAugmentedReceiver extends InnerModelObject implements AugmentedReceiver {
    protected Counter64 pushedNotifications;
    protected Counter64 excludedNotifications;
    protected SubscriptionStatusType status;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public Counter64 pushedNotifications() {
        return pushedNotifications;
    }

    @Override
    public Counter64 excludedNotifications() {
        return excludedNotifications;
    }

    @Override
    public SubscriptionStatusType status() {
        return status;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void pushedNotifications(Counter64 pushedNotifications) {
        valueLeafFlags.set(LeafIdentifier.PUSHEDNOTIFICATIONS.getLeafIndex());
        this.pushedNotifications = pushedNotifications;
    }

    @Override
    public void excludedNotifications(Counter64 excludedNotifications) {
        valueLeafFlags.set(LeafIdentifier.EXCLUDEDNOTIFICATIONS.getLeafIndex());
        this.excludedNotifications = excludedNotifications;
    }

    @Override
    public void status(SubscriptionStatusType status) {
        valueLeafFlags.set(LeafIdentifier.STATUS.getLeafIndex());
        this.status = status;
    }

    @Override
    public int hashCode() {
        return Objects.hash(pushedNotifications, excludedNotifications, status, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultAugmentedReceiver) {
            DefaultAugmentedReceiver other = (DefaultAugmentedReceiver) obj;
            return
                Objects.deepEquals(pushedNotifications, other.pushedNotifications) &&
                Objects.deepEquals(excludedNotifications, other.excludedNotifications) &&
                Objects.deepEquals(status, other.status) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("pushedNotifications=" + pushedNotifications)
            .add("excludedNotifications=" + excludedNotifications)
            .add("status=" + status)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultAugmentedReceiver.
     */
    public DefaultAugmentedReceiver() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}