/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yangautoprefixlist.rev20160624;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yangautoprefixlist.rev20160624.list4.L4;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of list4.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class List4OpParam extends InnerModelObject implements List4 {
    protected List<String> leaf4;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<L4> l4;

    @Override
    public List<String> leaf4() {
        return leaf4;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<L4> l4() {
        return l4;
    }

    @Override
    public void leaf4(List<String> leaf4) {
        valueLeafFlags.set(LeafIdentifier.LEAF4.getLeafIndex());
        this.leaf4 = leaf4;
    }

    @Override
    public void l4(List<L4> l4) {
        this.l4 = l4;
    }

    @Override
    public void addToLeaf4(String addTo) {
        if (leaf4 == null) {
            leaf4 = new ArrayList<>();
        }
        leaf4.add(addTo);
    }


    @Override
    public void addToL4(L4 addTo) {
        if (l4 == null) {
            l4 = new ArrayList<>();
        }
        l4.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(leaf4, valueLeafFlags, l4);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof List4OpParam) {
            List4OpParam other = (List4OpParam) obj;
            return
                Objects.deepEquals(leaf4, other.leaf4) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(l4, other.l4);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("leaf4=" + leaf4)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("l4=" + l4)
            .toString();
    }

    /**
     * Creates an instance of list4OpParam.
     */
    public List4OpParam() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }


    @Override
    public void addAugmentation(InnerModelObject obj) {
    }

    @Override
    public void removeAugmentation(InnerModelObject obj) {
    }

    @Override
    public Map<Class<? extends InnerModelObject>, InnerModelObject> augmentations() {

        return null;
    }

    @Override
    public <T extends InnerModelObject> T augmentation(Class<T> c) {

        return null;
    }
}