/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.subscriptions;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.Encoding;
import org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.SubscriptionId;
import org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.notificationorigininfo.NotificationOrigin;
import org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.receiverinfo.Receivers;
import org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.subscriptionpolicynonmodifiable.Target;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of subscription.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultSubscription extends InnerModelObject 
        implements Subscription, MultiInstanceObject<SubscriptionKeys> {
    protected SubscriptionId identifier;
    protected boolean configuredSubscription;
    protected Encoding encoding;
    protected BitSet valueLeafFlags = new BitSet();
    protected Target target;
    protected NotificationOrigin notificationOrigin;
    protected Receivers receivers;

    @Override
    public SubscriptionId identifier() {
        return identifier;
    }

    @Override
    public boolean configuredSubscription() {
        return configuredSubscription;
    }

    @Override
    public Encoding encoding() {
        return encoding;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public Target target() {
        return target;
    }

    @Override
    public NotificationOrigin notificationOrigin() {
        return notificationOrigin;
    }

    @Override
    public Receivers receivers() {
        return receivers;
    }

    @Override
    public void identifier(SubscriptionId identifier) {
        valueLeafFlags.set(LeafIdentifier.IDENTIFIER.getLeafIndex());
        this.identifier = identifier;
    }

    @Override
    public void configuredSubscription(boolean configuredSubscription) {
        valueLeafFlags.set(LeafIdentifier.CONFIGUREDSUBSCRIPTION.getLeafIndex());
        this.configuredSubscription = configuredSubscription;
    }

    @Override
    public void encoding(Encoding encoding) {
        valueLeafFlags.set(LeafIdentifier.ENCODING.getLeafIndex());
        this.encoding = encoding;
    }

    @Override
    public void target(Target target) {
        this.target = target;
    }

    @Override
    public void notificationOrigin(NotificationOrigin notificationOrigin) {
        this.notificationOrigin = notificationOrigin;
    }

    @Override
    public void receivers(Receivers receivers) {
        this.receivers = receivers;
    }

    @Override
    public int hashCode() {
        return Objects.hash(identifier, configuredSubscription, encoding, valueLeafFlags, target, notificationOrigin, receivers);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultSubscription) {
            DefaultSubscription other = (DefaultSubscription) obj;
            return
                Objects.deepEquals(identifier, other.identifier) &&
                Objects.deepEquals(configuredSubscription, other.configuredSubscription) &&
                Objects.deepEquals(encoding, other.encoding) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(target, other.target) &&
                Objects.deepEquals(notificationOrigin, other.notificationOrigin) &&
                Objects.deepEquals(receivers, other.receivers);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("identifier=" + identifier)
            .add("configuredSubscription=" + configuredSubscription)
            .add("encoding=" + encoding)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("target=" + target)
            .add("notificationOrigin=" + notificationOrigin)
            .add("receivers=" + receivers)
            .toString();
    }


    /**
     * Creates an instance of defaultSubscription.
     */
    public DefaultSubscription() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}